<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}
use \Elementor\Plugin as KFEl;
use \Elementor\Widget_Base as WB;
use \Elementor\Controls_Manager as KFCM;

use \Elementor\Group_Control_Background as KFGCBG;
use \Elementor\Group_Control_Typography as KFGCT;
use \Elementor\Group_Control_Border as KFGCBR;
use \Elementor\Group_Control_Text_Shadow as KFGCTS;
use \Elementor\Group_Control_Text_Stroke as KFGCTST;

class KFW_Content_Switch extends WB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-content-switch';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Content Switch', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_core_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-custom';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'kfw', 'kin', 'switch', 'content' ];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
		return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/content-switch';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

		wp_register_style( 'kfw-content-switch',
			KFE_CONST_URL . 'libraries/elementor/assets/css/content-switch' . KFE_CONST_DEBUG_SUFFIX . '.css',
			[],
			KFE_CONST_VERSION
		);

        wp_register_script( 'kfw-content-switch',
            KFE_CONST_URL . 'libraries/elementor/assets/js/content-switch' . KFE_CONST_DEBUG_SUFFIX . '.js',
            [ 'jquery', 'elementor-frontend' ],
            KFE_CONST_VERSION,
            true
        );

    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'kfw-content-switch' ];
	}

	/**
	 * Get script dependencies.
	 *
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @return array Element scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'kfw-content-switch' ];
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
        $this->_register_primary_section();
        $this->_register_secondary_section();
        $this->_register_settings_section();

        $this->_register_title_style_section();
        $this->_register_switch_style_section();
    }

    public function _register_primary_section() {
		$this->start_controls_section( 'section_primary', [
			'label' => esc_html__( 'Primary', 'onnat-extra' )
		]);
            $this->add_control( 'primary-title', [
                'label'     => esc_html__( 'Title', 'onnat-extra' ),
                'type'      => KFCM::TEXT,
                'default'   => esc_html__( 'Primary', 'onnat-extra' ),
            ]);
            $this->add_control( 'primary-template', [
                'label'     => esc_html__('Template', 'onnat-extra'),
                'type'      => KFCM::SELECT,
                'default'   => 0,
                'options'   => kf_extra_plugin_elementor()->get_elementor_library(),
            ]);
        $this->end_controls_section();
    }

    public function _register_secondary_section() {
		$this->start_controls_section( 'section_secondary', [
			'label' => esc_html__( 'Secondary', 'onnat-extra' )
		]);
            $this->add_control( 'secondary-title', [
                'label'     => esc_html__( 'Title', 'onnat-extra' ),
                'type'      => KFCM::TEXT,
                'default'   => esc_html__( 'Secondary', 'onnat-extra' ),
            ]);
            $this->add_control( 'secondary-template', [
                'label'     => esc_html__('Template', 'onnat-extra'),
                'type'      => KFCM::SELECT,
                'default'   => 0,
                'options'   => kf_extra_plugin_elementor()->get_elementor_library(),
            ]);
        $this->end_controls_section();
    }

    public function _register_settings_section() {
		$this->start_controls_section( 'section_settings', [
			'label' => esc_html__( 'Settings', 'onnat-extra' )
		]);
            $this->add_control('active',[
                'label'   => esc_html__( 'Active', 'onnat-extra' ),
                'type'    => KFCM::CHOOSE,
                'default' => 'kinfw-primary-switch',
                'options' => [
                    'kinfw-primary-switch'   => [
                        'title' => esc_html__( 'Primary', 'onnat-extra' ),
                        'icon'  => 'kfw-ctrl-icon kfw-ctrl-primary-icon',
                    ],
                    'kinfw-secondary-switch' => [
                        'title' => esc_html__( 'Secondary', 'onnat-extra' ),
                        'icon'  => 'kfw-ctrl-icon kfw-ctrl-secondary-icon',
                    ],
                ],
                'toggle'  => false,
            ]);
            $this->add_control('switch-size',[
                'label'   => esc_html__( 'Switch Size', 'onnat-extra' ),
                'type'    => KFCM::CHOOSE,
                'default' => 'kinfw-switch-md-size',
                'options' => [
                    'kinfw-switch-sm-size' => [
                        'title' => esc_html__( 'Small', 'onnat-extra' ),
                        'icon'  => 'kfw-ctrl-icon kfw-ctrl-sm-size',
                    ],
                    'kinfw-switch-md-size' => [
                        'title' => esc_html__( 'Medium', 'onnat-extra' ),
                        'icon'  => 'kfw-ctrl-icon kfw-ctrl-md-size',
                    ],
                    'kinfw-switch-lg-size' => [
                        'title' => esc_html__( 'Large', 'onnat-extra' ),
                        'icon'  => 'kfw-ctrl-icon kfw-ctrl-lg-size',
                    ],
                    'kinfw-switch-xl-size' => [
                        'title' => esc_html__( 'Extra Large', 'onnat-extra' ),
                        'icon'  => 'kfw-ctrl-icon kfw-ctrl-xl-size',
                    ],
                ],
                'toggle'  => false,
            ]);
            $this->add_responsive_control('align', [
                'label'          => esc_html__( 'Alignment', 'onnat-extra' ),
                'type'           => KFCM::CHOOSE,
                'options'        => [
                    'left'   => [
                        'title' => esc_html__( 'Left', 'onnat-extra' ),
                        'icon'  => 'eicon-text-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__( 'Center', 'onnat-extra' ),
                        'icon'  => 'eicon-text-align-center',
                    ],
                    'right'  => [
                        'title' => esc_html__( 'Right', 'onnat-extra' ),
                        'icon'  => 'eicon-text-align-right',
                    ],
                ],
                'toggle'               => false,
                'selectors_dictionary' => [
                    'left'   => 'justify-content:start;',
                    'center' => 'justify-content:center;',
                    'right'  => 'justify-content:end;',
                ],
                'selectors'            => [
                    '{{WRAPPER}} .kfw-content-switch-header-wrap' => '{{VALUE}};'
                ],
            ]);            
        $this->end_controls_section();
    }

    public function _register_title_style_section() {
        $this->_register_wrapper_style_section();
        $this->_register_primary_title_style_section();
        $this->_register_secondary_title_style_section();
    }

    public function _register_wrapper_style_section() {
		$this->start_controls_section( 'section_wrap_style', [
			'label' => esc_html__( 'Wrapper', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_responsive_control( 'wrap_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-content-switch-header-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ]
            ]);
            $this->add_responsive_control( 'wrap_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-content-switch-header-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ]
            ]);
        $this->end_controls_section();
    }

    public function _register_primary_title_style_section() {
		$this->start_controls_section( 'section_primary_title_style', [
			'label' => esc_html__( 'Primary Title', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_group_control(KFGCT::get_type(), [
                'name'     => 'primary_title_typo',
                'label'    => esc_html__( 'Typography', 'onnat-extra' ),
                'selector' => '{{WRAPPER}} .kfw-content-switch-header-wrap .kfw-primary-title',
            ]);
            $this->add_responsive_control( 'primary_title_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-content-switch-header-wrap .kfw-primary-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'primary_title_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-content-switch-header-wrap .kfw-primary-title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_control( 'primary_title_color', [
                'label'     => esc_html__('Color', 'onnat-extra' ),
                'type'      => KFCM::COLOR,
                'selectors' => ['{{WRAPPER}} .kfw-content-switch-header-wrap .kfw-primary-title' => 'color: {{VALUE}};' ],
            ]);
            $this->add_group_control( KFGCTS::get_type(), [
                'name'     => 'primary_title_txt_shadow',
                'selector' => '{{WRAPPER}} .kfw-content-switch-header-wrap .kfw-primary-title',
            ]);
            $this->add_group_control( KFGCTST::get_type(), [
                'name'     => 'primary_title_txt_stroke',
                'selector' => '{{WRAPPER}} .kfw-content-switch-header-wrap .kfw-primary-title',
            ]);
		$this->end_controls_section();
    }

    public function _register_secondary_title_style_section() {
		$this->start_controls_section( 'section_secondary_title_style', [
			'label' => esc_html__( 'Secondary Title', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_group_control(KFGCT::get_type(), [
                'name'     => 'secondary_title_typo',
                'label'    => esc_html__( 'Typography', 'onnat-extra' ),
                'selector' => '{{WRAPPER}} .kfw-content-switch-header-wrap .kfw-secondary-title',
            ]);
            $this->add_responsive_control( 'secondary_title_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-content-switch-header-wrap .kfw-secondary-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'secondary_title_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-content-switch-header-wrap .kfw-secondary-title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_control( 'secondary_title_color', [
                'label'     => esc_html__('Color', 'onnat-extra' ),
                'type'      => KFCM::COLOR,
                'selectors' => ['{{WRAPPER}} .kfw-content-switch-header-wrap .kfw-secondary-title' => 'color: {{VALUE}};' ],
            ]);
            $this->add_group_control( KFGCTS::get_type(), [
                'name'     => 'secondary_title_txt_shadow',
                'selector' => '{{WRAPPER}} .kfw-content-switch-header-wrap .kfw-secondary-title',
            ]);
            $this->add_group_control( KFGCTST::get_type(), [
                'name'     => 'secondary_title_txt_stroke',
                'selector' => '{{WRAPPER}} .kfw-content-switch-header-wrap .kfw-secondary-title',
            ]);
		$this->end_controls_section();
    }

    public function _register_switch_style_section() {
		$this->start_controls_section( 'section_switch_style', [
			'label' => esc_html__( 'Switch', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
            $this->start_controls_tabs( 'switch_customize_tabs' );
                $this->start_controls_tab( 'switch_normal_state', [
                    'label' => esc_html__( 'Normal', 'onnat-extra' ),
                ]);
                    $this->add_control('wrapper_heading_hr_before', [
                        'type' => KFCM::DIVIDER,
                    ]);
                    $this->add_control('wrapper_heading',[
                        'label' => esc_html__( 'WRAPPER', 'onnat-extra' ),
                        'type'  => KFCM::HEADING,
                    ]);
                    $this->add_control('wrapper_heading_hr_after', [
                        'type' => KFCM::DIVIDER,
                    ]);
                    $this->add_group_control( KFGCBG::get_type(), [
                        'name'           => 'switch_normal_bg',
                        'types'          => ['classic', 'gradient'],
                        'exclude'        => [ 'image'],
                        'selector'       => '{{WRAPPER}} .kfw-switch',
                        'fields_options' => [
                            'background' => [
                                'label' => esc_html__( 'Background', 'onnat-extra' ),
                                'default' => 'classic'
                            ],
                            'color'      => [
                                'default' => ''
                            ],
                        ],
                    ]);
                    $this->add_group_control( KFGCBR::get_type(), [
                        'name'      => 'switch_normal_border',
                        'selector'  => '{{WRAPPER}} .kfw-switch',
                    ]);
                    $this->add_responsive_control( 'switch_normal_border_radius', [
                        'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                        'type'       => KFCM::DIMENSIONS,
                        'size_units' => [ 'px', '%' ],
                        'selectors'  => [
                            '{{WRAPPER}} .kfw-switch' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                        ],
                    ]);
                    $this->add_control('dot_heading_hr_before', [
                        'type' => KFCM::DIVIDER,
                    ]);
                    $this->add_control('dot_heading',[
                        'label' => esc_html__( 'DOT', 'onnat-extra' ),
                        'type'  => KFCM::HEADING,
                    ]);
                    $this->add_control('dot_heading_hr_after', [
                        'type' => KFCM::DIVIDER,
                    ]);                    
					$this->add_control( 'dot_normal_bg_color', [
						'label'     => esc_html__('Background Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => [
							'{{WRAPPER}} .kfw-switch-toggle' => 'background:{{VALUE}};',
						],
					]);
                    $this->add_group_control( KFGCBR::get_type(), [
                        'name'      => 'dot_normal_border',
                        'selector'  => '{{WRAPPER}} .kfw-switch-toggle',
                    ]);
                    $this->add_responsive_control( 'dot_normal_border_radius', [
                        'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                        'type'       => KFCM::DIMENSIONS,
                        'size_units' => [ 'px', '%' ],
                        'selectors'  => [
                            '{{WRAPPER}} .kfw-switch-toggle' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                        ],
                    ]);
                $this->end_controls_tab();
                $this->start_controls_tab( 'switch_hover_state', [
                    'label' => esc_html__( 'Active', 'onnat-extra' ),
                ]);
                    $this->add_control('wrapper_heading_before_hr_hover', [
                        'type' => KFCM::DIVIDER,
                    ]);
                    $this->add_control('wrapper_heading_hover',[
                        'label' => esc_html__( 'WRAPPER', 'onnat-extra' ),
                        'type'  => KFCM::HEADING,
                    ]);
                    $this->add_control('wrapper_heading_after_hr_hover', [
                        'type' => KFCM::DIVIDER,
                    ]);
                    $this->add_group_control( KFGCBG::get_type(), [
                        'name'           => 'switch_hover_bg',
                        'types'          => ['classic', 'gradient'],
                        'exclude'        => [ 'image'],
                        'selector'       => '{{WRAPPER}} .kfw-switch.kinfw-secondary-switch',
                        'fields_options' => [
                            'background' => [
                                'label' => esc_html__( 'Background', 'onnat-extra' ),
                                'default' => 'classic'
                            ],
                            'color'      => [
                                'default' => ''
                            ],
                        ],
                    ]);
                    $this->add_group_control( KFGCBR::get_type(), [
                        'name'      => 'switch_hover_border',
                        'selector'  => '{{WRAPPER}} .kfw-switch.kinfw-secondary-switch',
                    ]);
                    $this->add_responsive_control( 'switch_hover_border_radius', [
                        'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                        'type'       => KFCM::DIMENSIONS,
                        'size_units' => [ 'px', '%' ],
                        'selectors'  => [
                            '{{WRAPPER}} .kfw-switch.kinfw-secondary-switch' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                        ],
                    ]);
                    $this->add_control('dot_heading_hr_hover__before', [
                        'type' => KFCM::DIVIDER,
                    ]);
                    $this->add_control('dot_heading_hover',[
                        'label' => esc_html__( 'DOT', 'onnat-extra' ),
                        'type'  => KFCM::HEADING,
                    ]);
                    $this->add_control('dot_heading_hr_hover_after', [
                        'type' => KFCM::DIVIDER,
                    ]);
					$this->add_control( 'dot_hover_bg_color', [
						'label'     => esc_html__('Background Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => [
							'{{WRAPPER}}' => '--kfw-switch-toggle-active-color:{{VALUE}};',
						],
					]);
                    $this->add_group_control( KFGCBR::get_type(), [
                        'name'      => 'dot_hover_border',
                        'selector'  => '{{WRAPPER}} .kfw-switch.kinfw-secondary-switch .kfw-switch-toggle',
                    ]);
                    $this->add_responsive_control( 'dot_hover_border_radius', [
                        'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                        'type'       => KFCM::DIMENSIONS,
                        'size_units' => [ 'px', '%' ],
                        'selectors'  => [
                            '{{WRAPPER}} .kfw-switch.kinfw-secondary-switch .kfw-switch-toggle' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                        ],
                    ]);
                $this->end_controls_tabs();
            $this->end_controls_tabs();
		$this->end_controls_section();
    }

	/**
	 * Get template content
	 */
	public function _generate_template_content( $template_id ) {
		return KFEl::instance()->frontend->get_builder_content_for_display( $template_id );
	}

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-content-switch");
		include $render;
    }

}