<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use \Elementor\Plugin as KFEl;
use \Elementor\Widget_Base as KFWB;
use \Elementor\Controls_Manager as KFCM;
use \Elementor\Icons_Manager as KFIM;

use \Elementor\Group_Control_Typography as KFGCT;
use \Elementor\Group_Control_Text_Shadow as KFGCTS;
use \Elementor\Group_Control_Text_Stroke as KFGCTST;
use \Elementor\Group_Control_Border as KFGCBR;
use \Elementor\Group_Control_Box_Shadow as KFGCBS;
use \Elementor\Group_Control_Background as KFGCBG;
use \Elementor\Repeater as KFRTR;

class KFW_Custom_List extends KFWB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-custom-list';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('List', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_core_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-editor-list-ul';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'kfw', 'kin', 'info', 'list', 'minimal'];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
		return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/custom-list';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

		wp_register_style( 'kfw-custom-list',
			KFE_CONST_URL . 'libraries/elementor/assets/css/custom-list' . KFE_CONST_DEBUG_SUFFIX . '.css',
			[],
			KFE_CONST_VERSION
		);

        wp_register_script( 'lordicon',
            KFE_CONST_URL . 'libraries/elementor/assets/js/lordicon.js',
            [],
            KFE_CONST_VERSION,
            true
        );

    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'kfw-custom-list' ];
	}

	/**
	 * Get script dependencies.
	 *
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @return array Element scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'lordicon' ];
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
		$this->_register_items_section();
		$this->_register_list_style_section();
        $this->_register_icon_style_section();
        $this->_register_title_style_section();
		$this->_register_sub_title_style_section();
    }

    public function _register_items_section() {
		$repeater = new KFRTR();

		$this->start_controls_section( 'section_items', [
			'label' => esc_html__( 'Items', 'onnat-extra' )
		]);
            $repeater->start_controls_tabs('items_tabs');
                $repeater->start_controls_tab('tab_content',[
                    'label' => esc_html__( 'Content', 'onnat-extra' ),
                ]);
                    $repeater->add_control('title',[
                        'label'       => esc_html__( 'Title', 'onnat-extra' ),
                        'type'        => KFCM::TEXT,
                        'dynamic'     => [ 'active' => true ],
                        'default'     => esc_html__( 'Item Title' , 'onnat-extra' ),
                        'label_block' => true,
                    ]);
                    $repeater->add_control('sub_title',[
                        'label'       => esc_html__( 'Sub Title', 'onnat-extra' ),
                        'type'        => KFCM::TEXT,
                        'dynamic'     => [ 'active' => true, ],
                        'placeholder' => esc_html__( 'Sub Title', 'onnat-extra' ),
                        'label_block' => true,
                    ]);
                    $repeater->add_control('link',[
                        'label'     => esc_html__( 'Link', 'onnat-extra' ),
                        'type'      => KFCM::URL,
                        'dynamic'   => [
                            'active' => true
                        ],
                        'options'   => [
                            'url',
                            'is_external',
                            'nofollow'
                        ],
                        'default'   => [
                            'url'         => '',
                            'is_external' => true,
                            'nofollow'    => true,
                        ],
                    ]);
                $repeater->end_controls_tab();
                $repeater->start_controls_tab('tab_icon',[
                    'label' => esc_html__( 'Icon', 'onnat-extra' ),
                ]);
                    $repeater->add_control('icons_settings_des', [
                        'type'            => KFCM::RAW_HTML,
                        'content_classes' => 'kfw-elementor-editor-notice-ctrl',
                        'raw'             => sprintf(
                            esc_html__('Icon settings reflects only if %1$s Use Icon option %2$s is set to %1$sYES%2$s.', 'onnat-extra'),
                            '<b> <i> ', '<i> </b>'
                        ),
                    ]);
                    $repeater->add_control('icon_type',[
                        'label'   => esc_html__( 'Icon Type', 'onnat-extra' ),
                        'type'    => KFCM::CHOOSE,
                        'options' => [
                            'blank'             => [
                                'title' => esc_html__('Blank', 'onnat-extra'),
                                'icon'  => '',
                            ],
                            'icon'              => [
                                'title' => esc_html__('Icon', 'onnat-extra'),
                                'icon'  => 'eicon-plus-circle-o',
                            ],
                            'image'             => [
                                'title' => esc_html__('Image', 'onnat-extra'),
                                'icon'  => 'eicon-image',
                            ],
                            'lordicon_media'    => [
                                'title' => esc_html__('Upload Lordicon From Media', 'onnat-extra'),
                                'icon'  => 'eicon-library-upload',
                            ],
                            'lordicon_external' => [
                                'title' => esc_html__('Upload Lordicon From Extrnal', 'onnat-extra'),
                                'icon'  => 'eicon-editor-external-link',
                            ],
                        ],
                        'default' => 'blank',
                        'toggle'  => false
                    ]);
                    $repeater->add_control( 'icon', [
                        'label'     => esc_html__( 'Icon', 'onnat-extra' ),
                        'type'      => KFCM::ICONS,
                        'default'   => [
                            'value'   => 'fas fa-heart',
                            'library' => 'fa-solid',
                        ],
                        'condition' => [
                            'icon_type' => 'icon'
                        ],
                    ]);
                    $repeater->add_control( 'image', [
                        'label'     => esc_html__( 'Image', 'onnat-extra' ),
                        'type'      => KFCM::MEDIA,
                        'dynamic'   => [ 'active' => true ],
                        'default'   => [
                            'url' => \Elementor\Utils::get_placeholder_image_src()
                        ],
                        'condition' => [
                            'icon_type' => 'image'
                        ],
                    ]);
                    $repeater->add_control('lordicon_des', [
                        'type'            => KFCM::RAW_HTML,
                        'content_classes' => 'kfw-elementor-editor-notice-ctrl',
                        'raw'             => sprintf(
                            esc_html__('You can get lordicon code from %1$s ', 'onnat-extra'),
                            sprintf('<a href="%1$s" target="_blank">%2$s</a>', esc_url( 'https://lordicon.com/' ), esc_html_x( 'Lordicon.com', 'link text for lordicon.com', 'onnat-extra' ) )
                        ),
                        'condition'       => [
                            'icon_type' => [ 'lordicon_media', 'lordicon_external' ]
                        ],
                    ]);
                    $repeater->add_control('lordicon_media', [
                        'label'      => esc_html__( 'Upload JSON File', 'onnat-extra' ),
                        'type'       => KFCM::MEDIA,
                        'media_type' => 'application/json',
                        'default'    => [
                            'url' => KFE_CONST_URL . 'libraries/elementor/assets/json/lordicon-clock-icon.json'
                        ],
                        'condition'  => [
                            'icon_type' => 'lordicon_media'
                        ]
                    ]);
                    $repeater->add_control('lordicon_external', [
                        'label'       => esc_html__( 'Upload JSON File', 'onnat-extra' ),
                        'type'        => KFCM::URL,
                        'dynamic'     => [ 'active' => true, ],
                        'placeholder' => esc_html__( 'Enter your URL', 'onnat-extra' ),
                        'options'     => [ 'url' ],
                        'default'     => [
                            'url' => KFE_CONST_URL . 'libraries/elementor/assets/json/lordicon-clock-icon.json'
                        ],
                        'condition'   => [
                            'icon_type' => 'lordicon_external'
                        ]
                    ]);
                $repeater->end_controls_tab();
            $repeater->end_controls_tabs();

            $this->add_control('items', [
                'type'    => KFCM::REPEATER,
                'fields'  => $repeater->get_controls(),
                'default' => [
                    [
                        'title'     => 'Item #1',
                        'sub_title' => '',
                        'icon_type' => 'icon',
                        'icon'      => [
                            'value'   => 'fas fa-home',
                            'library' => 'fa-solid',
                        ],
                    ],
                    [
                        'title'     => 'Item #2',
                        'sub_title' => '',
                        'icon_type' => 'icon',
                        'icon'      => [
                            'value'   => 'fas fa-check',
                            'library' => 'fa-solid',
                        ],
                    ],
                    [
                        'title'     => 'Item #3',
                        'sub_title' => '',
                        'icon_type' => 'icon',
                        'icon'      => [
                            'value'   => 'fas fa-heart',
                            'library' => 'fa-solid',
                        ],
                    ],
                ]
            ]);

            $this->add_control('hr0', [
                'type' => KFCM::DIVIDER,
            ]);
            $this->add_control('use_title_icon',[
                'label'        => esc_html__( 'Use Icon?', 'onnat-extra' ),
                'type'         => KFCM::SWITCHER,
                'return_value' => 'yes',
                'default'      => 'yes'
            ]);
			$this->add_control('icon_position', [
				'label'        => esc_html__( 'Position', 'onnat-extra' ),
				'type'         => KFCM::CHOOSE,
				'default'      => 'left',
				'toggle'       => false,
				'options'      => [
					'left'   => [
						'title' => esc_html__( 'Left', 'onnat-extra' ),
						'icon'  => 'eicon-h-align-left',
					],
					'right'  => [
						'title' => esc_html__( 'Right', 'onnat-extra' ),
						'icon'  => 'eicon-h-align-right',
					],
                ],
				'prefix_class' => 'kfw-ele-custom-list-icon-position-',
				'condition'    => [
                    'use_title_icon!' => '',
                ],
            ]);
		$this->end_controls_section();
    }

    public function _register_list_style_section() {
		$this->start_controls_section( 'section_type_style', [
			'label' => esc_html__( 'List', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_responsive_control('items_align', [
                'label'          => esc_html__( 'Alignment', 'onnat-extra' ),
                'type'           => KFCM::CHOOSE,
                'options'        => [
                    'left'   => [
                        'title' => esc_html__( 'Left', 'onnat-extra' ),
                        'icon'  => 'eicon-text-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__( 'Center', 'onnat-extra' ),
                        'icon'  => 'eicon-text-align-center',
                    ],
                    'right'  => [
                        'title' => esc_html__( 'Right', 'onnat-extra' ),
                        'icon'  => 'eicon-text-align-right',
                    ],
                ],
                'toggle'               => false,
                'selectors_dictionary' => [
                    'left'   => 'justify-content:start;',
                    'center' => 'justify-content:center;',
                    'right'  => 'justify-content:end;',
                ],
                'selectors'            => [
                    '{{WRAPPER}} .kfw-elementor-custom-list-item' => '{{VALUE}};'
                ],
            ]);
            $this->add_responsive_control( 'items_gap', [
                'label'      => esc_html__('Items Gap', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'size_units' => [ 'px' ],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 200,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-elementor-custom-list-item:not(:last-child)' => 'margin-bottom:{{SIZE}}{{UNIT}};',
                ],
            ]);
            $this->add_responsive_control( 'items_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-elementor-custom-list-item' => 'padding:{{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->start_controls_tabs( 'items_customize_tabs');
                $this->start_controls_tab( 'items_normal_state', [
                    'label'     => esc_html__( 'Normal', 'onnat-extra' ),
                ]);
                    $this->add_group_control( KFGCBG::get_type(), [
                        'name'           => 'items_bg',
                        'types'          => ['classic', 'gradient'],
                        'exclude'        => [ 'image'],
                        'fields_options' => [
                            'background' => [ 'default' => 'classic' ],
                            'color'      => [ 'default' => '' ],
                        ],
                        'selector'       => '{{WRAPPER}} .kfw-elementor-custom-list-item',
                    ]);
                    $this->add_group_control( KFGCBR::get_type(), [
                        'name'      => 'items_border',
                        'selector'  => '{{WRAPPER}} .kfw-elementor-custom-list-item'
                    ]);
                    $this->add_responsive_control( 'items_border_radius', [
                        'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                        'type'       => KFCM::DIMENSIONS,
                        'size_units' => [ 'px', '%' ],
                        'selectors'  => [
                            '{{WRAPPER}} .kfw-elementor-custom-list-item' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                        ]
                    ]);
                    $this->add_group_control( KFGCBS::get_type(), [
                        'name'      => 'items_box_shadow',
                        'selector'  => '{{WRAPPER}} .kfw-elementor-custom-list-item'
                    ]);
                $this->end_controls_tab();
                $this->start_controls_tab( 'items_hover_state', [
                    'label'     => esc_html__( 'Hover', 'onnat-extra' ),
                ]);
                    $this->add_group_control( KFGCBG::get_type(), [
                        'name'           => 'items_hover_bg',
                        'types'          => ['classic', 'gradient'],
                        'exclude'        => [ 'image'],
                        'fields_options' => [
                            'background' => [ 'default' => 'classic' ],
                            'color'      => [ 'default' => '' ],
                        ],
                        'selector'       => '{{WRAPPER}} .kfw-elementor-custom-list-item:hover',
                    ]);
                    $this->add_group_control( KFGCBR::get_type(), [
                        'name'      => 'items_hover_border',
                        'selector'  => '{{WRAPPER}} .kfw-elementor-custom-list-item:hover'
                    ]);
                    $this->add_responsive_control( 'items_hover_border_radius', [
                        'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                        'type'       => KFCM::DIMENSIONS,
                        'size_units' => [ 'px', '%' ],
                        'selectors'  => [
                            '{{WRAPPER}} .kfw-elementor-custom-list-item:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                        ]
                    ]);
                    $this->add_group_control( KFGCBS::get_type(), [
                        'name'      => 'items_hover_box_shadow',
                        'selector'  => '{{WRAPPER}} .kfw-elementor-custom-list-item:hover'
                    ]);
                $this->end_controls_tab();
                $this->start_controls_tab( 'items_active_state', [
                    'label'     => esc_html__( 'Active', 'onnat-extra' ),
                ]);
                    $this->add_group_control( KFGCBG::get_type(), [
                        'name'           => 'items_active_bg',
                        'types'          => ['classic', 'gradient'],
                        'exclude'        => [ 'image'],
                        'fields_options' => [
                            'background' => [ 'default' => 'classic' ],
                            'color'      => [ 'default' => '' ],
                        ],
                        'selector'       => '{{WRAPPER}} .kfw-elementor-custom-list-item.kfw-elementor-custom-list-active-item',
                    ]);
                    $this->add_group_control( KFGCBR::get_type(), [
                        'name'      => 'items_active_border',
                        'selector'  => '{{WRAPPER}} .kfw-elementor-custom-list-item.kfw-elementor-custom-list-active-item'
                    ]);
                    $this->add_responsive_control( 'items_active_border_radius', [
                        'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                        'type'       => KFCM::DIMENSIONS,
                        'size_units' => [ 'px', '%' ],
                        'selectors'  => [
                            '{{WRAPPER}} .kfw-elementor-custom-list-item.kfw-elementor-custom-list-active-item' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                        ]
                    ]);
                    $this->add_group_control( KFGCBS::get_type(), [
                        'name'      => 'items_active_box_shadow',
                        'selector'  => '{{WRAPPER}} .kfw-elementor-custom-list-item.kfw-elementor-custom-list-active-item'
                    ]);
                $this->end_controls_tab();
            $this->end_controls_tabs();
		$this->end_controls_section();
    }

    public function _register_icon_style_section() {
		$this->start_controls_section( 'section_icon_style', [
			'label' => esc_html__( 'Icon', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_responsive_control( 'icon_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-custom-list-title-icon-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'icon_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-custom-list-title-icon-wrap .kfw-elementor-icon-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);

            $this->start_controls_tabs( 'icon_wrap_customize_tabs');
                $this->start_controls_tab( 'icon_wrap_normal_state', [
                    'label' => esc_html__( 'Normal', 'onnat-extra' ),
                ]);
                    $this->add_group_control( KFGCBG::get_type(), [
                        'name'           => 'icon_wrap_bg',
                        'types'          => ['classic', 'gradient'],
                        'exclude'        => [ 'image'],
                        'fields_options' => [
                            'background' => [ 'default' => 'classic' ],
                            'color'      => [ 'default' => '' ],
                        ],
                        'selector'       => '{{WRAPPER}} .kfw-custom-list-title-icon-wrap .kfw-elementor-icon-wrap',
                    ]);
                    $this->add_group_control( KFGCBR::get_type(), [
                        'name'      => 'handler_wrap_border',
                        'selector'  => '{{WRAPPER}} .kfw-custom-list-title-icon-wrap .kfw-elementor-icon-wrap'
                    ]);
                    $this->add_responsive_control( 'handler_wrap_border_radius', [
                        'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                        'type'       => KFCM::DIMENSIONS,
                        'size_units' => [ 'px', '%' ],
                        'selectors'  => [
                            '{{WRAPPER}} .kfw-custom-list-title-icon-wrap .kfw-elementor-icon-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                        ]
                    ]);
                    $this->add_group_control( KFGCBS::get_type(), [
                        'name'      => 'handler_wrap_box_shadow',
                        'selector'  => '{{WRAPPER}} .kfw-custom-list-title-icon-wrap .kfw-elementor-icon-wrap'
                    ]);
                $this->end_controls_tab();
                $this->start_controls_tab( 'icon_wrap_hover_state', [
                    'label' => esc_html__( 'Hover', 'onnat-extra' ),
                ]);
                    $this->add_group_control( KFGCBG::get_type(), [
                        'name'           => 'icon_wrap_hover_bg',
                        'types'          => ['classic', 'gradient'],
                        'exclude'        => [ 'image'],
                        'fields_options' => [
                            'background' => [ 'default' => 'classic' ],
                            'color'      => [ 'default' => '' ],
                        ],
                        'selector'       => '{{WRAPPER}} .kfw-elementor-custom-list-item:hover .kfw-custom-list-title-icon-wrap .kfw-elementor-icon-wrap',
                    ]);
                    $this->add_group_control( KFGCBR::get_type(), [
                        'name'      => 'handler_wrap_hover_border',
                        'selector'  => '{{WRAPPER}} .kfw-elementor-custom-list-item:hover .kfw-custom-list-title-icon-wrap .kfw-elementor-icon-wrap'
                    ]);
                    $this->add_responsive_control( 'handler_wrap_hover_border_radius', [
                        'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                        'type'       => KFCM::DIMENSIONS,
                        'size_units' => [ 'px', '%' ],
                        'selectors'  => [
                            '{{WRAPPER}} .kfw-elementor-custom-list-item:hover .kfw-custom-list-title-icon-wrap .kfw-elementor-icon-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                        ]
                    ]);
                    $this->add_group_control( KFGCBS::get_type(), [
                        'name'      => 'handler_wrap_hover_box_shadow',
                        'selector'  => '{{WRAPPER}} .kfw-elementor-custom-list-item:hover .kfw-custom-list-title-icon-wrap .kfw-elementor-icon-wrap'
                    ]);
                $this->end_controls_tab();
                $this->start_controls_tab( 'icon_wrap_active_state', [
                    'label' => esc_html__( 'Active', 'onnat-extra' ),
                ]);
                    $this->add_group_control( KFGCBG::get_type(), [
                        'name'           => 'icon_wrap_active_bg',
                        'types'          => ['classic', 'gradient'],
                        'exclude'        => [ 'image'],
                        'fields_options' => [
                            'background' => [ 'default' => 'classic' ],
                            'color'      => [ 'default' => '' ],
                        ],
                        'selector'       => '{{WRAPPER}} .kfw-elementor-custom-list-item.kfw-elementor-custom-list-active-item .kfw-custom-list-title-icon-wrap .kfw-elementor-icon-wrap',
                    ]);
                    $this->add_group_control( KFGCBR::get_type(), [
                        'name'      => 'handler_wrap_active_border',
                        'selector'  => '{{WRAPPER}} .kfw-elementor-custom-list-item.kfw-elementor-custom-list-active-item .kfw-custom-list-title-icon-wrap .kfw-elementor-icon-wrap'
                    ]);
                    $this->add_responsive_control( 'handler_wrap_active_border_radius', [
                        'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                        'type'       => KFCM::DIMENSIONS,
                        'size_units' => [ 'px', '%' ],
                        'selectors'  => [
                            '{{WRAPPER}} .kfw-elementor-custom-list-item.kfw-elementor-custom-list-active-item .kfw-custom-list-title-icon-wrap .kfw-elementor-icon-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                        ]
                    ]);
                    $this->add_group_control( KFGCBS::get_type(), [
                        'name'      => 'handler_wrap_active_box_shadow',
                        'selector'  => '{{WRAPPER}} .kfw-elementor-custom-list-item.kfw-elementor-custom-list-active-item .kfw-custom-list-title-icon-wrap .kfw-elementor-icon-wrap'
                    ]);
                $this->end_controls_tab();
            $this->end_controls_tabs();

            /**
             * Style: Icon Type
             */
                $this->add_control('icon_type_style_before_hr', [
                    'type' => KFCM::DIVIDER,
                ]);
                $this->add_control('icon_type_style_heading',[
					'label' => esc_html__( 'Icon Type Style', 'onnat-extra' ),
					'type'  => KFCM::HEADING,
				]);
				$this->add_control('icon_type_style_after_hr', [
					'type' => KFCM::DIVIDER,
				]);
                $this->add_responsive_control( 'icon_type_size', [
                    'label'      => esc_html__('Size', 'onnat-extra' ),
                    'type'       => KFCM::SLIDER,
                    'size_units' => [ 'px' ],
                    'range'      => [
                        'px' => [
                            'min' => 0,
                            'max' => 200,
                        ],
                    ],
                    'selectors'  => [
                        '{{WRAPPER}} .kfw-custom-list-title-icon-wrap.kfw-custom-list-title-icon-type .kfw-elementor-icon-wrap' => 'font-size:{{SIZE}}{{UNIT}};'
                    ],
                ]);
				$this->add_control( 'icon_color_settings', [
                    'label'        => esc_html__('Colors', 'onnat-extra'),
					'type'         => KFCM::POPOVER_TOGGLE,
					'return_value' => 'yes',
				]);
                $this->start_popover();
                    $this->add_control( 'icon_type_color', [
                        'label'     => esc_html__('Normal Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => [
                            '{{WRAPPER}} .kfw-custom-list-title-icon-wrap.kfw-custom-list-title-icon-type .kfw-elementor-icon-wrap svg' => 'fill:{{VALUE}};',
                            '{{WRAPPER}} .kfw-custom-list-title-icon-wrap.kfw-custom-list-title-icon-type .kfw-elementor-icon-wrap i'   => 'color:{{VALUE}};',
                        ],
                    ]);
                    $this->add_control( 'icon_type_hover_color', [
                        'label'     => esc_html__('Hover Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => [
                            '{{WRAPPER}} .kfw-elementor-custom-list-item:hover .kfw-custom-list-title-icon-wrap.kfw-custom-list-title-icon-type .kfw-elementor-icon-wrap svg' => 'fill:{{VALUE}};',
                            '{{WRAPPER}} .kfw-elementor-custom-list-item:hover .kfw-custom-list-title-icon-wrap.kfw-custom-list-title-icon-type .kfw-elementor-icon-wrap i'   => 'color:{{VALUE}};',
                        ],
                    ]);
                    $this->add_control( 'icon_type_active_color', [
                        'label'     => esc_html__('Active Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => [
                            '{{WRAPPER}} .kfw-elementor-custom-list-item.kfw-elementor-custom-list-active-item .kfw-custom-list-title-icon-wrap.kfw-custom-list-title-icon-type .kfw-elementor-icon-wrap svg' => 'fill:{{VALUE}};',
                            '{{WRAPPER}} .kfw-elementor-custom-list-item.kfw-elementor-custom-list-active-item .kfw-custom-list-title-icon-wrap.kfw-custom-list-title-icon-type .kfw-elementor-icon-wrap i'   => 'color:{{VALUE}};',
                        ],
                    ]);
                $this->end_popover();

            /**
             * Style: Image Type
             */
                $this->add_control('img_type_style_before_hr', [
                    'type' => KFCM::DIVIDER,
                ]);
                $this->add_control('img_type_style_heading',[
                    'label' => esc_html__( 'Image Type Style', 'onnat-extra' ),
                    'type'  => KFCM::HEADING,
                ]);
                $this->add_control('img_type_style_after_hr', [
                    'type' => KFCM::DIVIDER,
                ]);
                $this->add_responsive_control( 'img_type_size', [
                    'label'      => esc_html__('Size', 'onnat-extra' ),
                    'type'       => KFCM::SLIDER,
                    'size_units' => ['px'],
                    'range'      => [
                        'px' => [
                            'min' => 5,
                            'max' => 200,
                        ],
                    ],
                    'selectors'  => [
                        '{{WRAPPER}} .kfw-custom-list-title-icon-wrap.kfw-custom-list-title-image-type img' => 'max-width:{{SIZE}}px;'
                    ],
                ]);

            /**
             * Style: Lordicon Type
             */
                $this->add_control('lordicon_type_style_before_hr', [
                    'type' => KFCM::DIVIDER,
                ]);
                $this->add_control('lordicon_type_style_heading',[
                    'label' => esc_html__( 'LordIcon Type Style', 'onnat-extra' ),
                    'type'  => KFCM::HEADING,
                ]);
                $this->add_control('lordicon_type_style_after_hr', [
                    'type' => KFCM::DIVIDER,
                ]);
                $this->add_responsive_control( 'lordicon_type_size', [
                    'label'      => esc_html__('Size', 'onnat-extra' ),
                    'type'       => KFCM::SLIDER,
                    'size_units' => [ 'px' ],
                    'range'      => [
                        'px' => [
                            'min' => 1,
                            'max' => 300,
                        ],
                    ],
                    'selectors'  => [
                        '{{WRAPPER}} .kfw-custom-list-title-icon-wrap.kfw-custom-list-title-lordicon-type .kfw-elementor-lordicon' => 'width:{{SIZE}}{{UNIT}};height:{{SIZE}}{{UNIT}};'
                    ]
                ]);
				$this->add_control( 'lordicon_color_settings', [
                    'label'        => esc_html__('Colors', 'onnat-extra'),
					'type'         => KFCM::POPOVER_TOGGLE,
					'return_value' => 'yes',
				]);
                $this->start_popover();
                    $this->add_control('lordicon_primary_color',[
                        'label'     => __( 'Primary Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'default'   => '#121331',
                    ]);
                    $this->add_control('lordicon_secondary_color',[
                        'label'   => __( 'Secondary Color', 'onnat-extra' ),
                        'type'    => KFCM::COLOR,
                        'default' => '#121331',
                    ]);
                    $this->add_control('lordicon_tertiary_color',[
                        'label'   => __( 'Tertiary Color', 'onnat-extra' ),
                        'type'    => KFCM::COLOR,
                        'default' => '#0816A8',
                    ]);
                $this->end_popover();
		$this->end_controls_section();
    }

    public function _register_title_style_section() {
		$this->start_controls_section( 'section_title_style', [
			'label' => esc_html__( 'Title', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_group_control(KFGCT::get_type(), [
                'name'     => 'title_typo',
                'label'    => esc_html__( 'Typography', 'onnat-extra' ),
                'selector' => '{{WRAPPER}} .kfw-custom-list-title',
            ]);
            $this->add_responsive_control( 'title_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-custom-list-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'title_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-custom-list-title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->start_controls_tabs( 'title_customize_tabs' );
                $this->start_controls_tab( 'title_normal_state', [
                    'label' => esc_html__( 'Normal', 'onnat-extra' ),
                ]);
                    $this->add_control( 'title_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kfw-custom-list-title' => 'color: {{VALUE}};' ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'     => 'title_txt_shadow',
                        'selector' => '{{WRAPPER}} .kfw-custom-list-title',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'     => 'title_txt_stroke',
                        'selector' => '{{WRAPPER}} .kfw-custom-list-title',
                    ]);
                $this->end_controls_tab();
                $this->start_controls_tab( 'title_hover_state', [
                    'label' => esc_html__( 'Hover', 'onnat-extra' ),
                ]);
                    $this->add_control( 'title_hover_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kfw-elementor-custom-list-item:hover .kfw-custom-list-title' => 'color: {{VALUE}};' ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'     => 'title_hover_txt_shadow',
                        'selector' => '{{WRAPPER}} .kfw-elementor-custom-list-item:hover .kfw-custom-list-title',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'     => 'title_hover_txt_stroke',
                        'selector' => '{{WRAPPER}} .kfw-elementor-custom-list-item:hover .kfw-custom-list-title',
                    ]);
                $this->end_controls_tab();
                $this->start_controls_tab( 'title_active_state', [
                    'label' => esc_html__( 'Active', 'onnat-extra' ),
                ]);
                    $this->add_control( 'title_active_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kfw-elementor-custom-list-item.kfw-elementor-custom-list-active-item .kfw-custom-list-title' => 'color: {{VALUE}};' ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'     => 'title_active_txt_shadow',
                        'selector' => '{{WRAPPER}} .kfw-elementor-custom-list-item.kfw-elementor-custom-list-active-item .kfw-custom-list-title',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'     => 'title_active_txt_stroke',
                        'selector' => '{{WRAPPER}} .kfw-elementor-custom-list-item.kfw-elementor-custom-list-active-item .kfw-custom-list-title',
                    ]);
                $this->end_controls_tab();
            $this->end_controls_tabs();
		$this->end_controls_section();
    }

    public function _register_sub_title_style_section() {
		$this->start_controls_section( 'section_sub_title_style', [
			'label' => esc_html__( 'Sub Title', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_group_control(KFGCT::get_type(), [
                'name'     => 'sub_title_typo',
                'label'    => esc_html__( 'Typography', 'onnat-extra' ),
                'selector' => '{{WRAPPER}} .kfw-custom-list-sub-title',
            ]);
            $this->add_responsive_control( 'sub_title_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-custom-list-sub-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'sub_title_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-custom-list-sub-title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->start_controls_tabs( 'sub_title_customize_tabs' );
                $this->start_controls_tab( 'sub_title_normal_state', [
                    'label' => esc_html__( 'Normal', 'onnat-extra' ),
                ]);
                    $this->add_control( 'sub_title_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kfw-custom-list-sub-title' => 'color: {{VALUE}};' ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'     => 'sub_title_txt_shadow',
                        'selector' => '{{WRAPPER}} .kfw-custom-list-sub-title',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'     => 'sub_title_txt_stroke',
                        'selector' => '{{WRAPPER}} .kfw-custom-list-sub-title',
                    ]);
                $this->end_controls_tab();
                $this->start_controls_tab( 'sub_title_hover_state', [
                    'label' => esc_html__( 'Hover', 'onnat-extra' ),
                ]);
                    $this->add_control( 'sub_title_hover_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kfw-elementor-custom-list-item:hover .kfw-custom-list-sub-title' => 'color: {{VALUE}};' ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'     => 'sub_title_hover_txt_shadow',
                        'selector' => '{{WRAPPER}} .kfw-elementor-custom-list-item:hover .kfw-custom-list-sub-title',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'     => 'sub_title_hover_txt_stroke',
                        'selector' => '{{WRAPPER}} .kfw-elementor-custom-list-item:hover .kfw-custom-list-sub-title',
                    ]);
                $this->end_controls_tab();
                $this->start_controls_tab( 'sub_title_active_state', [
                    'label' => esc_html__( 'Active', 'onnat-extra' ),
                ]);
                    $this->add_control( 'sub_title_active_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kfw-elementor-custom-list-item.kfw-elementor-custom-list-active-item .kfw-custom-list-sub-title' => 'color: {{VALUE}};' ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'     => 'sub_title_active_txt_shadow',
                        'selector' => '{{WRAPPER}} .kfw-elementor-custom-list-item.kfw-elementor-custom-list-active-item .kfw-custom-list-sub-title',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'     => 'sub_title_active_txt_stroke',
                        'selector' => '{{WRAPPER}} .kfw-elementor-custom-list-item.kfw-elementor-custom-list-active-item .kfw-custom-list-sub-title',
                    ]);
                $this->end_controls_tab();
            $this->end_controls_tabs();
		$this->end_controls_section();
    }

	public function _render_icon( $icon , $echo = true ) {
		$icon = array_filter( $icon );

		if( !empty( $icon ) ) {
            echo '<div class="kfw-custom-list-title-icon-wrap kfw-custom-list-title-icon-type">';
                echo '<span class="kfw-elementor-icon-wrap">';
                    KFIM::render_icon( $icon, [ 'aria-hidden' => 'true' ] );
                echo '</span>';
            echo '</div>';
		}
    }

    public function _render_img( $settings = [] ) {
        if( !empty( $settings['url'] ) ) {
            $image_src = $settings['url'];

			$image_title = $this->_get_image_title( $settings );
			$image_title = empty( $image_title ) ? $this->get_title() : $image_title;

			$image_alt = $this->_get_image_alt( $settings );
			$image_alt = empty( $image_alt ) ? $image_title : $image_title;

            echo '<div class="kfw-custom-list-title-icon-wrap kfw-custom-list-title-image-type">';
                echo '<span class="kfw-elementor-icon-wrap">';
                    echo '<img alt="'. esc_attr( $image_alt ) .'" title="'. esc_attr( $image_title ) .'" src="'. esc_url( $image_src ) .'"/>';
                echo '</span>';
            echo '</div>';
        }
    }

	public function _get_image_title( $attachment ) {
		if ( empty( $attachment['id'] ) ) {
			return '';
		}

		return get_the_title( $attachment['id'] );
	}

	public function _get_image_alt( $instance ) {
		if ( empty( $instance['id'] ) ) {
			// For `Insert From URL` images.
			return isset( $instance['alt'] ) ? trim( strip_tags( $instance['alt'] ) ) : '';
		}

		$attachment_id = $instance['id'];
		if ( ! $attachment_id ) {
			return '';
		}

		$attachment = get_post( $attachment_id );
		if ( ! $attachment ) {
			return '';
		}

		$alt = get_post_meta( $attachment_id, '_wp_attachment_image_alt', true );
		if ( ! $alt ) {
			$alt = $attachment->post_excerpt;
			if ( ! $alt ) {
				$alt = $attachment->post_title;
			}
		}
		return trim( strip_tags( $alt ) );
	}

    public function _is_editor_mode() {
        $preview = KFEl::instance()->preview->is_preview_mode();
        $editor  = KFEl::instance()->editor->is_edit_mode();

        return ( $preview || $editor ) ? true : false;
    }

    public function _get_current_url() {
        $protocol = is_ssl() ? 'https://' : 'http://';
        return ($protocol) . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'];
    }

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-custom-list");
		include $render;
    }
}