<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}
use \Elementor\Plugin as KFEl;
use \Elementor\Widget_Base as WB;
use \Elementor\Controls_Manager as KFCM;
use \Elementor\Icons_Manager as KFIM;
use \Elementor\Repeater as KFRTR;

use \Elementor\Group_Control_Image_Size as KFGCIS;
use \Elementor\Group_Control_Background as KFGCBG;
use \Elementor\Group_Control_Typography as KFGCT;
use \Elementor\Group_Control_Border as KFGCBR;
use \Elementor\Group_Control_Box_Shadow as KFGCBS;
use \Elementor\Group_Control_Text_Shadow as KFGCTS;
use \Elementor\Group_Control_Text_Stroke as KFGCTST;

class KFW_Horizontal_Scroll_Element extends WB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-horizontal-scroll-element';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Horizontal Scroll Element', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_core_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-navigation-horizontal';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'kfw', 'kin', 'horiz', 'scroll', 'ele', 'gsap' ];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
        return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/horizontal-scroll-element';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

        if ( ! wp_script_is( 'kfw-gsap', 'enqueued' ) ) {
            wp_register_script( 'kfw-gsap', KFE_CONST_URL . 'assets/js/gsap.min.js', [], KFE_CONST_VERSION, true );
        }

        if ( ! wp_script_is( 'kfw-gsap-scroll-trigger', 'enqueued' ) ) {
            wp_register_script( 'kfw-gsap-scroll-trigger', KFE_CONST_URL . 'assets/js/ScrollTrigger.min.js', ['kfw-gsap'], KFE_CONST_VERSION, true );
        }

        wp_register_script( 'lordicon',
            KFE_CONST_URL . 'libraries/elementor/assets/js/lordicon.js',
            [],
            KFE_CONST_VERSION,
            true
        );

		wp_register_script( 'kfw-horizontal-scroll-element',
			KFE_CONST_URL . 'libraries/elementor/assets/js/horizontal-scroll-element' . KFE_CONST_DEBUG_SUFFIX . '.js',
			[ 'jquery', 'elementor-frontend' ],
			KFE_CONST_VERSION,
			true
		);

		wp_register_style( 'kfw-horizontal-scroll-element',
			KFE_CONST_URL . 'libraries/elementor/assets/css/horizontal-scroll-element' . KFE_CONST_DEBUG_SUFFIX . '.css',
			[],
			KFE_CONST_VERSION
		);

    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'kfw-horizontal-scroll-element' ];
	}

	/**
	 * Get script dependencies.
	 *
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @return array Element scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'lordicon', 'kfw-gsap', 'kfw-gsap-scroll-trigger', 'kfw-horizontal-scroll-element' ];
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
        $this->_register_items_section();
		$this->_register_content_wrap_style_section();
		$this->_register_title_style_section();
		$this->_register_prefix_style_section();
		$this->_register_icon_style_section();
		$this->_register_content_style_section();
		$this->_register_line_style_section();
		$this->_register_skin_style_section();
	}

	public function _register_items_section() {
		$repeater = new KFRTR();
		$this->start_controls_section( 'section_items', [
			'label' => esc_html__( 'Items', 'onnat-extra' )
		]);
			$repeater->start_controls_tabs('tabs');
				$repeater->start_controls_tab('tab_content',[
					'label' => esc_html__( 'Content', 'onnat-extra' ),
				]);
					$repeater->add_control('prefix',[
						'label'       => esc_html__( 'Prefix', 'onnat-extra' ),
						'type'        => KFCM::TEXT,
						'dynamic'     => [ 'active' => true, ],
						'default'     => '01',
						'label_block' => true,
					]);
					$repeater->add_control('title',[
						'label'       => esc_html__( 'Title', 'onnat-extra' ),
						'type'        => KFCM::TEXT,
						'dynamic'     => [ 'active' => true ],
						'default'     => esc_html__( 'Enter Title' , 'onnat-extra' ),
						'label_block' => true,
					]);
                    $repeater->add_control( 'editor', [
                        'label'   => esc_html__('Editor', 'onnat-extra'),
                        'type'    => KFCM::WYSIWYG,
                        'default' => "Fusce at erat libero. Curabitur tortor metus, laoreet vitae aliquam rutrum, imperdiet ac orci. Phasellus sagittis mollis viverra."
                    ]);
				$repeater->end_controls_tab();
				$repeater->start_controls_tab('tab_icon',[
					'label' => esc_html__( 'Icon', 'onnat-extra' ),
				]);
					$repeater->add_control('icon_type',[
						'label'   => esc_html__( 'Icon Type', 'onnat-extra' ),
						'type'    => KFCM::CHOOSE,
						'options' => [
							'blank'             => [
								'title' => esc_html__('Blank', 'onnat-extra'),
								'icon'  => '',
							],
							'icon'              => [
								'title' => esc_html__('Icon', 'onnat-extra'),
								'icon'  => 'eicon-plus-circle-o',
							],
							'image'             => [
								'title' => esc_html__('Image', 'onnat-extra'),
								'icon'  => 'eicon-image',
							],
							'text'           => [
								'title' => esc_html__('Text', 'onnat-extra'),
								'icon'  => 'eicon-t-letter',
							],
							'lordicon_media' => [
								'title' => esc_html__('Upload Lordicon From Media', 'onnat-extra'),
								'icon'  => 'eicon-library-upload',
							],
							'lordicon_external' => [
								'title' => esc_html__('Upload Lordicon From Extrnal', 'onnat-extra'),
								'icon'  => 'eicon-editor-external-link',
							],
						],
						'default' => 'blank',
						'toggle'  => false
					]);
					$repeater->add_control( 'icon', [
                        'label'     => esc_html__( 'Icon', 'onnat-extra' ),
                        'type'      => KFCM::ICONS,
                        'default'   => [
                            'value'   => 'fas fa-heart',
                            'library' => 'fa-solid',
                        ],
                        'condition' => [
                            'icon_type' => 'icon'
                        ],
                    ]);
                    $repeater->add_control( 'image', [
                        'label'     => esc_html__( 'Image', 'onnat-extra' ),
                        'type'      => KFCM::MEDIA,
                        'dynamic'   => [ 'active' => true ],
                        'default'   => [
                            'url' => \Elementor\Utils::get_placeholder_image_src()
                        ],
                        'condition' => [
                            'icon_type' => 'image'
                        ],
                    ]);
					$repeater->add_control('text',[
						'label'       => esc_html__( 'Text', 'onnat-extra' ),
						'type'        => KFCM::TEXT,
						'dynamic'     => [ 'active' => true ],
						'default'     => esc_html__( 'A' , 'onnat-extra' ),
						'label_block' => true,
						'condition'   => [
                            'icon_type' => 'text'
                        ],
					]);
                    $repeater->add_control('lordicon_des', [
                        'type'            => KFCM::RAW_HTML,
                        'content_classes' => 'kfw-elementor-editor-notice-ctrl',
                        'raw'             => sprintf(
                            esc_html__('You can get lordicon code from %1$s ', 'onnat-extra'),
                            sprintf('<a href="%1$s" target="_blank">%2$s</a>', esc_url( 'https://lordicon.com/' ), esc_html_x( 'Lordicon.com', 'link text for lordicon.com', 'onnat-extra' ) )
                        ),
                        'condition'       => [
                            'icon_type' => [ 'lordicon_media', 'lordicon_external' ]
                        ],
                    ]);
                    $repeater->add_control('lordicon_media', [
                        'label'      => esc_html__( 'Upload JSON File', 'onnat-extra' ),
                        'type'       => KFCM::MEDIA,
                        'media_type' => 'application/json',
                        'default'    => [
                            'url' => KFE_CONST_URL . 'libraries/elementor/assets/json/lordicon-clock-icon.json'
                        ],
                        'condition'  => [
                            'icon_type' => 'lordicon_media'
                        ]
                    ]);
                    $repeater->add_control('lordicon_external', [
                        'label'       => esc_html__( 'Upload JSON File', 'onnat-extra' ),
                        'type'        => KFCM::URL,
                        'dynamic'     => [ 'active' => true, ],
                        'placeholder' => esc_html__( 'Enter your URL', 'onnat-extra' ),
                        'options'     => [ 'url' ],
                        'default'     => [
                            'url' => KFE_CONST_URL . 'libraries/elementor/assets/json/lordicon-clock-icon.json'
                        ],
                        'condition'   => [
                            'icon_type' => 'lordicon_external'
                        ]
                    ]);
				$repeater->end_controls_tab();
			$repeater->end_controls_tabs();

			$this->add_control('items', [
				'type'        => KFCM::REPEATER,
				'fields'      => $repeater->get_controls(),
				'default'     => [
					[
						'prefix'    => '01',
						'title'     => 'Item 01',
						'editor'    => 'Fusce at erat libero. Curabitur tortor metus, laoreet vitae aliquam rutrum, imperdiet ac orci. Phasellus sagittis mollis viverra.',
						'icon_type' => 'icon',
						'icon'      => [
                            'value'   => 'fas fa-home',
                            'library' => 'fa-solid',
						]
					],
					[
						'prefix'    => '02',
						'title'     => 'Item 02',
						'editor'    => 'Fusce at erat libero. Curabitur tortor metus, laoreet vitae aliquam rutrum, imperdiet ac orci. Phasellus sagittis mollis viverra.',
						'icon_type' => 'icon',
						'icon'      => [
                            'value'   => 'fas fa-chess-king',
                            'library' => 'fa-solid',
						]
					],
					[
						'prefix'    => '03',
						'title'     => 'Item 03',
						'editor'    => 'Fusce at erat libero. Curabitur tortor metus, laoreet vitae aliquam rutrum, imperdiet ac orci. Phasellus sagittis mollis viverra.',
						'icon_type' => 'icon',
						'icon'      => [
                            'value'   => 'fas fa-chess-queen',
                            'library' => 'fa-solid',
						]
					],
					[
						'prefix'    => '04',
						'title'     => 'Item 04',
						'editor'    => 'Fusce at erat libero. Curabitur tortor metus, laoreet vitae aliquam rutrum, imperdiet ac orci. Phasellus sagittis mollis viverra.',
						'icon_type' => 'icon',
						'icon'      => [
                            'value'   => 'fas fa-chess-bishop',
                            'library' => 'fa-solid',
						]
					],
					[
						'prefix'    => '05',
						'title'     => 'Item 05',
						'editor'    => 'Fusce at erat libero. Curabitur tortor metus, laoreet vitae aliquam rutrum, imperdiet ac orci. Phasellus sagittis mollis viverra.',
						'icon_type' => 'icon',
						'icon'      => [
                            'value'   => 'fas fa-chess-rook',
                            'library' => 'fa-solid',
						]
					],
					[
						'prefix'    => '06',
						'title'     => 'Item 06',
						'editor'    => 'Fusce at erat libero. Curabitur tortor metus, laoreet vitae aliquam rutrum, imperdiet ac orci. Phasellus sagittis mollis viverra.',
						'icon_type' => 'icon',
						'icon'      => [
                            'value'   => 'fas fa-chess-knight',
                            'library' => 'fa-solid',
						]
					],
					[
						'prefix'    => '07',
						'title'     => 'Item 07',
						'editor'    => 'Fusce at erat libero. Curabitur tortor metus, laoreet vitae aliquam rutrum, imperdiet ac orci. Phasellus sagittis mollis viverra.',
						'icon_type' => 'icon',
						'icon'      => [
                            'value'   => 'fas fa-chess-pawn',
                            'library' => 'fa-solid',
						]
					],
					[
						'prefix'    => '08',
						'title'     => 'Item 08',
						'editor'    => 'Fusce at erat libero. Curabitur tortor metus, laoreet vitae aliquam rutrum, imperdiet ac orci. Phasellus sagittis mollis viverra.',
						'icon_type' => 'icon',
						'icon'      => [
                            'value'   => 'fas fa-heart',
                            'library' => 'fa-solid',
						]
					],
				],
				'title_field' => '{{{ title }}}'
			]);
            $this->add_control('hr_0', [
                'type' => KFCM::DIVIDER,
            ]);
			$this->add_control( 'top_section', [
				'label'   => esc_html__('Top Section', 'onnat-extra'),
				'type'    => KFCM::SELECT,
				'default' => 0,
				'options' => kf_extra_plugin_elementor()->get_elementor_library(),
			]);
            $this->add_control('use_line',[
                'label'        => esc_html__( 'Add Line?', 'onnat-extra' ),
                'type'         => KFCM::SWITCHER,
                'return_value' => 'add',
                'default'      => 'add',
                'prefix_class' => 'kfw-horiz-scroll-ele-line-',
            ]);
			$this->add_responsive_control('cols', [
				'label'          => esc_html__( 'Columns', 'onnat-extra' ),
				'type'           => KFCM::CHOOSE,
				'options'        => [
					'100%'   => [ 'title' => esc_html__('1 Column', 'onnat-extra'), 'icon' => 'kfw-ctrl-icon kfw-ctrl-numeric-icon' ],
					'50%'    => [ 'title' => esc_html__('2 Columns', 'onnat-extra'), 'icon' => 'kfw-ctrl-icon kfw-ctrl-numeric-2-icon' ],
					'33.33%' => [ 'title' => esc_html__('3 Columns', 'onnat-extra'), 'icon' => 'kfw-ctrl-icon kfw-ctrl-numeric-3-icon' ],
					'25%'    => [ 'title' => esc_html__('4 Columns', 'onnat-extra'), 'icon' => 'kfw-ctrl-icon kfw-ctrl-numeric-4-icon' ],
					'20%'    => [ 'title' => esc_html__('5 Columns', 'onnat-extra'), 'icon' => 'kfw-ctrl-icon kfw-ctrl-numeric-5-icon' ],
				],
				'default'        => '33.33%',
				'tablet_default' => '50%',
				'mobile_default' => '100%',
				'selectors'      => [
					'{{WRAPPER}} .kfw-elementor-split-horiz-scroll-ele-item' => 'flex: 0 0 {{SIZE}};',
				],
				'toggle'         => false
			]);

		$this->end_controls_section();
	}

	public function _register_content_wrap_style_section() {
		$this->start_controls_section( 'section_content_wrap_style', [
			'label' => esc_html__( 'Content Wrap', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_responsive_control( 'content_wrap_margin', [
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-horiz-scroll-ele-info-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_responsive_control( 'content_wrap_padding', [
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-horiz-scroll-ele-info-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_group_control( KFGCBG::get_type(), [
				'name'           => 'content_wrap_bg',
				'types'          => ['classic', 'gradient'],
				'exclude'        => [ 'image'],
				'fields_options' => [
					'background' => [ 'default' => 'classic' ],
					'color'      => [ 'default' => '' ],
				],
				'selector'       => '{{WRAPPER}} .kfw-horiz-scroll-ele-info-wrap',
			]);
			$this->add_group_control( KFGCBR::get_type(), [
				'name'      => 'content_wrap_border',
				'selector'  => '{{WRAPPER}} .kfw-horiz-scroll-ele-info-wrap'
			]);
			$this->add_responsive_control( 'content_wrap_border_radius', [
				'label'      => esc_html__('Border Radius', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-horiz-scroll-ele-info-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_group_control( KFGCBS::get_type(), [
				'name'      => 'content_wrap_box_shadow',
				'selector'  => '{{WRAPPER}} .kfw-horiz-scroll-ele-info-wrap'
			]);
		$this->end_controls_section();
	}

    public function _register_title_style_section() {
		$this->start_controls_section( 'section_title_style', [
			'label' => esc_html__( 'Title', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_group_control(KFGCT::get_type(), [
                'name'     => 'title_typo',
                'label'    => esc_html__( 'Typography', 'onnat-extra' ),
                'selector' => '{{WRAPPER}} .kfw-horiz-scroll-ele-title',
            ]);
            $this->add_responsive_control( 'title_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-horiz-scroll-ele-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'title_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-horiz-scroll-ele-title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
			$this->add_control( 'title_color', [
				'label'     => esc_html__('Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => ['{{WRAPPER}} .kfw-horiz-scroll-ele-title' => 'color: {{VALUE}};' ],
			]);
			$this->add_group_control( KFGCTS::get_type(), [
				'name'     => 'title_txt_shadow',
				'selector' => '{{WRAPPER}} .kfw-horiz-scroll-ele-title',
			]);
			$this->add_group_control( KFGCTST::get_type(), [
				'name'     => 'title_txt_stroke',
				'selector' => '{{WRAPPER}} .kfw-horiz-scroll-ele-title',
			]);
		$this->end_controls_section();
    }

    public function _register_prefix_style_section() {
		$this->start_controls_section( 'section_prefix_style', [
			'label' => esc_html__( 'Prefix', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_group_control(KFGCT::get_type(), [
                'name'     => 'prefix_typo',
                'label'    => esc_html__( 'Typography', 'onnat-extra' ),
                'selector' => '{{WRAPPER}} .kfw-horiz-scroll-ele-prefix',
            ]);
            $this->add_responsive_control( 'prefix_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-horiz-scroll-ele-prefix' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'prefix_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-horiz-scroll-ele-prefix' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
			$this->add_control( 'prefix_color', [
				'label'     => esc_html__('Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => ['{{WRAPPER}} .kfw-horiz-scroll-ele-prefix' => 'color: {{VALUE}};' ],
			]);
			$this->add_group_control( KFGCTS::get_type(), [
				'name'     => 'prefix_txt_shadow',
				'selector' => '{{WRAPPER}} .kfw-horiz-scroll-ele-prefix',
			]);
			$this->add_group_control( KFGCTST::get_type(), [
				'name'     => 'prefix_txt_stroke',
				'selector' => '{{WRAPPER}} .kfw-horiz-scroll-ele-prefix',
			]);
		$this->end_controls_section();
    }

    public function _register_icon_style_section() {
		$this->start_controls_section( 'section_icon_style', [
			'label' => esc_html__( 'Icon', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_control('icon_wrap_style_heading',[
				'label' => esc_html__( 'Icon Wrap', 'onnat-extra' ),
				'type'  => KFCM::HEADING,
			]);
			$this->add_control('icon_wrap_style_after_hr', [
				'type' => KFCM::DIVIDER,
			]);
			$this->add_responsive_control( 'icon_wrap_size', [
				'label'      => esc_html__('Size', 'onnat-extra' ),
				'type'       => KFCM::SLIDER,
				'size_units' => ['px'],
				'range'      => [
					'px' => [
						'min' => 5,
						'max' => 200,
					],
				],
				'selectors'  => [
					'{{WRAPPER}} .kfw-horiz-scroll-ele-title-icon-wrap' => 'width:{{SIZE}}px; height:{{SIZE}}px;',
				],
			]);
			$this->add_group_control( KFGCBG::get_type(), [
				'name'           => 'icon_bg',
				'types'          => ['classic', 'gradient'],
				'exclude'        => [ 'image'],
				'fields_options' => [
					'background' => [ 'default' => 'classic' ],
					'color'      => [ 'default' => '' ],
				],
				'selector'       => '{{WRAPPER}} .kfw-horiz-scroll-ele-title-icon-wrap',
			]);
			$this->add_group_control( KFGCBR::get_type(), [
				'name'      => 'icon_border',
				'selector'  => '{{WRAPPER}} .kfw-horiz-scroll-ele-title-icon-wrap'
			]);
			$this->add_responsive_control( 'icon_border_radius', [
				'label'      => esc_html__('Border Radius', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-horiz-scroll-ele-title-icon-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				]
			]);
			$this->add_group_control( KFGCBS::get_type(), [
				'name'      => 'icon_box_shadow',
				'selector'  => '{{WRAPPER}} .kfw-horiz-scroll-ele-title-icon-wrap'
			]);
            $this->add_responsive_control( 'icon_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-horiz-scroll-ele-title-icon-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'icon_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-horiz-scroll-ele-title-icon-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);

            /**
             * Style: Icon Type
             */
                $this->add_control('icon_type_style_before_hr', [
                    'type' => KFCM::DIVIDER,
                ]);
                $this->add_control('icon_type_style_heading',[
					'label' => esc_html__( 'Icon Type Style', 'onnat-extra' ),
					'type'  => KFCM::HEADING,
				]);
				$this->add_control('icon_type_style_after_hr', [
					'type' => KFCM::DIVIDER,
				]);
                $this->add_responsive_control( 'icon_type_size', [
                    'label'      => esc_html__('Size', 'onnat-extra' ),
                    'type'       => KFCM::SLIDER,
                    'size_units' => [ 'px' ],
                    'range'      => [
                        'px' => [
                            'min' => 0,
                            'max' => 200,
                        ],
                    ],
                    'selectors'  => [
                        '{{WRAPPER}} .kfw-horiz-scroll-ele-title-icon-wrap svg, {{WRAPPER}} .kfw-horiz-scroll-ele-title-icon-wrap i' => 'font-size: {{SIZE}}{{UNIT}};'
                    ],
                ]);
				$this->add_control( 'icon_type_color', [
					'label'     => esc_html__('Normal Color', 'onnat-extra' ),
					'type'      => KFCM::COLOR,
					'selectors' => [
						'{{WRAPPER}} .kfw-horiz-scroll-ele-title-icon-wrap svg, {{WRAPPER}} .kfw-horiz-scroll-ele-title-icon-wrap i' => 'color:{{VALUE}};fill:{{VALUE}};',
					],
				]);

			/**
			 * Style: Text Type
			 */
				$this->add_control('txt_type_style_before_hr', [
					'type' => KFCM::DIVIDER,
				]);
				$this->add_control('txt_type_style_heading',[
					'label' => esc_html__( 'Text Type Style', 'onnat-extra' ),
					'type'  => KFCM::HEADING,
				]);
				$this->add_control('txt_type_style_after_hr', [
					'type' => KFCM::DIVIDER,
				]);
				$this->add_group_control(KFGCT::get_type(), [
					'name'     => 'txt_type_typo',
					'label'    => esc_html__( 'Typography', 'onnat-extra' ),
					'selector' => '{{WRAPPER}} .kfw-horiz-scroll-ele-title-icon-wrap.kfw-horiz-scroll-ele-title-txt-type .kfw-elementor-icon-wrap',
				]);
				$this->add_control( 'txt_type_color', [
					'label'     => esc_html__('Color', 'onnat-extra' ),
					'type'      => KFCM::COLOR,
					'selectors' => [
						'{{WRAPPER}} .kfw-horiz-scroll-ele-title-icon-wrap.kfw-horiz-scroll-ele-title-txt-type .kfw-elementor-icon-wrap' => 'color: {{VALUE}};'
					],
				]);
				$this->add_group_control( KFGCTS::get_type(), [
					'name'     => 'txt_type_txt_shadow',
					'selector' => '{{WRAPPER}} .kfw-horiz-scroll-ele-title-icon-wrap.kfw-horiz-scroll-ele-title-txt-type .kfw-elementor-icon-wrap',
				]);
				$this->add_group_control( KFGCTST::get_type(), [
					'name'     => 'txt_type_txt_stroke',
					'selector' => '{{WRAPPER}} .kfw-horiz-scroll-ele-title-icon-wrap.kfw-horiz-scroll-ele-title-txt-type .kfw-elementor-icon-wrap',
				]);

            /**
             * Style: Image Type
             */
                $this->add_control('img_type_style_before_hr', [
                    'type' => KFCM::DIVIDER,
                ]);
                $this->add_control('img_type_style_heading',[
                    'label' => esc_html__( 'Image Type Style', 'onnat-extra' ),
                    'type'  => KFCM::HEADING,
                ]);
                $this->add_control('img_type_style_after_hr', [
                    'type' => KFCM::DIVIDER,
                ]);
                $this->add_responsive_control( 'img_type_size', [
                    'label'      => esc_html__('Size', 'onnat-extra' ),
                    'type'       => KFCM::SLIDER,
                    'size_units' => ['px'],
                    'range'      => [
                        'px' => [
                            'min' => 5,
                            'max' => 200,
                        ],
                    ],
                    'selectors'  => [
                        '{{WRAPPER}} .kfw-horiz-scroll-ele-title-icon-wrap.kfw-horiz-scroll-ele-title-image-type img' => 'max-width:{{SIZE}}px;'
                    ],
                ]);

            /**
             * Style: Lordicon Type
             */
                $this->add_control('lordicon_type_style_before_hr', [
                    'type' => KFCM::DIVIDER,
                ]);
                $this->add_control('lordicon_type_style_heading',[
                    'label' => esc_html__( 'LordIcon Type Style', 'onnat-extra' ),
                    'type'  => KFCM::HEADING,
                ]);
                $this->add_control('lordicon_type_style_after_hr', [
                    'type' => KFCM::DIVIDER,
                ]);
                $this->add_responsive_control( 'lordicon_type_size', [
                    'label'      => esc_html__('Size', 'onnat-extra' ),
                    'type'       => KFCM::SLIDER,
                    'size_units' => [ 'px' ],
                    'range'      => [
                        'px' => [
                            'min' => 1,
                            'max' => 300,
                        ],
                    ],
                    'selectors'  => [
                        '{{WRAPPER}} .kfw-horiz-scroll-ele-title-icon-wrap.kfw-horiz-scroll-ele-title-lordicon-type .kfw-elementor-icon-wrap,
                        {{WRAPPER}} .kfw-horiz-scroll-ele-title-icon-wrap.kfw-horiz-scroll-ele-title-lordicon-type .kfw-elementor-lordicon' => 'width:{{SIZE}}{{UNIT}};height:{{SIZE}}{{UNIT}};'
                    ]
                ]);
				$this->add_control( 'lordicon_color_settings', [
                    'label'        => esc_html__('Colors', 'onnat-extra'),
					'type'         => KFCM::POPOVER_TOGGLE,
					'return_value' => 'yes',
				]);
                $this->start_popover();
                    $this->add_control('lordicon_primary_color',[
                        'label'     => __( 'Primary Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'default'   => '#121331',
                    ]);
                    $this->add_control('lordicon_secondary_color',[
                        'label'   => __( 'Secondary Color', 'onnat-extra' ),
                        'type'    => KFCM::COLOR,
                        'default' => '#121331',
                    ]);
                    $this->add_control('lordicon_tertiary_color',[
                        'label'   => __( 'Tertiary Color', 'onnat-extra' ),
                        'type'    => KFCM::COLOR,
                        'default' => '#0816A8',
                    ]);
                $this->end_popover();
		$this->end_controls_section();
    }

    public function _register_content_style_section() {
		$this->start_controls_section( 'section_content_style', [
			'label' => esc_html__( 'Content', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_group_control(KFGCT::get_type(), [
                'name'     => 'content_typo',
                'label'    => esc_html__( 'Typography', 'onnat-extra' ),
                'selector' => '{{WRAPPER}} .kfw-horiz-scroll-ele-content-wrap',
            ]);
            $this->add_responsive_control( 'content_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-horiz-scroll-ele-content-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'content_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-horiz-scroll-ele-content-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_control( 'content_color', [
                'label'     => esc_html__('Color', 'onnat-extra' ),
                'type'      => KFCM::COLOR,
                'selectors' => ['{{WRAPPER}} .kfw-horiz-scroll-ele-content-wrap' => 'color: {{VALUE}};' ],
            ]);
            $this->add_group_control( KFGCTS::get_type(), [
                'name'     => 'content_txt_shadow',
                'selector' => '{{WRAPPER}} .kfw-horiz-scroll-ele-content-wrap',
            ]);
            $this->add_group_control( KFGCTST::get_type(), [
                'name'     => 'content_txt_stroke',
                'selector' => '{{WRAPPER}} .kfw-horiz-scroll-ele-content-wrap',
            ]);
		$this->end_controls_section();
    }

    public function _register_line_style_section() {
		$this->start_controls_section( 'section_line_style', [
            'label'     => esc_html__( 'Line', 'onnat-extra' ),
            'tab'       => KFCM::TAB_STYLE,
            'condition' => [
                'use_line!' => ''
            ],
		]);
            $this->add_responsive_control( 'line_size', [
                'label'      => esc_html__('Size', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'size_units' => ['px'],
                'condition'  => [
                    'use_line!' => ''
                ],
                'range'      => [
                    'px' => [
                        'min' => 5,
                        'max' => 200,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}}.kfw-horiz-scroll-ele-line-add .kfw-elementor-split-horiz-scroll-ele-item:before' => 'height:{{SIZE}}px;',
                ],
            ]);
            $this->add_responsive_control( 'line_position', [
                'label'              => esc_html__('Position', 'onnat-extra' ),
                'type'               => KFCM::DIMENSIONS,
                'size_units'         => [ 'px' ],
                'allowed_dimensions' => [ 'top' ],
                'placeholder'        => [
                    'top'    => '',
                    'right'  => 'auto',
                    'bottom' => 'auto',
                    'left'   => 'auto',
                ],
                'default'            => [
                    'top'      => '',
                    'right'    => 'auto',
                    'bottom'   => 'auto',
                    'left'     => 'auto',
                    'unit'     => 'px',
                    'isLinked' => false,
                ],
                'condition'          => [
                    'use_line!' => '',
                ],
                'selectors'          => [
                    '{{WRAPPER}}.kfw-horiz-scroll-ele-line-add .kfw-elementor-split-horiz-scroll-ele-item:before' => 'top:{{TOP}}{{UNIT}};'
                ],
            ]);
            $this->add_group_control( KFGCBG::get_type(), [
                'name'           => 'line_bg',
                'types'          => ['classic', 'gradient'],
                'exclude'        => [ 'image'],
                'condition'      => [
                    'use_line!' => ''
                ],
                'fields_options' => [
                    'background' => [ 'default' => 'classic' ],
                    'color'      => [
                        'default'   => '',
                        'selectors' => [
                            '{{SELECTOR}}' => 'background:{{VALUE}};',
                        ],
                    ],
                ],
                'selector'       => '{{WRAPPER}}.kfw-horiz-scroll-ele-line-add .kfw-elementor-split-horiz-scroll-ele-item:before',
            ]);
		$this->end_controls_section();
    }

	public function _register_skin_style_section() {
		$this->start_controls_section( 'section_skin_style', [
            'label' => esc_html__( 'Alternate Color', 'onnat-extra' ),
            'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_control( 'alt_bg_color', [
				'label'     => esc_html__('Background Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}}' => '--kfw-horiz-scroll-ele-alt-color:{{VALUE}};',
				],
			]);
			$this->add_control( 'alt_icon_color', [
				'label'     => esc_html__('Icon', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}}' => '--kfw-horiz-scroll-ele-alt-icon-color:{{VALUE}};',
				],
			]);
		$this->end_controls_section();
	}

	public function _render_text( $txt ) {
		if( !empty( $txt ) ) {

			printf('<div class="kfw-horiz-scroll-ele-title-icon-wrap kfw-horiz-scroll-ele-title-txt-type">
					<span class="kfw-elementor-icon-wrap">%s</span>
				</div>', $txt
			);

		}
	}

	public function _render_icon( $icon , $echo = true ) {
		$icon = array_filter( $icon );

		if( !empty( $icon ) ) {
            echo '<div class="kfw-horiz-scroll-ele-title-icon-wrap kfw-horiz-scroll-ele-title-icon-type">';
                echo '<span class="kfw-elementor-icon-wrap">';
                    KFIM::render_icon( $icon, [ 'aria-hidden' => 'true' ] );
                echo '</span>';
            echo '</div>';
		}
    }

    public function _render_img( $settings = [], $wrap = true ) {
        if( !empty( $settings['url'] ) ) {
            $image_src = $settings['url'];

			$image_title = $this->_get_image_title( $settings );
			$image_title = empty( $image_title ) ? $this->get_title() : $image_title;

			$image_alt = $this->_get_image_alt( $settings );
			$image_alt = empty( $image_alt ) ? $image_title : $image_title;

			if( $wrap ) {
				echo '<div class="kfw-horiz-scroll-ele-title-icon-wrap kfw-horiz-scroll-ele-title-image-type">';
			}
                echo '<span class="kfw-elementor-icon-wrap">';
                    echo '<img alt="'. esc_attr( $image_alt ) .'" title="'. esc_attr( $image_title ) .'" src="'. esc_url( $image_src ) .'"/>';
                echo '</span>';
			if( $wrap ) {
				echo '</div>';
			}
        }
    }

    public static function _get_image_src( $attachment_id, $image_size_key, array $settings ) {
        return KFGCIS::get_attachment_image_src( $attachment_id, $image_size_key, $settings );
    }

	public function _get_image_title( $attachment ) {
		if ( empty( $attachment['id'] ) ) {
			return '';
		}

		return get_the_title( $attachment['id'] );
	}

	public function _get_image_alt( $instance ) {
		if ( empty( $instance['id'] ) ) {
			// For `Insert From URL` images.
			return isset( $instance['alt'] ) ? trim( strip_tags( $instance['alt'] ) ) : '';
		}

		$attachment_id = $instance['id'];
		if ( ! $attachment_id ) {
			return '';
		}

		$attachment = get_post( $attachment_id );
		if ( ! $attachment ) {
			return '';
		}

		$alt = get_post_meta( $attachment_id, '_wp_attachment_image_alt', true );
		if ( ! $alt ) {
			$alt = $attachment->post_excerpt;
			if ( ! $alt ) {
				$alt = $attachment->post_title;
			}
		}
		return trim( strip_tags( $alt ) );
	}

	/**
	 * Get template content
	 */
	public function _generate_template_content( $template_id ) {
		return KFEl::instance()->frontend->get_builder_content_for_display( $template_id );
	}

    /**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-horizontal-scroll-element");
		include $render;
    }
}