<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use \Elementor\Widget_Base as KFWB;
use \Elementor\Controls_Manager as KFCM;
use \Elementor\Group_Control_Image_Size as KFGCIS;
use \Elementor\Group_Control_Typography as KFGCT;
use \Elementor\Group_Control_Border as KFGCB;
use \Elementor\Group_Control_Box_Shadow as KFGCBS;
use \Elementor\Group_Control_Background as KFGCBG;

class KFW_Image_Comparison extends KFWB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-image-comparison';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Image Comparison', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_core_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-image-before-after';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'images', 'compare', 'before', 'after' ];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
        return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/image-comparison';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

		wp_register_script( 'jquery-event-move',
			KFE_CONST_URL . 'libraries/elementor/assets/js/jquery.event.move' . KFE_CONST_DEBUG_SUFFIX . '.js',
			[ 'jquery' ],
			KFE_CONST_VERSION,
			true
		);

		wp_register_script( 'kfw-twentytwenty',
			KFE_CONST_URL . 'libraries/elementor/assets/js/jquery.kfw.twentytwenty' . KFE_CONST_DEBUG_SUFFIX . '.js',
			[ 'jquery' ],
			KFE_CONST_VERSION,
			true
		);

		wp_register_script( 'kfw-image-comparsion',
			KFE_CONST_URL . 'libraries/elementor/assets/js/image-comparison' . KFE_CONST_DEBUG_SUFFIX . '.js',
			[ 'jquery', 'elementor-frontend' ],
			KFE_CONST_VERSION,
			true
		);

		wp_register_style( 'kfw-image-comparsion',
			KFE_CONST_URL . 'libraries/elementor/assets/css/image-comparison' . KFE_CONST_DEBUG_SUFFIX . '.css',
			[],
			KFE_CONST_VERSION
		);
    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'kfw-image-comparsion' ];
	}

	/**
	 * Get script dependencies.
	 *
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @return array Element scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'jquery-event-move', 'kfw-twentytwenty', 'imagesloaded', 'kfw-image-comparsion' ];
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
        $this->_register_before_image_section();
        $this->_register_after_image_section();
        $this->_register_general_settings_section();
        $this->_register_labels_style_section();
        $this->_register_drag_handle_style_section();
        $this->_register_drag_handle_separator_style_section();
	}

	public function _register_before_image_section() {
		$this->start_controls_section( 'section_before_image',[
			'label' => esc_html__( 'Before Image', 'onnat-extra' )
        ]);
			$this->add_control( 'before_image', [
				'label'   => esc_html__( 'Image', 'onnat-extra' ),
				'type'    => KFCM::MEDIA,
				'default' => [ 'url' => KFE_CONST_URL .'libraries/elementor/assets/images/image-comparision-widget-before.svg' ],
				'dynamic' => [ 'active' => true ],
            ]);
			$this->add_control( 'before_image_alt', [
				'label'       => esc_html__( 'Alternative Text', 'onnat-extra' ),
				'label_block' => true,
				'type'        => KFCM::TEXT,
				'default'     => esc_html__( 'Before Image', 'onnat-extra' )
            ]);
			$this->add_control( 'show_before_label', [
				'label'        => esc_html__( 'Label', 'onnat-extra' ),
				'type'         => KFCM::SWITCHER,
				'label_on'     => esc_html__( 'Show', 'onnat-extra' ),
				'label_off'    => esc_html__( 'Hide', 'onnat-extra' ),
				'return_value' => 'yes',
				'default'      => 'yes',
            ]);
			$this->add_control( 'before_label', [
				'label'       => esc_html__( 'Label Text', 'onnat-extra' ),
				'label_block' => true,
				'type'        => KFCM::TEXT,
				'condition'   => [ 'show_before_label' => 'yes' ],
				'default'     => esc_html__( 'Before', 'onnat-extra' ),
            ]);
        $this->end_controls_section();
	}

	public function _register_after_image_section() {
		$this->start_controls_section( 'section_after_image',[
			'label' => esc_html__( 'After Image', 'onnat-extra' )
        ]);
			$this->add_control( 'after_image', [
				'label'   => esc_html__( 'Image', 'onnat-extra' ),
                'type'    => KFCM::MEDIA,
				'default' => [ 'url' => KFE_CONST_URL .'libraries/elementor/assets/images/image-comparision-widget-after.svg' ],
				'dynamic' => [ 'active' => true ],
            ]);
			$this->add_control( 'after_image_alt', [
				'label'       => esc_html__( 'Alternative Text', 'onnat-extra' ),
				'label_block' => true,
				'type'        => KFCM::TEXT,
				'default'     => esc_html__( 'After Image', 'onnat-extra' )
            ]);
			$this->add_control( 'show_after_label', [
				'label'        => esc_html__( 'Label', 'onnat-extra' ),
				'type'         => KFCM::SWITCHER,
				'label_on'     => esc_html__( 'Show', 'onnat-extra' ),
				'label_off'    => esc_html__( 'Hide', 'onnat-extra' ),
				'return_value' => 'yes',
				'default'      => 'yes',
            ]);
			$this->add_control( 'after_label', [
				'label'       => esc_html__( 'Label Text', 'onnat-extra' ),
				'label_block' => true,
				'type'        => KFCM::TEXT,
				'condition'   => [ 'show_after_label' => 'yes' ],
				'default'     => esc_html__( 'After', 'onnat-extra' )
            ]);
        $this->end_controls_section();
	}

	public function _register_general_settings_section() {
		$this->start_controls_section( 'section_settings',[
			'label' => esc_html__( 'Settings', 'onnat-extra' )
        ]);
			$this->add_group_control( KFGCIS::get_type(), [
				'name'    => 'image_size',
				'default' => 'full',
            ]);
			$this->add_control( 'orientation', [
				'label'   => esc_html__( 'Orientation', 'onnat-extra' ),
				'type'    => KFCM::CHOOSE,
				'options' => [
					'horizontal' => [
						'title' => esc_html__( 'Horizontal', 'onnat-extra' ),
						'icon'  => 'eicon-column'
					],
					'vertical'   => [
						'title' => esc_html__( 'Vertical', 'onnat-extra' ),
						'icon'  => 'eicon-section'
					],
				],
				'toggle'  => false,
				'default' => 'horizontal',
            ]);
			$this->add_control( 'offset', [
				'label'   => esc_html__( 'Visiblity', 'onnat-extra' ),
				'type'    => KFCM::SLIDER,
				'range'   => [
					'px' => [
						'min'  => 0.1,
						'max'  => 1,
						'step' => 0.1,
					]
				],
				'default' => [
					'size' => 0.5
				]
            ]);
			$this->add_control( 'move_slider', [
				'label'   => esc_html__( 'Move Slider', 'onnat-extra' ),
				'type'    => KFCM::SELECT,
				'options' => [
					'drag'        => esc_html__( 'Drag', 'onnat-extra' ),
					'mouse_move'  => esc_html__( 'Mouse Move', 'onnat-extra' ),
					'mouse_click' => esc_html__( 'Mouse Click', 'onnat-extra' ),
				],
				'default' => 'drag',
            ]);
			$this->add_control( 'show_drag_handle_separator', [
				'label'        => esc_html__( 'Drag Handle Separator', 'onnat-extra' ),
				'type'         => KFCM::SWITCHER,
				'label_on'     => esc_html__( 'Show', 'onnat-extra' ),
				'label_off'    => esc_html__( 'Hide', 'onnat-extra' ),
				'return_value' => 'yes',
				'default'      => 'yes',
            ]);
        $this->end_controls_section();
	}

	public function _register_labels_style_section() {
		$this->start_controls_section( 'section_label_style_settings',[
			'label'      => esc_html__( 'Labels', 'onnat-extra' ),
			'tab'        => KFCM::TAB_STYLE,
			'conditions' => [
				'relation' => 'or',
				'terms'    => [
					[ 'name' => 'show_before_label', 'value' => 'yes' ],
					[ 'name' => 'show_after_label', 'value' => 'yes' ],
				],
			]
        ]);
            $this->start_controls_tabs( 'tabs_labels_style' );
				$this->start_controls_tab( 'tab_label_before_style', [
					'label'     => esc_html__('Before', 'onnat-extra' ),
					'condition' => [ 'show_before_label' => 'yes' ]
                ]);
					$this->add_control( 'label_before_background_color',[
						'label'     => esc_html__('Background Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'condition' => [ 'show_before_label' => 'yes' ],
						'selectors' => [ '{{WRAPPER}} .kfw-ele-image-compare-before-label span' => 'background-color: {{VALUE}};' ]
                    ]);
					$this->add_control( 'label_before_color',[
						'label'     => esc_html__('Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'condition' => [ 'show_before_label' => 'yes' ],
						'selectors' => [ '{{WRAPPER}} .kfw-ele-image-compare-before-label span' => 'color: {{VALUE}};' ]
                    ]);
					$this->add_group_control( KFGCT::get_type(), [
						'name'      => 'label_before_typography',
						'condition' => [ 'show_before_label' => 'yes' ],
						'selector'  => '{{WRAPPER}} .kfw-ele-image-compare-before-label span'
                    ]);
					$this->add_group_control( KFGCB::get_type(), [
						'name'      => 'label_before_border',
						'condition' => [ 'show_before_label' => 'yes' ],
						'selector'  => '{{WRAPPER}} .kfw-ele-image-compare-before-label span'
                    ]);
					$this->add_control( 'label_before_border_radius', [
						'label'      => esc_html__( 'Border Radius', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'condition'  => [ 'show_before_label' => 'yes' ],
						'size_units' => [ 'px', '%' ],
						'selectors'  => [ '{{WRAPPER}} .kfw-ele-image-compare-before-label span' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ]
                    ]);
					$this->add_group_control( KFGCBS::get_type(), [
						'name'      => 'label_before_box_shadow',
						'condition' => [ 'show_before_label' => 'yes' ],
						'selector'  => '{{WRAPPER}} .kfw-ele-image-compare-before-label span'
                    ]);
					$this->add_control( 'label_before_horizontal_alignment', [
						'label'     => esc_html__( 'Horizontal Alignment', 'onnat-extra' ),
						'type'      => KFCM::CHOOSE,
						'separator' => 'before',
						'condition' => [ 'show_before_label' => 'yes' ],
						'options'   => [
							'left'   => [
								'title' => esc_html__( 'Left', 'onnat-extra' ),
								'icon'  => 'eicon-h-align-left'
							],
							'center' => [
								'title' => esc_html__( 'Center', 'onnat-extra' ),
								'icon'  => 'eicon-h-align-center'
							],
							'right'  => [
								'title' => esc_html__( 'Right', 'onnat-extra' ),
								'icon'  => 'eicon-h-align-right'
							],
						],
						'default'   => 'left',
                    ]);
					$this->add_control( 'label_before_vertical_alignment', [
						'label'     => esc_html__( 'Vertical Alignment', 'onnat-extra' ),
						'type'      => KFCM::CHOOSE,
						'separator' => 'after',
						'condition' => [ 'show_before_label' => 'yes' ],
						'options'   => [
							'top'    => [
								'title' => esc_html__( 'Top', 'onnat-extra' ),
								'icon'  => 'eicon-v-align-top'
							],
							'middle' => [
								'title' => esc_html__( 'Middle', 'onnat-extra' ),
								'icon' => 'eicon-v-align-middle'
							],
							'bottom' => [
								'title' => esc_html__( 'Bottom', 'onnat-extra' ),
								'icon'  => 'eicon-v-align-bottom'
							],
						],
						'default'   => 'middle',
                    ]);
					$this->add_responsive_control( 'label_before_padding',[
						'label'      => esc_html__('Padding', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'condition'  => [ 'show_before_label' => 'yes' ],
						'size_units' => [ 'px', 'em', '%' ],
						'selectors'  => [
							'{{WRAPPER}} .kfw-ele-image-compare-before-label span' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ]
                    ]);
                $this->end_controls_tab();

				$this->start_controls_tab( 'tab_label_after_style', [
					'label'     => esc_html__('After', 'onnat-extra'),
					'condition' => [ 'show_after_label' => 'yes' ]
                ]);
					$this->add_control( 'label_after_background_color',[
						'label'     => esc_html__('Background Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'condition' => [ 'show_after_label' => 'yes' ],
						'selectors' => [ '{{WRAPPER}} .kfw-ele-image-compare-after-label span' => 'background-color: {{VALUE}};' ]
                    ]);
					$this->add_control( 'label_after_color',[
						'label'     => esc_html__('Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'condition' => [ 'show_after_label' => 'yes' ],
						'selectors' => [ '{{WRAPPER}} .kfw-ele-image-compare-after-label span' => 'color: {{VALUE}};' ]
                    ]);
					$this->add_group_control( KFGCT::get_type(), [
						'name'      => 'label_after_typography',
						'condition' => [ 'show_after_label' => 'yes' ],
						'selector'  => '{{WRAPPER}} .kfw-ele-image-compare-after-label span'
                    ]);
					$this->add_group_control( KFGCB::get_type(), [
						'name'      => 'label_after_border',
						'condition' => [ 'show_after_label' => 'yes' ],
						'selector'  => '{{WRAPPER}} .kfw-ele-image-compare-after-label span'
                    ]);
					$this->add_control( 'label_after_border_radius', [
						'label'      => esc_html__( 'Border Radius', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'condition'  => [ 'show_after_label' => 'yes' ],
						'size_units' => [ 'px', '%' ],
						'selectors'  => [ '{{WRAPPER}} .kfw-ele-image-compare-after-label span' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ]
                    ]);
					$this->add_group_control( KFGCBS::get_type(), [
						'name'      => 'label_after_box_shadow',
						'condition' => [ 'show_after_label' => 'yes' ],
						'selector'  => '{{WRAPPER}} .kfw-ele-image-compare-after-label span'
                    ]);
					$this->add_control( 'label_after_horizontal_alignment', [
						'label'     => esc_html__( 'Horizontal Alignment', 'onnat-extra' ),
						'type'      => KFCM::CHOOSE,
						'separator' => 'before',
						'condition' => [ 'show_after_label' => 'yes' ],
						'options'   => [
							'left'   => [
								'title' => esc_html__( 'Left', 'onnat-extra' ),
								'icon'  => 'eicon-h-align-left'
							],
							'center' => [
								'title' => esc_html__( 'Center', 'onnat-extra' ),
								'icon'  => 'eicon-h-align-center'
							],
							'right'  => [
								'title' => esc_html__( 'Right', 'onnat-extra' ),
								'icon'  => 'eicon-h-align-right'
							],
						],
						'default'   => 'right',
                    ]);
					$this->add_control( 'label_after_vertical_alignment', [
						'label'     => esc_html__( 'Vertical Alignment', 'onnat-extra' ),
						'type'      => KFCM::CHOOSE,
						'separator' => 'after',
						'condition' => [ 'show_after_label' => 'yes' ],
						'options'   => [
							'top'    => [
								'title' => esc_html__( 'Top', 'onnat-extra' ),
								'icon'  => 'eicon-v-align-top'
							],
							'middle' => [
								'title' => esc_html__( 'Middle', 'onnat-extra' ),
								'icon'  => 'eicon-v-align-middle'
							],
							'bottom' => [
								'title' => esc_html__( 'Bottom', 'onnat-extra' ),
								'icon'  => 'eicon-v-align-bottom'
							],
						],
						'default'   => 'middle',
                    ]);
					$this->add_responsive_control( 'label_after_padding',[
						'label'      => esc_html__('Padding', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'condition'  => [ 'show_after_label' => 'yes' ],
						'size_units' => [ 'px', 'em', '%' ],
						'selectors'  => [
							'{{WRAPPER}} .kfw-ele-image-compare-after-label span' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ]
                    ]);
                $this->end_controls_tab();
            $this->end_controls_tabs();
        $this->end_controls_section();
	}

	public function _register_drag_handle_style_section() {
		$this->start_controls_section( 'section_handle_style_settings',[
			'label' => esc_html__( 'Drag Handle', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
        ]);
			$this->add_responsive_control( 'drag_handle_width', [
				'label'      => esc_html__('Width', 'onnat-extra'),
				'type'       => KFCM::SLIDER,
				'size_units' => [ 'px' ],
				'range'      => [
					'px' => [
						'min' => 20,
						'max' => 100
					]
				],
				'selectors'  => [ '{{WRAPPER}} .kfw-ele-image-compare-handle' => 'width:{{SIZE}}{{UNIT}};' ]
            ]);
			$this->add_responsive_control( 'drag_handle_height', [
				'label'      => esc_html__('Height', 'onnat-extra'),
				'type'       => KFCM::SLIDER,
				'size_units' => [ 'px' ],
				'range'      => [
					'px' => [
						'min' => 20,
						'max' => 100
					]
				],
				'selectors'  => [ '{{WRAPPER}} .kfw-ele-image-compare-handle' => 'height:{{SIZE}}{{UNIT}};' ]
			]);
            $this->start_controls_tabs( 'tabs_drag_style' );
				$this->start_controls_tab( 'tab_drag_normal_style', [
					'label' => esc_html__('Normal', 'onnat-extra')
                ]);
					$this->add_group_control( KFGCBG::get_type(), [
						'name'     => 'drag_handle_background_color',
						'selector' => '{{WRAPPER}} .kfw-ele-image-compare-handle'
                    ]);
					$this->add_group_control( KFGCBS::get_type(), [
						'name'      => 'drag_handle_box_shadow',
						'selector'  => '{{WRAPPER}} .kfw-ele-image-compare-handle'
                    ]);
					$this->add_group_control( KFGCB::get_type(), [
						'name'     => 'drag_handle_border',
						'selector' => '{{WRAPPER}} .kfw-ele-image-compare-handle'
                    ]);
				$this->end_controls_tab();
				$this->start_controls_tab( 'tab_drag_hover_style', [
					'label' => esc_html__('Hover', 'onnat-extra')
                ]);
					$this->add_group_control( KFGCBG::get_type(), [
						'name'     => 'drag_handle_background_color_hover',
						'selector' => '{{WRAPPER}} .kfw-ele-image-compare-handle:hover'
                    ]);
					$this->add_group_control( KFGCBS::get_type(), [
						'name'      => 'drag_handle__box_shadow_hover',
						'selector'  => '{{WRAPPER}} .kfw-ele-image-compare-handle:hover'
                    ]);
					$this->add_group_control( KFGCB::get_type(), [
						'name'     => 'drag_handle_border_hover',
						'selector' => '{{WRAPPER}} .kfw-ele-image-compare-handle:hover'
                    ]);
                $this->end_controls_tab();
            $this->end_controls_tabs();
			$this->add_responsive_control( 'drag_handle_border_radius', [
				'label'      => esc_html__('Border Radius', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-image-compare-handle' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
            ]);
			$this->add_responsive_control( 'drag_handle_padding', [
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-image-compare-handle' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
            ]);
			$this->add_control( 'drag_handle_arrow_hr', [
				'type'  => KFCM::DIVIDER,
				'style' => 'thick'
            ]);
			$this->add_control( 'drag_handle_arrow_heading', [
				'label' => esc_html__('Arrow', 'onnat-extra'),
				'type'  => KFCM::HEADING,
            ]);
			$this->add_control( 'drag_handle_arrow', [
				'label'   => esc_html__('Arrow', 'onnat-extra'),
				'type'    => KFCM::SELECT,
				'options' => [
					'angle'          => esc_html__( 'Angle', 'onnat-extra' ),
					'angle-double'   => esc_html__( 'Angle Double', 'onnat-extra' ),
					'arrow'          => esc_html__( 'Arrow', 'onnat-extra' ),
					'arrow-circle'   => esc_html__( 'Arrow Circle', 'onnat-extra' ),
					'caret'          => esc_html__( 'Caret', 'onnat-extra' ),
					'caret-square-o' => esc_html__( 'Caret Square', 'onnat-extra' ),
					'chevron'        => esc_html__( 'Chevron', 'onnat-extra' ),
					'chevron-circle' => esc_html__( 'Chevron Circle', 'onnat-extra' ),
					'long-arrow'     => esc_html__( 'Long Arrow', 'onnat-extra' ),
				],
				'default' => 'caret',
            ]);
			$this->add_responsive_control( 'drag_handle_arrow_size', [
				'label'      => esc_html__('Size', 'onnat-extra'),
				'type'       => KFCM::SLIDER,
				'size_units' => [ 'px' ],
				'range'      => [
					'px' => [
						'min' => 1,
						'max' => 40
					]
				],
				'selectors'  => [ '{{WRAPPER}} .kfw-ele-image-compare-arrow'  => 'font-size:{{SIZE}}{{UNIT}};' ]
            ]);
			$this->add_responsive_control( 'drag_handle_arrow_margin', [
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-image-compare-handle span' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
            ]);
            $this->start_controls_tabs( 'tabs_drag_handle_arrow_style' );
				$this->start_controls_tab( 'tab_drag_handle_arrow_normal_style', [
					'label' => esc_html__('Normal', 'onnat-extra')
                ]);
					$this->add_control( 'drag_handle_arrow_color',[
						'label'     => esc_html__('Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => [ '{{WRAPPER}} .kfw-ele-image-compare-arrow' => 'color: {{VALUE}};' ]
                    ]);
                $this->end_controls_tab();
				$this->start_controls_tab( 'tab_drag_handle_arrow_hover_style', [
					'label' => esc_html__('Hover', 'onnat-extra')
                ]);
					$this->add_control( 'drag_handle_arrow_color_hover',[
						'label'     => esc_html__('Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => [ '{{WRAPPER}} .kfw-ele-image-compare-handle:hover .kfw-ele-image-compare-arrow' => 'color: {{VALUE}};' ]
                    ]);
                $this->end_controls_tab();
			$this->end_controls_tabs();
		$this->end_controls_section();
	}

	public function _register_drag_handle_separator_style_section() {
		$this->start_controls_section( 'section_handle_separator_style_settings',[
			'label'     => esc_html__( 'Handle Separator', 'onnat-extra' ),
			'tab'       => KFCM::TAB_STYLE,
			'condition' => [ 'show_drag_handle_separator' => 'yes' ],
        ]);
			$this->add_control( 'handle_separator_color',[
				'label'     => esc_html__('Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [ '{{WRAPPER}} .kfw-ele-image-compare-handle:after, {{WRAPPER}} .kfw-ele-image-compare-handle:before' => 'background: {{VALUE}};' ]
            ]);
			$this->add_responsive_control( 'handle_separator_spacing', [
				'label'          => esc_html__('Spacing', 'onnat-extra' ),
				'type'           => KFCM::SLIDER,
				'range'          => [
					'px' => [
						'min' => 0,
						'max' => 200
					]
				],
				'selectors'      => [
					'{{WRAPPER}} .kfw-ele-image-compare-horizontal .kfw-ele-image-compare-handle:after'  => 'top:{{SIZE}}%',
					'{{WRAPPER}} .kfw-ele-image-compare-horizontal .kfw-ele-image-compare-handle:before' => 'bottom:{{SIZE}}%',
					'{{WRAPPER}} .kfw-ele-image-compare-vertical .kfw-ele-image-compare-handle:after'    => 'right:{{SIZE}}%',
					'{{WRAPPER}} .kfw-ele-image-compare-vertical .kfw-ele-image-compare-handle:before'   => 'left:{{SIZE}}%',
				]
            ]);
			$this->add_responsive_control( 'handle_separator_width', [
				'label'          => esc_html__('Width', 'onnat-extra' ),
				'type'           => KFCM::SLIDER,
				'size_units'     => [ 'px', '%' ],
				'range'          => [
					'px' => [
						'max' => 20
					]
				],
				'tablet_default' => [
					'unit' => 'px'
				],
				'mobile_default' => [
					'unit' => 'px'
				],
				'default'        => [
					'size' => 2,
					'unit' => 'px'
				],
				'selectors'      => [ '{{WRAPPER}} .kfw-ele-image-compare-handle:after, {{WRAPPER}} .kfw-ele-image-compare-handle:before' => 'width: {{SIZE}}{{UNIT}};' ],
				'condition' 	 => [ 'orientation' => 'horizontal' ]
            ]);
			$this->add_responsive_control( 'handle_separator_height', [
				'label'          => esc_html__('Height', 'onnat-extra' ),
				'type'           => KFCM::SLIDER,
				'size_units'     => [ 'px', '%' ],
				'range'          => [
					'px' => [
						'max' => 20
					]
				],
				'tablet_default' => [
					'unit' => 'px'
				],
				'mobile_default' => [
					'unit' => 'px'
				],
				'default'        => [
					'size' => 2,
					'unit' => 'px'
				],
				'selectors'      => [ '{{WRAPPER}} .kfw-ele-image-compare-handle:after, {{WRAPPER}} .kfw-ele-image-compare-handle:before' => 'height: {{SIZE}}{{UNIT}};' ],
				'condition' 	 => [ 'orientation' => 'vertical' ]
            ]);
			$this->add_group_control( KFGCBS::get_type(), [
				'name'      => 'handle_separator_box_shadow',
				'selector'  => '{{WRAPPER}} .kfw-ele-image-compare-handle:after, {{WRAPPER}} .kfw-ele-image-compare-handle:before'
			]);
        $this->end_controls_section();
    }

	/**
	 * Generate setting json
	 */
	protected function _generate_setting_json( array $settings ) {
		extract( $settings );

		return wp_json_encode([
			'show_before_label'          => ( $show_before_label == "yes" ) ? true : false,
			'before_label'               => $before_label,
			'before_label_position'      => [
				'kfw-ele-image-compare-before-label-horizontal-alignment-'.$label_before_horizontal_alignment,
				'kfw-ele-image-compare-before-label-vertical-alignment-'.$label_before_vertical_alignment
			],
			'show_after_label'           => ( $show_after_label == "yes" ) ? true : false,
			'after_label'                => $after_label,
			'after_label_position'       => [
				'kfw-ele-image-compare-after-label-horizontal-alignment-'.$label_after_horizontal_alignment,
				'kfw-ele-image-compare-after-label-vertical-alignment-'.$label_after_vertical_alignment
			],
			'orientation'                => $orientation,
			'offset'                     => $offset['size'],
			'slider_on_hover'            => ( $move_slider == 'mouse_move' ) ? true : false,
			'slider_with_handle'         => ( $move_slider == 'drag' ) ? true : false,
			'slider_with_click'          => ( $move_slider == 'mouse_click' ) ? true : false,
			'show_drag_handle_separator' => ( $show_drag_handle_separator == "yes" ) ? true : false,
			'arrow' 					 => $drag_handle_arrow
		]);
	}

	/**
	 * Get attachment image src.
	 */
	public static function _get_image_src( $attachment_id, $image_size_key, array $settings ) {
        return KFGCIS::get_attachment_image_src( $attachment_id, $image_size_key, $settings );
	}

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-image-comparison");
		include $render;
    }
}