<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use \Elementor\Widget_Base as KFWB;
use \Elementor\Controls_Manager as KFCM;
use \Elementor\Icons_Manager as KFIM;
use \Elementor\Group_Control_Image_Size as KFGCIS;
use \Elementor\Group_Control_Typography as KFGCT;
use \Elementor\Group_Control_Background as KFGCBG;
use \Elementor\Group_Control_Box_Shadow as KFGCBS;
use \Elementor\Group_Control_Text_Shadow as KFGCTS;
use \Elementor\Group_Control_Text_Stroke as KFGCTST;
use \Elementor\Repeater as KFRTR;
use \Elementor\Utils as KFUtils;
use \Elementor\Plugin as KFEl;

class KFW_Image_Hotspots extends KFWB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-image-hotspots';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Image Hotspots', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_core_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-image-hotspot';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'images', 'hotspots', 'hotspot', 'image' ];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
        return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/image-hotspots';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

		wp_register_script( 'tooltipster.bundle',
			KFE_CONST_URL . 'libraries/elementor/assets/js/tooltipster.bundle' . KFE_CONST_DEBUG_SUFFIX . '.js',
			[ 'jquery' ],
			KFE_CONST_VERSION,
			true
		);

		wp_register_script( 'kfw-image-hotspots',
			KFE_CONST_URL . 'libraries/elementor/assets/js/image-hotspots' . KFE_CONST_DEBUG_SUFFIX . '.js',
			[ 'jquery', 'elementor-frontend' ],
			KFE_CONST_VERSION,
			true
		);

		wp_register_style( 'tooltipster.bundle',
			KFE_CONST_URL . 'libraries/elementor/assets/css/tooltipster.bundle' . KFE_CONST_DEBUG_SUFFIX . '.css',
			[],
			KFE_CONST_VERSION
		);

		wp_register_style( 'kfw-image-hotspots',
			KFE_CONST_URL . 'libraries/elementor/assets/css/image-hotspots' . KFE_CONST_DEBUG_SUFFIX . '.css',
			[],
			KFE_CONST_VERSION
		);
    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'tooltipster.bundle', 'kfw-image-hotspots' ];
	}

	/**
	 * Get script dependencies.
	 *
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @return array Element scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'tooltipster.bundle', 'kfw-image-hotspots' ];
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
		$this->_register_image_section();
		$this->_register_hotspots_section();
		$this->_register_hotspots_shape_style_section();
		$this->_register_hotspots_dot_type_style_section();
		$this->_register_hotspots_icon_type_style_section();
		$this->_register_hotspots_text_type_style_section();
		$this->_register_tooltip_style_section();
	}

	public function _register_image_section() {
		$this->start_controls_section( 'section_image', [
			'label' => esc_html__( 'Image', 'onnat-extra' )
		]);
			$this->add_control( 'image', [
				'label'   => esc_html__( 'Image', 'onnat-extra' ),
				'type'    => KFCM::MEDIA,
				'default' => [ 'url' => KFE_CONST_URL .'libraries/elementor/assets/images/image-hotspots-widget.svg' ],
				'dynamic' => [ 'active' => true ],
			]);

			$this->add_group_control( KFGCIS::get_type(), [
				'name'    => 'image_size',
				'default' => 'full',
            ]);

			$this->add_responsive_control( 'align', [
				'label'     => esc_html__( 'Alignment', 'onnat-extra' ),
				'type'      => KFCM::CHOOSE,
				'options'   => [
					'left'   => [
						'title' => esc_html__('Left', 'onnat-extra'),
						'icon'  => 'eicon-text-align-left',
					],
					'center' => [
						'title' => esc_html__('Center', 'onnat-extra'),
						'icon'  => 'eicon-text-align-center',
					],
					'right'  => [
						'title' => esc_html__('Right', 'onnat-extra'),
						'icon'  => 'eicon-text-align-right',
					],
				],
				'default'   => 'center',
				'selectors' => [
					'{{WRAPPER}} .kfw-elementor-widget-image-hotspots-wrapper' => 'text-align: {{VALUE}};',
                ],

			]);
		$this->end_controls_section();
	}

	public function _register_hotspots_section() {
		$this->start_controls_section( 'section_hotspots', [
			'label' => esc_html__( 'Hotspots', 'onnat-extra' )
        ]);

		$repeater = new KFRTR();

		$repeater->start_controls_tabs('tabs_hotspots');
			/**
			 * Tab: Hotspot
			 */
			$repeater->start_controls_tab('tab_hotspot',[
				'label' => esc_html__( 'Hotspot', 'onnat-extra' )
			]);

				$repeater->add_control( 'shape_settings', [
					'label'        => esc_html__('Shape Settings', 'onnat-extra'),
					'type'         => KFCM::POPOVER_TOGGLE,
					'label_off'    => __('None', 'onnat-extra'),
					'label_on'     => __('Custom', 'onnat-extra'),
					'return_value' => 'yes',
				] );

					$repeater->start_popover();
						$repeater->add_control( 'use_global_shape_settings', [
							'label'   => esc_html__( 'Use Global Style', 'onnat-extra' ),
							'type'    => KFCM::SWITCHER,
							'default' => 'yes',
						]);
						$repeater->add_control( 'shape', [
							'label'     => esc_html__( 'Type', 'onnat-extra' ),
							'type'      => KFCM::SELECT,
							'default'   => 'global',
							'condition' => [ 'use_global_shape_settings!' => 'yes' ],
							'options'   => [
								'global'         => esc_html__('Global', 'onnat-extra'),
								'bevel'          => esc_html__('Bevel', 'onnat-extra'),
								'circle'         => esc_html__('Circle', 'onnat-extra'),
								'heptagon'       => esc_html__('Heptagon', 'onnat-extra'),
								'hexagon'        => esc_html__('Hexagon', 'onnat-extra'),
								'nonagon'        => esc_html__('Nonagon', 'onnat-extra'),
								'octagon'        => esc_html__('Octagon', 'onnat-extra'),
								'pentagon'       => esc_html__('Pentagon', 'onnat-extra'),
								'rectangle'      => esc_html__('Rectangle', 'onnat-extra'),
								'rhombus'        => esc_html__('Rhombus', 'onnat-extra'),
								'rounded-square' => esc_html__('Rounded Square', 'onnat-extra'),
								'square'         => esc_html__('Square', 'onnat-extra'),
								'star'           => esc_html__('Star', 'onnat-extra'),
							]
						]);
						$repeater->add_responsive_control( 'shape_size', [
							'label'     => esc_html__( 'Size', 'onnat-extra' ),
							'type'      => KFCM::SLIDER,
							'range'     => [ 'px' => [  'min' => 10, 'max' => 200  ] ],
							'condition' => [ 'use_global_shape_settings!' => 'yes' ],
							'selectors' => [
								'{{WRAPPER}} {{CURRENT_ITEM}}.kfw-elementor-image-hotspot.kfw-elementor-image-hotspot-shape'                                                  => 'width:{{SIZE}}{{UNIT}};height:{{SIZE}}{{UNIT}};',
								'{{WRAPPER}} {{CURRENT_ITEM}}.kfw-elementor-image-hotspot.kfw-elementor-image-hotspot-shape.kfw-elementor-image-hotspot-rectangle-shape-type' => 'width:calc( {{SIZE}}{{UNIT}} + {{SIZE}}{{UNIT}} );height:{{SIZE}}{{UNIT}};'
							]
						]);
						$repeater->add_group_control( KFGCBG::get_type(), [
							'name'           => 'shape_normal_color',
							'condition'      => [ 'use_global_shape_settings!' => 'yes' ],
							'types'          => ['classic', 'gradient'],
							'exclude'        => [ 'image'],
							'selector'       => '{{WRAPPER}} {{CURRENT_ITEM}}.kfw-elementor-image-hotspot.kfw-elementor-image-hotspot-shape',
							'fields_options' => [
								'background' => [
									'label' => esc_html__( 'Normal Color Type', 'onnat-extra' ),
									'type' => KFCM::CHOOSE,
									'render_type' => 'ui',
								]
							],
						]);
						$repeater->add_group_control( KFGCBG::get_type(), [
							'name'           => 'shape_hover_color',
							'condition'      => [ 'use_global_shape_settings!' => 'yes' ],
							'types'          => ['classic', 'gradient'],
							'exclude'        => [ 'image'],
							'selector'       => '{{WRAPPER}} {{CURRENT_ITEM}}.kfw-elementor-image-hotspot.kfw-elementor-image-hotspot-shape:hover',
							'fields_options' => [
								'background' => [
									'label' => esc_html__( 'Hover Color Type', 'onnat-extra' ),
									'type' => KFCM::CHOOSE,
									'render_type' => 'ui',
								]
							],
						]);
					$repeater->end_popover();

				$repeater->add_control( 'type', [
					'label'   => esc_html__( 'Type', 'onnat-extra' ),
					'type'    => KFCM::CHOOSE,
					'options' => [
						'blank' => [
							'title' => esc_html__('Blank', 'onnat-extra'),
							'icon'  => '',
						],
						'dot'   => [
							'title' => esc_html__('Dot', 'onnat-extra'),
							'icon'  => 'eicon-dot-circle-o',
						],
						'icon'  => [
							'title' => esc_html__('Icon', 'onnat-extra'),
							'icon'  => 'eicon-plus-circle-o',
						],
						'image' => [
							'title' => esc_html__('Image', 'onnat-extra'),
							'icon'  => 'eicon-image-bold',
						],
						'text'  => [
							'title' => esc_html__('Text', 'onnat-extra'),
							'icon'  => 'eicon-animation-text',
						],
					],
					'default' => 'dot',
					'toggle'  => false,
				]);

				$repeater->add_control( 'icon', [
					'label'     => esc_html__( 'Icon', 'onnat-extra' ),
					'type'      => KFCM::ICONS,
					'default'   => [ '' ],
					'condition' => [ 'type' => 'icon' ],
				]);

				$repeater->add_control( 'image', [
					'label'     => esc_html__( 'Image', 'onnat-extra' ),
					'type'      => KFCM::MEDIA,
					'dynamic'   => [
						'active' => true,
					],
					'default'   => [
						'url' => KFUtils::get_placeholder_image_src(),
					],
					'condition' => [ 'type' => 'image' ],
				]);

				$repeater->add_control( 'text', [
					'label'     => esc_html__( 'Text', 'onnat-extra' ),
					'type'      => KFCM::TEXT,
					'dynamic'   => [
						'active' => true,
					],
					'condition' => [ 'type' => 'text' ],
					'default'   => '1',
				]);

				$repeater->add_group_control(KFGCT::get_type(), [
					'name'      => 'type_text_typo',
					'label'     => esc_html__( 'Typography', 'onnat-extra' ),
					'condition' => [ 'type' => 'text' ],
					'exclude'   => [ 'font_size' ],
					'selector'  => '{{WRAPPER}} {{CURRENT_ITEM}}.kfw-elementor-image-hotspot.kfw-elementor-image-hotspot-text-type span.kfw-elementor-image-hotspot-span'
				]);

				$repeater->add_control( 'type_settings', [
					'label'        => esc_html__('Type Settings', 'onnat-extra'),
					'type'         => KFCM::POPOVER_TOGGLE,
					'label_off'    => __('None', 'onnat-extra'),
					'label_on'     => __('Custom', 'onnat-extra'),
					'return_value' => 'yes',
					'condition'    => [ 'type' => [ 'dot', 'icon', 'text' ] ],
				] );
					$repeater->start_popover();
						$repeater->add_control( 'use_global_type_settings', [
							'label'   => esc_html__( 'Use Global Style', 'onnat-extra' ),
							'type'    => KFCM::SWITCHER,
							'default' => 'yes',
						]);
						$repeater->add_responsive_control( 'type_size', [
							'label'     => esc_html__( 'Size', 'onnat-extra' ),
							'type'      => KFCM::SLIDER,
							'range'     => [ 'px' => [  'min' => 1, 'max' => 200  ] ],
							'condition' => [ 'use_global_type_settings!' => 'yes', ],
							'selectors' => [
								'{{WRAPPER}} {{CURRENT_ITEM}}.kfw-elementor-image-hotspot.kfw-elementor-image-hotspot-dot-type span.kfw-elementor-image-hotspot-span.kfw-elementor-widget-image-hotspots-icon-dot' => 'height:{{SIZE}}{{UNIT}};width:{{SIZE}}{{UNIT}};',
								'{{WRAPPER}} {{CURRENT_ITEM}}.kfw-elementor-image-hotspot.kfw-elementor-image-hotspot-text-type span.kfw-elementor-image-hotspot-span' => 'font-size:{{SIZE}}{{UNIT}};',
								'{{WRAPPER}} {{CURRENT_ITEM}}.kfw-elementor-image-hotspot.kfw-elementor-image-hotspot-icon-type span.kfw-elementor-image-hotspot-span' => 'font-size:{{SIZE}}{{UNIT}};'
							]
						]);
						$repeater->add_control( 'type_normal_color', [
							'label'     => esc_html__('Normal Color', 'onnat-extra' ),
							'type'      => KFCM::COLOR,
							'condition' => [ 'use_global_type_settings!' => 'yes', ],
							'selectors' => [
								'{{WRAPPER}} {{CURRENT_ITEM}}.kfw-elementor-image-hotspot.kfw-elementor-image-hotspot-dot-type span.kfw-elementor-image-hotspot-span.kfw-elementor-widget-image-hotspots-icon-dot' => 'background-color:{{VALUE}};',
								'{{WRAPPER}} {{CURRENT_ITEM}}.kfw-elementor-image-hotspot.kfw-elementor-image-hotspot-text-type span.kfw-elementor-image-hotspot-span' => 'color:{{VALUE}};',
								'{{WRAPPER}} {{CURRENT_ITEM}}.kfw-elementor-image-hotspot.kfw-elementor-image-hotspot-icon-type span.kfw-elementor-image-hotspot-span' => 'color:{{VALUE}};'
							]
						]);
						$repeater->add_control( 'type_hover_color', [
							'label'     => esc_html__('Hover Color', 'onnat-extra' ),
							'type'      => KFCM::COLOR,
							'condition' => [ 'use_global_type_settings!' => 'yes', ],
							'selectors' => [
								'{{WRAPPER}} {{CURRENT_ITEM}}.kfw-elementor-image-hotspot.kfw-elementor-image-hotspot-shape:hover.kfw-elementor-image-hotspot-dot-type span.kfw-elementor-image-hotspot-span.kfw-elementor-widget-image-hotspots-icon-dot' => 'background-color:{{VALUE}};',
								'{{WRAPPER}} {{CURRENT_ITEM}}.kfw-elementor-image-hotspot.kfw-elementor-image-hotspot-text-type.kfw-elementor-image-hotspot-shape:hover span.kfw-elementor-image-hotspot-span' => 'color:{{VALUE}};',
								'{{WRAPPER}} {{CURRENT_ITEM}}.kfw-elementor-image-hotspot.kfw-elementor-image-hotspot-icon-type.kfw-elementor-image-hotspot-shape:hover span.kfw-elementor-image-hotspot-span' => 'color:{{VALUE}};',
							]
						]);
					$repeater->end_popover();

				$repeater->add_control('link_before_position', [ 'type' => KFCM::DIVIDER ]);

				$repeater->add_responsive_control( 'x_pos', [
					'label'     => esc_html__('X Postion', 'onnat-extra'),
					'type'      => KFCM::SLIDER,
					'range'     => [
						'px' 	=> [
							'min'  => 0,
							'max'  => 100,
							'step' => 0.1,
						],
					],
					'selectors' => [
						'{{WRAPPER}} {{CURRENT_ITEM}}' => 'top:{{SIZE}}%;'
					]
				]);

				$repeater->add_responsive_control( 'y_pos', [
					'label'     => esc_html__('Y Postion', 'onnat-extra'),
					'type'      => KFCM::SLIDER,
					'range'     => [
						'px' 	=> [
							'min'  => 0,
							'max'  => 100,
							'step' => 0.1,
						],
					],
					'selectors' => [
						'{{WRAPPER}} {{CURRENT_ITEM}}' => 'left:{{SIZE}}%;'
					]
				]);

				$repeater->add_control('link_before_hr', [ 'type' => KFCM::DIVIDER ]);

				$repeater->add_control( 'link', [
					'label'     => esc_html__( 'Link', 'onnat-extra' ),
					'type'      => KFCM::URL,
					'dynamic'   => [ 'active' => true, ],
					'default'   => [ 'url' => '' ],
				]);

				$repeater->add_control('link_after_hr', [ 'type' => KFCM::DIVIDER ]);

				$repeater->add_control( 'pulse', [
					'label'        => esc_html__( 'Pulse Effect', 'onnat-extra' ),
					'type'         => KFCM::SWITCHER,
					'default'      => 'none',
					'return_value' => 'block',
					'selectors'    => [ '{{WRAPPER}} .kfw-elementor-image-hotspot-has-pulse {{CURRENT_ITEM}}:before' => 'display:{{VALUE}};' ],
                ]);

				$repeater->add_control( 'pulse_notice', [
					'type'            => KFCM::RAW_HTML,
					'raw'             => '<div style="background:#fff;border:1px solid #c3c4c7;border-left-width:4px;border-left-color: #d63638;padding:2px;line-height:20px;">'. esc_html__('Pulse effect only works with circle, square and rounded square shape only', 'onnat-extra') .'</div>',
					'condition'       => [ 'pulse' => 'block' ],
					'content_classes' => ''
				]);

				$repeater->add_control( 'css_classes', [
					'label'        => __('CSS Classes', 'onnat-extra'),
					'title'        => __('Add your custom class without the dot. { e.g: custom-class }', 'onnat-extra'),
					'type'         => KFCM::TEXT,
					'default'      => '',
					'prefix_class' => '',
					'dynamic'      => [ 'active' => true, ],
					'render_type'  => 'ui',
				]);

			$repeater->end_controls_tab();

			/**
			 * Tab: Tooltip
			 */
			$repeater->start_controls_tab('tab_tooltip',[
				'label' => esc_html__( 'Tooltip', 'onnat-extra' )
			]);
				$repeater->add_control( 'tooltip', [
					'label'   => esc_html__( 'Enable Tooltip', 'onnat-extra' ),
					'type'    => KFCM::SWITCHER,
					'default' => 'yes',
				]);
				$repeater->add_control( 'open', [
					'label'     => esc_html__( 'Is Open?', 'onnat-extra' ),
					'type'      => KFCM::SWITCHER,
					'default'   => 'no',
					'condition' => [ 'tooltip' => 'yes' ],
				]);
				$repeater->add_control( 'source', [
					'label'     => esc_html__( 'Content Source', 'onnat-extra' ),
					'type'      => KFCM::CHOOSE,
					'options'   => [
						'editor'  => [
							'title' => esc_html__('Editor', 'onnat-extra'),
							'icon'  => 'eicon-text-area',
						],
						'template'  => [
							'title' => esc_html__('Template', 'onnat-extra'),
							'icon'  => 'eicon-t-letter',
						],
					],
					'default'   => 'editor',
					'toggle'    => false,
					'condition' => [ 'tooltip' => 'yes' ],
				]);
				$repeater->add_control( 'editor', [
					'label' => esc_html__('Editor', 'onnat-extra'),
					'type'      => KFCM::WYSIWYG,
					'default'   => esc_html__( 'Tooltip Content', 'onnat-extra'),
					'condition' => [
						'source'  => 'editor',
						'tooltip' => 'yes'
					],
				]);
				$repeater->add_control( 'template', [
					'label'     => esc_html__('Template', 'onnat-extra'),
					'type'      => KFCM::SELECT,
					'default'   => 0,
					'options'   => kf_extra_plugin_elementor()->get_elementor_library(),
					'condition' => [
						'source'  => 'template',
						'tooltip' => 'yes'
					],
				]);
				$repeater->add_control( 'animation', [
					'label'     => esc_html__( 'Animation', 'onnat-extra' ),
					'type'      => KFCM::SELECT,
					'default'   => 'global',
					'options'   => [
						'global' => esc_html__( 'Global', 'onnat-extra' ),
						'fade'   => esc_html__( 'Fade', 'onnat-extra' ),
						'grow'   => esc_html__( 'Grow', 'onnat-extra' ),
						'swing'  => esc_html__( 'Swing', 'onnat-extra' ),
						'slide'  => esc_html__( 'Slide', 'onnat-extra' ),
						'fall'   => esc_html__( 'Fall', 'onnat-extra' ),
					],
					'condition' =>  [ 'tooltip' => 'yes' ]
				]);
				$repeater->add_control( 'arrow', [
					'label'     => esc_html__( 'Add Arrow?', 'onnat-extra' ),
					'type'      => KFCM::CHOOSE,
					'options'   => [
						'yes'    => [
							'title' => esc_html__('Yes', 'onnat-extra'),
							'icon'  => 'eicon-check',
						],
						'no'     => [
							'title' => esc_html__('No', 'onnat-extra'),
							'icon'  => 'eicon-close',
						],
						'global' => [
							'title' => esc_html__('Use Global', 'onnat-extra'),
							'icon'  => 'eicon-global-settings',
						]
					],
					'default'   => 'global',
					'toggle'    => false,
					'condition' => [ 'tooltip' => 'yes' ]
				]);

				$repeater->add_control( 'position', [
					'label'     => esc_html__( 'Position', 'onnat-extra' ),
					'type'      => KFCM::SELECT,
					'default'   => 'global',
					'options'   => [
						'global' => esc_html__( 'Global', 'onnat-extra' ),
						'top'    => esc_html__( 'Top', 'onnat-extra' ),
						'right'  => esc_html__( 'Right', 'onnat-extra' ),
						'left'   => esc_html__( 'Left', 'onnat-extra' ),
						'bottom' => esc_html__( 'Bottom', 'onnat-extra' ),
					],
					'condition' =>  [ 'tooltip' => 'yes' ]
				]);
				$repeater->add_control( 'trigger', [
					'label'     => esc_html__( 'Trigger Tooltip On', 'onnat-extra' ),
					'type'      => KFCM::CHOOSE,
					'options'   => [
						'click'  => [
							'title' => esc_html__('Mouse Click', 'onnat-extra'),
							'icon'  => 'eicon-click',
						],
						'hover'  => [
							'title' => esc_html__('Mouse Hover', 'onnat-extra'),
							'icon'  => 'eicon-cursor-move',
						],
						'global' => [
							'title' => esc_html__('Use Global', 'onnat-extra'),
							'icon'  => 'eicon-global-settings',
						],
					],
					'default'   => 'global',
					'toggle'    => false,
					'condition' => [ 'tooltip' => 'yes' ],
				]);
				$repeater->add_control( 'customize', [
					'label'     => esc_html__( 'Customize It?', 'onnat-extra' ),
					'type'      => KFCM::SWITCHER,
					'default'   => 'yes',
					'condition' => [ 'tooltip' => 'yes' ]
				]);
				$repeater->add_control('customize_after_hr',[
					'type' => KFCM::DIVIDER,
					'condition' => [
						'tooltip'   => 'yes',
						'customize' => 'yes'
					]
				]);
				$repeater->add_control('maxwidth', [
					'label'     => esc_html__( 'Max Width', 'onnat-extra' ),
					'type'      => KFCM::SLIDER,
					'range'     => [ 'px' => [  'min' => 100, 'max' => 700  ] ],
					'condition' => [
						'tooltip' => 'yes',
						'customize' => 'yes'
					],
				]);
				$repeater->add_group_control(KFGCT::get_type(), [
					'name'      => 'typo',
					'label'     => esc_html__( 'Typography', 'onnat-extra' ),
					'selector'  => '{{CURRENT_ITEM}}-kfw-customized',
					'condition' => [
						'tooltip'   => 'yes',
						'customize' => 'yes'
					]
				]);
				$repeater->add_group_control( KFGCTS::get_type(), [
					'name'      => 'text_shadow',
					'condition' => [
						'tooltip'   => 'yes',
						'customize' => 'yes'
					],
					'selector'  => '{{CURRENT_ITEM}}-kfw-customized'
				]);
				$repeater->add_group_control( KFGCTST::get_type(), [
					'name'      => 'text_stroke',
					'condition' => [
						'tooltip'   => 'yes',
						'customize' => 'yes'
					],
					'selector'  => '{{CURRENT_ITEM}}-kfw-customized'
				]);
				$repeater->add_control('alignment',[
					'label'     => esc_html__( 'Alignment', 'onnat-extra' ),
					'type'      => KFCM::CHOOSE,
					'condition' => [
						'tooltip'   => 'yes',
						'customize' => 'yes'
					],
					'options'   => [
						'left'   => [
							'title' => esc_html__('Left', 'onnat-extra'),
							'icon'  => 'eicon-text-align-left',
						],
						'center' => [
							'title' => esc_html__('Center', 'onnat-extra'),
							'icon'  => 'eicon-text-align-center',
						],
						'right'  => [
							'title' => esc_html__('Right', 'onnat-extra'),
							'icon'  => 'eicon-text-align-right',
						],
					],
					'default'   => 'center',
					'selectors' => [
						'{{CURRENT_ITEM}}-kfw-customized' => 'text-align: {{VALUE}};'
					]
				]);
				$repeater->add_control( 'theme', [
					'label'        => esc_html__('Theme', 'onnat-extra'),
					'type'         => KFCM::POPOVER_TOGGLE,
					'label_off'    => __('None', 'onnat-extra'),
					'label_on'     => __('Custom', 'onnat-extra'),
					'return_value' => 'yes',
					'condition'    => [
						'tooltip' => 'yes',
						'customize' => 'yes'
					],
				]);
					$repeater->start_popover();
						$repeater->add_control( 'color_1', [
							'label'     => esc_html__('Color 1', 'onnat-extra' ),
							'type'      => KFCM::COLOR,
							'selectors' => [
								'{{CURRENT_ITEM}}-kfw-customized-wrap.tooltipster-sidetip .tooltipster-box' => 'background-color:{{VALUE}} !important;',

								'{{CURRENT_ITEM}}-kfw-customized-wrap.tooltipster-sidetip .tooltipster-box'                                 => 'background-color:{{VALUE}} !important; border: 2px solid {{VALUE}} !important;',

								'{{CURRENT_ITEM}}-kfw-customized-wrap.tooltipster-sidetip.tooltipster-top .tooltipster-arrow-border'        => 'border-top-color:{{VALUE}} !important;',
								'{{CURRENT_ITEM}}-kfw-customized-wrap.tooltipster-sidetip.tooltipster-top .tooltipster-arrow-background'    => 'border-top-color:{{VALUE}} !important;',

								'{{CURRENT_ITEM}}-kfw-customized-wrap.tooltipster-sidetip.tooltipster-right .tooltipster-arrow-border'      => 'border-right-color:{{VALUE}} !important;',
								'{{CURRENT_ITEM}}-kfw-customized-wrap.tooltipster-sidetip.tooltipster-right .tooltipster-arrow-background'  => 'border-right-color:{{VALUE}} !important;',

								'{{CURRENT_ITEM}}-kfw-customized-wrap.tooltipster-sidetip.tooltipster-bottom .tooltipster-arrow-border'     => 'border-bottom-color:{{VALUE}} !important;',
								'{{CURRENT_ITEM}}-kfw-customized-wrap.tooltipster-sidetip.tooltipster-bottom .tooltipster-arrow-background' => 'border-bottom-color:{{VALUE}} !important;',

								'{{CURRENT_ITEM}}-kfw-customized-wrap.tooltipster-sidetip.tooltipster-left .tooltipster-arrow-border'       => 'border-left-color:{{VALUE}} !important;',
								'{{CURRENT_ITEM}}-kfw-customized-wrap.tooltipster-sidetip.tooltipster-left .tooltipster-arrow-background'   => 'border-left-color:{{VALUE}} !important;',

							]
						]);
						$repeater->add_control( 'color_2', [
							'label'     => esc_html__('Color 2', 'onnat-extra' ),
							'type'      => KFCM::COLOR,
							'selectors' => [
								'{{CURRENT_ITEM}}-kfw-customized-wrap.tooltipster-sidetip .tooltipster-content' => 'color:{{VALUE}};'
							]
						]);
					$repeater->end_popover();
				$repeater->add_responsive_control( 'border_radius', [
					'label'      => esc_html__('Border Radius', 'onnat-extra' ),
					'type'       => KFCM::DIMENSIONS,
					'size_units' => [ 'px', '%' ],
					'condition'  => [
						'tooltip'   => 'yes',
						'customize' => 'yes'
					],
					'selectors'  => [ '{{CURRENT_ITEM}}-kfw-customized-wrap.tooltipster-sidetip .tooltipster-box' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ]
				]);
				$repeater->add_group_control( KFGCBS::get_type(), [
					'name'      => 'box_shadow',
					'condition' => [
						'tooltip'   => 'yes',
						'customize' => 'yes'
					],
					'selector'  => '{{CURRENT_ITEM}}-kfw-customized-wrap.tooltipster-sidetip .tooltipster-box'
				]);
				$repeater->add_responsive_control( 'padding',[
					'label'      => esc_html__('Padding', 'onnat-extra' ),
					'type'       => KFCM::DIMENSIONS,
					'condition'  => [
						'tooltip' => 'yes',
						'customize' => 'yes'
					],
					'size_units' => [ 'px', 'em', '%' ],
					'selectors'  => [ '{{CURRENT_ITEM}}-kfw-customized-wrap.tooltipster-sidetip .tooltipster-box' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ]
				]);
			$repeater->end_controls_tab();

		$repeater->end_controls_tabs();

		$this->add_control( 'hotspots', [
			'type'    => KFCM::REPEATER,
			'fields'  => $repeater->get_controls(),
			'default' => [
				[
					'use_global_shape_settings' => 'yes',
					'type'                      => 'dot',
					'use_global_type_settings'  => 'yes',
					'pulse'                     => 'block'
				]
			]
		]);

		$this->end_controls_section();
	}

	public function _register_hotspots_shape_style_section() {
		$this->start_controls_section( 'section_hotspot_shape_style',[
			'label' => esc_html__( 'Global Hotspots Shape', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_control('shape',[
				'label'   => esc_html__( 'Shape Type', 'onnat-extra' ),
				'type'    => KFCM::SELECT,
				'default' => 'circle',
				'options' => [
					'bevel'          => esc_html__('Bevel', 'onnat-extra'),
					'circle'         => esc_html__('Circle', 'onnat-extra'),
					'heptagon'       => esc_html__('Heptagon', 'onnat-extra'),
					'hexagon'        => esc_html__('Hexagon', 'onnat-extra'),
					'nonagon'        => esc_html__('Nonagon', 'onnat-extra'),
					'octagon'        => esc_html__('Octagon', 'onnat-extra'),
					'pentagon'       => esc_html__('Pentagon', 'onnat-extra'),
					'rectangle'      => esc_html__('Rectangle', 'onnat-extra'),
					'rhombus'        => esc_html__('Rhombus', 'onnat-extra'),
					'rounded-square' => esc_html__('Rounded Square', 'onnat-extra'),
					'square'         => esc_html__('Square', 'onnat-extra'),
					'star'           => esc_html__('Star', 'onnat-extra'),
				]
			]);
			$this->add_responsive_control( 'shape_size', [
				'label'     => esc_html__( 'Shape Size', 'onnat-extra' ),
				'type'      => KFCM::SLIDER,
				'range'     => [ 'px' => [  'min' => 10, 'max' => 200  ] ],
				'selectors' => [
					'{{WRAPPER}} .kfw-elementor-image-hotspot.kfw-elementor-image-hotspot-shape'                                                  => 'width:{{SIZE}}{{UNIT}};height:{{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .kfw-elementor-image-hotspot.kfw-elementor-image-hotspot-shape.kfw-elementor-image-hotspot-rectangle-shape-type' => 'width:calc( {{SIZE}}{{UNIT}} + {{SIZE}}{{UNIT}} );height:{{SIZE}}{{UNIT}};'
				]
			]);
			$this->start_controls_tabs('tabs_shape_colors');
				$this->start_controls_tab('tab_shape_normal_color',[
					'label' => esc_html__( 'Shape Normal Color', 'onnat-extra' )
				]);
					$this->add_group_control( KFGCBG::get_type(), [
						'name'           => 'shape_normal_color',
						'types'          => ['classic', 'gradient'],
						'exclude'        => [ 'image'],
						'selector'       => '{{WRAPPER}} .kfw-elementor-image-hotspot.kfw-elementor-image-hotspot-shape',
						'fields_options' => [
							'background' => [
								'label'       => esc_html__( 'Color Type', 'onnat-extra' ),
								'type'        => KFCM::CHOOSE,
								'render_type' => 'ui',
							]
						],
					]);
				$this->end_controls_tab();
				$this->start_controls_tab('tab_shape_hover_color',[
					'label' => esc_html__( 'Shape Hover Color', 'onnat-extra' )
				]);
					$this->add_group_control( KFGCBG::get_type(), [
						'name'           => 'shape_hover_color',
						'types'          => ['classic', 'gradient'],
						'exclude'        => [ 'image'],
						'selector'       => '{{WRAPPER}} .kfw-elementor-image-hotspot.kfw-elementor-image-hotspot-shape:hover',
						'fields_options' => [
							'background' => [
								'label'       => esc_html__( 'Color Type', 'onnat-extra' ),
								'type'        => KFCM::CHOOSE,
								'render_type' => 'ui',
							]
						],
					]);
				$this->end_controls_tab();
			$this->end_controls_tabs();
		$this->end_controls_section();
	}

	public function _register_hotspots_dot_type_style_section() {
		$this->start_controls_section( 'section_hotspot_dot_type_style',[
			'label' => esc_html__( 'Global Hotspots Dot Type', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_responsive_control( 'dot_size', [
				'label'     => esc_html__( 'Dot Size', 'onnat-extra' ),
				'type'      => KFCM::SLIDER,
				'range'     => [ 'px' => [  'min' => 5, 'max' => 200  ] ],
				'selectors' => [
					'{{WRAPPER}} .kfw-elementor-image-hotspot.kfw-elementor-image-hotspot-dot-type span.kfw-elementor-image-hotspot-span.kfw-elementor-widget-image-hotspots-icon-dot' => 'height:{{SIZE}}{{UNIT}};width:{{SIZE}}{{UNIT}};'
				]
			]);
			$this->add_control( 'dot_type_normal_color', [
				'label'     => esc_html__('Normal Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kfw-elementor-image-hotspot.kfw-elementor-image-hotspot-dot-type span.kfw-elementor-image-hotspot-span.kfw-elementor-widget-image-hotspots-icon-dot' => 'background-color:{{VALUE}};'
				]
			]);
			$this->add_control( 'dot_type_hover_color', [
				'label'     => esc_html__('Hover Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kfw-elementor-image-hotspot.kfw-elementor-image-hotspot-shape:hover span.kfw-elementor-image-hotspot-span.kfw-elementor-widget-image-hotspots-icon-dot' => 'background-color:{{VALUE}};',
				]
			]);
		$this->end_controls_section();
	}

	public function _register_hotspots_icon_type_style_section() {
		$this->start_controls_section( 'section_hotspot_icon_type_style',[
			'label' => esc_html__( 'Global Hotspots Icon Type', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_responsive_control( 'icon_size', [
				'label'     => esc_html__( 'Icon Size', 'onnat-extra' ),
				'type'      => KFCM::SLIDER,
				'range'     => [ 'px' => [  'min' => 5, 'max' => 200  ] ],
				'selectors' => [
					'{{WRAPPER}} .kfw-elementor-image-hotspot.kfw-elementor-image-hotspot-icon-type span.kfw-elementor-image-hotspot-span' => 'font-size:{{SIZE}}{{UNIT}};'
				]
			]);
			$this->add_control( 'icon_type_normal_color', [
				'label'     => esc_html__('Normal Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kfw-elementor-image-hotspot.kfw-elementor-image-hotspot-icon-type span.kfw-elementor-image-hotspot-span' => 'color:{{VALUE}};'
				]
			]);
			$this->add_control( 'icon_type_hover_color', [
				'label'     => esc_html__('Hover Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kfw-elementor-image-hotspot.kfw-elementor-image-hotspot-shape:hover span.kfw-elementor-image-hotspot-span' => 'color:{{VALUE}};'
				]
			]);
		$this->end_controls_section();
	}

	public function _register_hotspots_text_type_style_section() {
		$this->start_controls_section( 'section_hotspot_txt_type_style',[
			'label' => esc_html__( 'Global Hostspots Text Type', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_group_control(KFGCT::get_type(), [
				'name'     => 'type_text_typo',
				'label'    => esc_html__( 'Typography', 'onnat-extra' ),
				'selector' => '{{WRAPPER}} .kfw-elementor-image-hotspot.kfw-elementor-image-hotspot-text-type span.kfw-elementor-image-hotspot-span'
			]);
			$this->add_control( 'txt_type_normal_color', [
				'label'     => esc_html__('Normal Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kfw-elementor-image-hotspot.kfw-elementor-image-hotspot-text-type span.kfw-elementor-image-hotspot-span' => 'color:{{VALUE}};'
				]
			]);
			$this->add_control( 'txt_type_hover_color', [
				'label'     => esc_html__('Hover Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kfw-elementor-image-hotspot.kfw-elementor-image-hotspot-text-type.kfw-elementor-image-hotspot-shape:hover span.kfw-elementor-image-hotspot-span' => 'color:{{VALUE}};'
				]
			]);
		$this->end_controls_section();
	}

	public function _register_tooltip_style_section() {
		$this->start_controls_section( 'section_tooltips_style',[
			'label' => esc_html__( 'Global Tooltips', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_control( 'anim', [
				'label'              => esc_html__( 'Animation', 'onnat-extra' ),
				'type'               => KFCM::SELECT,
				'default'            => 'fade',
				'frontend_available' => true,
				'options'            => [
					'fade'  => esc_html__( 'Fade', 'onnat-extra' ),
					'grow'  => esc_html__( 'Grow', 'onnat-extra' ),
					'swing' => esc_html__( 'Swing', 'onnat-extra' ),
					'slide' => esc_html__( 'Slide', 'onnat-extra' ),
					'fall'  => esc_html__( 'Fall', 'onnat-extra' ),
				]
			]);
			$this->add_control( 'arrow', [
				'label'              => esc_html__( 'Add Arrow?', 'onnat-extra' ),
				'type'               => KFCM::CHOOSE,
				'options'            => [
					'yes'    => [
						'title' => esc_html__('Yes', 'onnat-extra'),
						'icon'  => 'eicon-check',
					],
					'no'     => [
						'title' => esc_html__('No', 'onnat-extra'),
						'icon'  => 'eicon-close',
					],
				],
				'frontend_available' => true,
				'default'            => 'yes',
				'toggle'             => false,
			]);
			$this->add_control( 'position', [
				'label'              => esc_html__( 'Position', 'onnat-extra' ),
				'type'               => KFCM::SELECT,
				'default'            => 'top',
				'frontend_available' => true,
				'options'            => [
					'top'    => esc_html__( 'Top', 'onnat-extra' ),
					'right'  => esc_html__( 'Right', 'onnat-extra' ),
					'left'   => esc_html__( 'Left', 'onnat-extra' ),
					'bottom' => esc_html__( 'Bottom', 'onnat-extra' ),
				],
			]);
			$this->add_control( 'trigger', [
				'label'              => esc_html__( 'Trigger Tooltip On', 'onnat-extra' ),
				'type'               => KFCM::CHOOSE,
				'frontend_available' => true,
				'options'            => [
					'click'  => [
						'title' => esc_html__('Mouse Click', 'onnat-extra'),
						'icon'  => 'eicon-click',
					],
					'hover'  => [
						'title' => esc_html__('Mouse Hover', 'onnat-extra'),
						'icon'  => 'eicon-cursor-move',
					],
				],
				'default'            => 'click',
				'toggle'             => false,
			]);
			$this->add_group_control(KFGCT::get_type(), [
				'name'     => 'typo',
				'label'    => esc_html__( 'Typography', 'onnat-extra' ),
				'selector' => '.tooltipster-kfw-theme-'. $this->get_id() .' .kfw-elementor-image-tooltip-content'
			]);
			$this->add_control('color',[
				'label'     => esc_html__( 'Text Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'.tooltipster-kfw-theme-'. $this->get_id() .' .kfw-elementor-image-tooltip-content' => 'color:{{VALUE}};'
				]
			]);
			$this->add_group_control( KFGCTS::get_type(), [
				'name'     => 'text_shadow',
				'selector' => '.tooltipster-kfw-theme-'. $this->get_id() .' .kfw-elementor-image-tooltip-content'
			]);
			$this->add_group_control( KFGCTST::get_type(), [
				'name'     => 'text_stroke',
				'selector' => '.tooltipster-kfw-theme-'. $this->get_id() .' .kfw-elementor-image-tooltip-content'
			]);
			$this->add_control('alignment',[
				'label'     => esc_html__( 'Alignment', 'onnat-extra' ),
				'type'      => KFCM::CHOOSE,
				'options'   => [
					'left'   => [
						'title' => esc_html__('Left', 'onnat-extra'),
						'icon'  => 'eicon-text-align-left',
					],
					'center' => [
						'title' => esc_html__('Center', 'onnat-extra'),
						'icon'  => 'eicon-text-align-center',
					],
					'right'  => [
						'title' => esc_html__('Right', 'onnat-extra'),
						'icon'  => 'eicon-text-align-right',
					],
				],
				'default'   => 'center',
				'selectors' => [
					'.tooltipster-kfw-theme-'. $this->get_id() .' .kfw-elementor-image-tooltip-content' => 'text-align: {{VALUE}};'
				]
			]);
			$this->add_control( 'theme', [
				'label'        => esc_html__('Theme', 'onnat-extra'),
				'type'         => KFCM::POPOVER_TOGGLE,
				'label_off'    => __('None', 'onnat-extra'),
				'label_on'     => __('Custom', 'onnat-extra'),
				'return_value' => 'yes',
			]);
				$this->start_popover();
					$this->add_control( 'color_1', [
						'label'     => esc_html__('Color 1', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => [
							'.tooltipster-sidetip.tooltipster-kfw-theme-'. $this->get_id() .' .tooltipster-box'                                 => 'background-color:{{VALUE}}; border: 2px solid {{VALUE}};',

							'.tooltipster-sidetip.tooltipster-kfw-theme-'. $this->get_id() .'.tooltipster-top .tooltipster-arrow-border'        => 'border-top-color:{{VALUE}};',
							'.tooltipster-sidetip.tooltipster-kfw-theme-'. $this->get_id() .'.tooltipster-top .tooltipster-arrow-background'    => 'border-top-color:{{VALUE}};',

							'.tooltipster-sidetip.tooltipster-kfw-theme-'. $this->get_id() .'.tooltipster-right .tooltipster-arrow-border'      => 'border-right-color:{{VALUE}};',
							'.tooltipster-sidetip.tooltipster-kfw-theme-'. $this->get_id() .'.tooltipster-right .tooltipster-arrow-background'  => 'border-right-color:{{VALUE}};',

							'.tooltipster-sidetip.tooltipster-kfw-theme-'. $this->get_id() .'.tooltipster-bottom .tooltipster-arrow-border'     => 'border-bottom-color:{{VALUE}};',
							'.tooltipster-sidetip.tooltipster-kfw-theme-'. $this->get_id() .'.tooltipster-bottom .tooltipster-arrow-background' => 'border-bottom-color:{{VALUE}};',

							'.tooltipster-sidetip.tooltipster-kfw-theme-'. $this->get_id() .'.tooltipster-left .tooltipster-arrow-border'       => 'border-left-color:{{VALUE}};',
							'.tooltipster-sidetip.tooltipster-kfw-theme-'. $this->get_id() .'.tooltipster-left .tooltipster-arrow-background'   => 'border-left-color:{{VALUE}};',
						]
					]);

					$this->add_control( 'color_2', [
						'label'     => esc_html__('Color 2', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => [
							'.tooltipster-sidetip.tooltipster-kfw-theme-'. $this->get_id() .' .tooltipster-content' => 'color:{{VALUE}};'
						]
					]);
				$this->end_popover();

				$this->add_responsive_control( 'border_radius', [
					'label'      => esc_html__('Border Radius', 'onnat-extra' ),
					'type'       => KFCM::DIMENSIONS,
					'size_units' => [ 'px', '%' ],
					'selectors'  => [
						'.tooltipster-sidetip.tooltipster-kfw-theme-'. $this->get_id() .' .tooltipster-box' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
					]
				]);
				$this->add_group_control( KFGCBS::get_type(), [
					'name'     => 'box_shadow',
					'selector' => '.tooltipster-sidetip.tooltipster-kfw-theme-'. $this->get_id() .' .tooltipster-box'
				]);
				$this->add_responsive_control( 'padding',[
					'label'      => esc_html__('Padding', 'onnat-extra' ),
					'type'       => KFCM::DIMENSIONS,
					'size_units' => [ 'px', 'em', '%' ],
					'selectors'  => [
						'.tooltipster-sidetip.tooltipster-kfw-theme-'. $this->get_id() .' .tooltipster-box' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
					]
				]);

		$this->end_controls_section();
	}

	/**
	 * Get attachment image src.
	 */
	public static function _get_image_src( $attachment_id, $image_size_key, array $settings ) {
        return KFGCIS::get_attachment_image_src( $attachment_id, $image_size_key, $settings );
	}

	public static function _get_image_html( $settings ) {
		return KFGCIS::get_attachment_image_html( $settings );
	}

	public function _get_image_title( $attachment ) {
		if ( empty( $attachment['id'] ) ) {
			return '';
		}

		return get_the_title( $attachment['id'] );
	}

	public function _get_image_alt( $instance ) {
		if ( empty( $instance['id'] ) ) {
			// For `Insert From URL` images.
			return isset( $instance['alt'] ) ? trim( strip_tags( $instance['alt'] ) ) : '';
		}

		$attachment_id = $instance['id'];
		if ( ! $attachment_id ) {
			return '';
		}

		$attachment = get_post( $attachment_id );
		if ( ! $attachment ) {
			return '';
		}

		$alt = get_post_meta( $attachment_id, '_wp_attachment_image_alt', true );
		if ( ! $alt ) {
			$alt = $attachment->post_excerpt;
			if ( ! $alt ) {
				$alt = $attachment->post_title;
			}
		}
		return trim( strip_tags( $alt ) );
	}

	/**
	 * Render icon from icon manger
	 */
    public function _render_icons( $icon , $echo = true ) {
		echo '<span class="kfw-elementor-image-hotspot-span kfw-elementor-icon-wrapper">';
			KFIM::render_icon( $icon, [ 'aria-hidden' => 'true' ] );
		echo '</span>';
	}

	/**
	 * Get template content
	 */
	public function _generate_template_content( $template_id ) {
		return KFEl::instance()->frontend->get_builder_content_for_display( $template_id );
	}

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-image-hotspots");
		include $render;
    }
}