<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use Elementor\Widget_Base as WB;
use Elementor\Controls_Manager as KFCM;

use Elementor\Group_Control_Background as KFGCBG;
use Elementor\Group_Control_Border as KFGCBR;
use Elementor\Group_Control_Box_Shadow as KFGCBS;

class KFW_Instagram extends WB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-instagram';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Instagram', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_core_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-instagram-post';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'kfw', 'kin', 'instagram', 'feed', 'image', 'images' ];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
        return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/instagram';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

		wp_register_style( 'kfw-instagram',
			KFE_CONST_URL . 'libraries/elementor/assets/css/instagram' . KFE_CONST_DEBUG_SUFFIX . '.css',
			[],
			KFE_CONST_VERSION
		);
    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'kfw-instagram' ];
	}

	/**
	 * Get script dependencies.
	 *
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @return array Element scripts dependencies.
	 */
	public function get_script_depends() {
		return [ ];
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
		$this->_register_insta_section();
		$this->_register_grid_section();
		$this->_register_item_style_section();
		$this->_register_overlay_style_section();
		$this->_register_style3_icon_style_section();
    }

    public function _register_insta_section() {
		$this->start_controls_section( 'section_insta', [
			'label' => esc_html__( 'API Settings', 'onnat-extra' )
		]);
			$this->add_control( 'access_token', [
                'label'       => esc_html__( 'Access Token', 'onnat-extra' ),
                'label_block' => true,
                'type'        => KFCM::TEXT,
				'description' => sprintf(
					esc_html__('To obtain an Instagram access token, you need to follow the %s for detailed instructions.', 'onnat-extra' ),
					sprintf( '<a href="%s" target="_blank">%s</a>', KFE_CONST_THEME_DOC . 'how-to/get-instagram-access-token', esc_html__('documentation', 'onnat-extra') )
				)
			]);
			$this->add_control( 'count', [
				'label'       => esc_html__( 'Image Count(s)', 'onnat-extra' ),
				'description' => esc_html__( 'Specify the number of Instagram images to display.', 'onnat-extra' ),
				'type'        => KFCM::NUMBER,
				'default'     => 6,
				'min'         => 1,
				'max'         => 50,
				'step'        => 1,
				'condition'   => [
					'access_token!' => ''
				]
			]);
			$this->add_control( 'enable_cache', [
				'label'        => esc_html__( 'Enable Cache Timeout?', 'onnat-extra' ),
				'type'         => KFCM::SWITCHER,
				'return_value' => 'yes',
				'default'      => 'yes',
				'condition'    => [
					'access_token!' => ''
				]
			]);
			$this->add_control( 'cache', [
				'label'     => esc_html__( 'Cache Timeout?', 'onnat-extra' ),
				'type'      => KFCM::SELECT,
				'default'   => 'hour',
				'options'   => [
                    'minute' => esc_html__( 'Minute', 'onnat-extra' ),
                    'hour'   => esc_html__( 'Hour', 'onnat-extra' ),
                    'day'    => esc_html__( 'Day', 'onnat-extra' ),
                    'week'   => esc_html__( 'Week', 'onnat-extra' ),
				],
				'condition' => [
					'enable_cache'  => 'yes',
					'access_token!' => ''
				]
			]);
		$this->end_controls_section();
    }

    public function _register_grid_section() {
		$this->start_controls_section( 'section_grid', [
			'label'     => esc_html__( 'Grid Settings', 'onnat-extra' ),
			'condition' => [
				'access_token!' => ''
			]
		]);
			$this->add_control('layout', [
				'label'   => esc_html__( 'Layout', 'onnat-extra' ),
				'type'    => KFCM::CHOOSE,
				'options' => [
					'style-1' => [
						'title' => esc_html__('Grid Style 1', 'onnat-extra' ),
						'icon'  => 'eicon-posts-grid'
					],
					'style-2' => [
						'title' => esc_html__('Grid Style 2', 'onnat-extra' ),
						'icon'  => 'eicon-form-vertical'
					],
					'style-3' => [
						'title' => esc_html__('Grid Style 3', 'onnat-extra' ),
						'icon'  => 'eicon-editor-link'
					],
				],
				'default'      => 'style-1',
				'prefix_class' => 'kfw-ele-widget-instagram-',
				'toggle'       => false
			]);
			$this->add_responsive_control('cols', [
				'label'          => esc_html__( 'Columns', 'onnat-extra' ),
				'type'           => KFCM::CHOOSE,
				'label_block'    => true,
				'options'        => [
					'1'  => [ 'title' => esc_html__('1 Column', 'onnat-extra'), 'icon' => 'kfw-ctrl-icon kfw-ctrl-numeric-icon' ],
					'2'  => [ 'title' => esc_html__('2 Columns', 'onnat-extra'), 'icon' => 'kfw-ctrl-icon kfw-ctrl-numeric-2-icon' ],
					'3'  => [ 'title' => esc_html__('3 Columns', 'onnat-extra'), 'icon' => 'kfw-ctrl-icon kfw-ctrl-numeric-3-icon' ],
					'4'  => [ 'title' => esc_html__('4 Columns', 'onnat-extra'), 'icon' => 'kfw-ctrl-icon kfw-ctrl-numeric-4-icon' ],
					'5'  => [ 'title' => esc_html__('5 Columns', 'onnat-extra'), 'icon' => 'kfw-ctrl-icon kfw-ctrl-numeric-5-icon' ],
					'6'  => [ 'title' => esc_html__('6 Columns', 'onnat-extra'), 'icon' => 'kfw-ctrl-icon kfw-ctrl-numeric-6-icon' ],
					'7'  => [ 'title' => esc_html__('7 Columns', 'onnat-extra'), 'icon' => 'kfw-ctrl-icon kfw-ctrl-numeric-7-icon' ],
					'8'  => [ 'title' => esc_html__('8 Columns', 'onnat-extra'), 'icon' => 'kfw-ctrl-icon kfw-ctrl-numeric-8-icon' ],
					'9'  => [ 'title' => esc_html__('9 Columns', 'onnat-extra'), 'icon' => 'kfw-ctrl-icon kfw-ctrl-numeric-9-icon' ],
					'10' => [ 'title' => esc_html__('10 Columns', 'onnat-extra'), 'icon' => 'kfw-ctrl-icon kfw-ctrl-numeric-10-icon' ],
					'11' => [ 'title' => esc_html__('11 Columns', 'onnat-extra'), 'icon' => 'kfw-ctrl-icon kfw-ctrl-numeric-11-icon' ],
					'12' => [ 'title' => esc_html__('12 Columns', 'onnat-extra'), 'icon' => 'kfw-ctrl-icon kfw-ctrl-numeric-12-icon' ],
				],
				'default'        => '3',
				'tablet_default' => '2',
				'mobile_default' => '1',
				'toggle'         => false,
				'selectors'      => [
					'{{WRAPPER}} .kfw-ele-widget-instagram-wrapper' => 'grid-template-columns: repeat({{SIZE}}, minmax(0, 1fr));',
				],
				'condition'      => [
					'access_token!' => ''
				]
			]);
			$this->add_responsive_control('col_gap',[
				'label'   => esc_html__( 'Item Gap', 'onnat-extra' ),
				'type'    => KFCM::SLIDER,
				'default' => [
					'size' => 20,
				],
				'range'   => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-instagram-wrapper' => 'grid-column-gap: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'access_token!' => ''
				]
			]);
			$this->add_responsive_control('row_gap',[
				'label'     => esc_html__( 'Row Gap', 'onnat-extra' ),
				'type'      => KFCM::SLIDER,
				'default'   => [ 'size' => 20,],
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-instagram-wrapper' => 'grid-row-gap: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'access_token!' => ''
				],
			]);
			$this->add_control('item_match_height', [
				'label'        => esc_html__( 'Item Match Hieght', 'onnat-extra' ),
				'type'         => KFCM::SWITCHER,
				'return_value' => 'yes',
				'prefix_class' => 'kfw-ele-instagram-image-item-match-height-',
				'render_type'  => 'template',
				'condition' => [
					'access_token!' => ''
				]
			]);
			$this->add_responsive_control('item_height',[
				'label'      => esc_html__( 'Item Hieght', 'onnat-extra' ),
				'type'       => KFCM::SLIDER,
				'range'      => [
					'px' => [
						'min' => 10,
						'max' => 1000,
					],
					'%'  => [
						'min' => 0,
						'max' => 100,
					],
				],
				'size_units' => ['px', '%', ],
				'default'    => [],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-instagram-img-wrap .kfw-ele-widget-instagram-img' => 'height: {{SIZE}}{{UNIT}};'
				],
				'condition'  => [
					'access_token!'     => '',
					'item_match_height' => 'yes'
				]
			]);
		$this->end_controls_section();
    }

	public function _register_item_style_section() {
		$this->start_controls_section( 'section_item_style',[
			'label' => esc_html__( 'Items', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->start_controls_tabs( 'tabs_item_style' );
				$this->start_controls_tab( 'tabs_item_normal', [
					'label' => esc_html__( 'Normal', 'onnat-extra' ),
				]);
					$this->add_group_control( KFGCBG::get_type(), [
						'name'     => 'items_bg',
						'selector' => '{{WRAPPER}} .kfw-ele-widget-instagram-img-wrap',
					]);
					$this->add_control( 'hr_1', [
						'type' => KFCM::DIVIDER,
					]);
					$this->add_responsive_control( 'items_padding',[
						'label'      => esc_html__('Padding', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', 'em', '%' ],
						'default'    => [
							'top'      => '',
							'right'    => '',
							'bottom'   => '',
							'left'     => '',
							'unit'     => 'px',
							'isLinked' => false
						],
						'selectors'  => [
							'{{WRAPPER}} .kfw-ele-widget-instagram-img-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
						],
					]);
					$this->add_responsive_control( 'items_margin',[
						'label'      => esc_html__('Margin', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', 'em', '%' ],
						'selectors'  => [
							'{{WRAPPER}} .kfw-ele-widget-instagram-img-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
						],
					]);
					$this->add_control( 'hr_2', [
						'type' => KFCM::DIVIDER,
					]);
					$this->add_group_control( KFGCBR::get_type(), [
						'name'     => 'items_border',
						'selector' => '{{WRAPPER}} .kfw-ele-widget-instagram-img-wrap'
					]);
					$this->add_control( 'hr_3', [
						'type' => KFCM::DIVIDER,
					]);
					$this->add_responsive_control( 'items_border_radius', [
						'label'      => esc_html__('Border Radius', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', '%' ],
						'default'    => [
							'top'      => '',
							'right'    => '',
							'bottom'   => '',
							'left'     => '',
							'unit'     => 'px',
							'isLinked' => false
						],
						'selectors'  => [
							'{{WRAPPER}} .kfw-ele-widget-instagram-img-wrap, {{WRAPPER}} .kfw-ele-widget-instagram-img-wrap .kfw-ele-widget-instagram-img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
						]
					]);
					$this->add_control( 'hr_4', [
						'type' => KFCM::DIVIDER,
					]);
					$this->add_group_control( KFGCBS::get_type(), [
						'name'     => 'items_box_shadow',
						'selector' => '{{WRAPPER}} .kfw-ele-widget-instagram-img-wrap'
					]);
				$this->end_controls_tab();
				$this->start_controls_tab( 'tabs_item_hover', [
					'label' => esc_html__( 'Hover', 'onnat-extra' ),
				]);
					$this->add_control('items_h_border_color',[
						'label'     => esc_html__( 'Border Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'condition' => [
							'items_border_border!' => '',
						],
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-instagram-img-wrap:hover' => 'border-color: {{VALUE}};',
						],
					]);
					$this->add_group_control(KFGCBS::get_type(),[
						'name'     => 'item_hover_box_shadow',
						'selector' => '{{WRAPPER}} .kfw-ele-widget-instagram-img-wrap:hover',
					]);
				$this->end_controls_tab();
			$this->end_controls_tabs();
		$this->end_controls_section();
	}

	public function _register_overlay_style_section() {
		$this->start_controls_section( 'section_overlay_style',[
			'label' => esc_html__( 'Overlay', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_responsive_control( 'icon_size', [
				'label'      => esc_html__('Icon Size', 'onnat-extra' ),
				'type'       => KFCM::SLIDER,
				'size_units' => [ 'px' ],
				'range'      => [
					'px' => [
						'min' => 0,
						'max' => 200,
					],
				],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-instagram-img-wrap .kfw-ele-widget-instagram-overlay .kinfw-icon' => 'font-size: {{SIZE}}{{UNIT}};'
				],
			]);
			$this->add_group_control( KFGCBG::get_type(), [
				'name'     => 'overlay_bg',
				'selector' => '{{WRAPPER}} .kfw-ele-widget-instagram-img-wrap .kfw-ele-widget-instagram-overlay',
			]);
			$this->add_control( 'icon_color', [
				'label'     => esc_html__('Icon Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-instagram-img-wrap .kfw-ele-widget-instagram-overlay .kinfw-icon' => 'color: {{VALUE}};'
				],
			]);
			$this->add_control('icon_rotate',[
				'label'     => esc_html__( 'Icon Rotate', 'onnat-extra' ),
				'type'      => KFCM::SLIDER,
				'default'   => [
					'size' => 0,
					'unit' => 'deg',
				],
				'range'     => [
					'deg' => [
						'max' => 360,
						'min' => -360,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-instagram-img-wrap:hover .kfw-ele-widget-instagram-overlay .kinfw-icon'   => 'transform: rotate({{SIZE}}{{UNIT}});',
				],
			]);
		$this->end_controls_section();
	}

	public function _register_style3_icon_style_section() {
		$this->start_controls_section( 'section_style_3_icon_style',[
			'label'     => esc_html__( 'Style 3 : Icon', 'onnat-extra' ),
			'tab'       => KFCM::TAB_STYLE,
			'condition' => [
				'layout' => 'style-3'
			]
		]);
			$this->add_responsive_control( 'wrap_size',[
				'label'      => esc_html__('Size', 'onnat-extra' ),
				'type'       => KFCM::SLIDER,
				'size_units' => [ 'px' ],
				'range'      => [
					'px' => [
						'min' => 10,
						'max' => 500,
					],
				],
				'selectors'  => [
					'{{WRAPPER}}.kfw-ele-widget-instagram-style-3 .kfw-ele-widget-instagram-icon' => 'width:{{SIZE}}{{UNIT}}; height:{{SIZE}}{{UNIT}};'
				]
			]);
			$this->add_responsive_control( 's3_icon_size', [
				'label'      => esc_html__('Icon Size', 'onnat-extra' ),
				'type'       => KFCM::SLIDER,
				'size_units' => [ 'px' ],
				'range'      => [
					'px' => [
						'min' => 0,
						'max' => 200,
					],
				],
				'selectors'  => [
					'{{WRAPPER}}.kfw-ele-widget-instagram-style-3 .kfw-ele-widget-instagram-icon .kinfw-icon' => 'font-size: {{SIZE}}{{UNIT}};'
				],
			]);
			$this->add_control( 's3_icon_color', [
				'label'     => esc_html__('Icon Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}}.kfw-ele-widget-instagram-style-3 .kfw-ele-widget-instagram-icon .kinfw-icon' => 'color: {{VALUE}};'
				],
			]);
			$this->add_group_control( KFGCBG::get_type(), [
				'name'     => 's3_icon_bg',
				'selector' => '{{WRAPPER}}.kfw-ele-widget-instagram-style-3 .kfw-ele-widget-instagram-icon',
			]);
			$this->add_group_control( KFGCBR::get_type(), [
				'name'     => 's3_icon_border',
				'selector' => '{{WRAPPER}}.kfw-ele-widget-instagram-style-3 .kfw-ele-widget-instagram-icon'
			]);
			$this->add_responsive_control( 's3_icon_border_radius', [
				'label'      => esc_html__('Border Radius', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'default'    => [
					'top'      => '',
					'right'    => '',
					'bottom'   => '',
					'left'     => '',
					'unit'     => 'px',
					'isLinked' => false
				],
				'selectors'  => [
					'{{WRAPPER}}.kfw-ele-widget-instagram-style-3 .kfw-ele-widget-instagram-icon' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_group_control(KFGCBS::get_type(),[
				'name'     => 's3_icon_box_shadow',
				'selector' => '{{WRAPPER}}.kfw-ele-widget-instagram-style-3 .kfw-ele-widget-instagram-icon',
			]);
		$this->end_controls_section();
	}

	public function _get_transient_key() {
		$settings = $this->get_settings();

		return sprintf(
			'kfw_insta_%s_post_count_%s',
			$this->get_id(),
			$settings['count'],
		);
	}

	public function _get_cache_timeout( $use_cache = 'yes', $cache = 'hour' ) {
		$cache_timeout = 1;

		if( $use_cache ) {
			switch( $cache ) {
				case 'minute':
					$cache_timeout = MINUTE_IN_SECONDS;
				break;

				case 'hour':
					$cache_timeout = HOUR_IN_SECONDS;
				break;

				case 'day':
					$cache_timeout = DAY_IN_SECONDS;
				break;

				case 'week':
					$cache_timeout = WEEK_IN_SECONDS;
				break;

				default:
					$cache_timeout = HOUR_IN_SECONDS;
				break;
			}
		}

		return $cache_timeout;
	}

	public function _get_items() {
		$settings = $this->get_settings();
		$items    = [];

		if( empty( $settings['access_token'] ) ) {
			return $items;
		}

		$cache_timeout = $this->_get_cache_timeout( $use_cache = filter_var( $settings['enable_cache'], FILTER_VALIDATE_BOOLEAN ), $cache = $settings['cache'] );

		$items = $this->_get_posts([
			'access_token'  => $settings['access_token'],
			'endpoint'      => 'self',
			'count'         => $settings['count'],
			'comments'      => isset( $settings['comments'] ) ? filter_var( $settings['comments'], FILTER_VALIDATE_BOOLEAN ) : false,
			'likes'         => isset( $settings['like'] ) ? filter_var( $settings['likes'], FILTER_VALIDATE_BOOLEAN ) : false,
			'cache_timeout' => $cache_timeout
		]);

		return $items;
	}


	public function _get_posts( $config ) {

		$transient_key = md5( $this->_get_transient_key() );
		$data          = get_transient( $transient_key );

		if ( ! empty( $data ) && 1 !== $config['cache_timeout'] ) {
			return $data;
		}

		$response = $this->_remote_get( $config );

        if ( is_wp_error( $response ) ) {
            return [];
        }

		$data = $this->_get_response_data_from_official_api( $response );

		if ( empty( $data ) ) {
            return [];
        }

		set_transient( $transient_key, $data, $config['cache_timeout'] );

		return $data;
	}

	public function _remote_get( $config ) {

		$url           = add_query_arg([
            'fields'       => 'id,username,media_type,media_url,thumbnail_url,permalink',
            'access_token' => $config['access_token'],
		], 'https://graph.instagram.com/me/media/' );
		$response      = wp_remote_get( $url, [ 'timeout' => 60 ]);
		$response_code = wp_remote_retrieve_response_code( $response );

		if ( 200 !== $response_code ) {
			$body = json_decode( wp_remote_retrieve_body( $response ), true );

            if ( is_array( $body ) && isset( $body['error']['message'] ) ) {
                $message = $body['error']['message'];
            } else {
                $message = esc_html__( 'Posts not found', 'onnat-extra' );
            }

			return new \WP_Error( $response_code, $message );
		}

		$result = json_decode( wp_remote_retrieve_body( $response ), true );

		if ( ! is_array( $result ) ) {
            return new \WP_Error( 'invalid-data', esc_html__( 'Invalid data', 'onnat-extra' ) );
        }

		return $result;
	}

	public function _get_response_data_from_official_api( $response ) {
		$data = [];

        if ( ! isset( $response['data'] ) ) {
            return $data;
        }

		$response_items = $response['data'];

		if ( empty( $response_items ) ) {
            return $data;
        }

		$response_items = $this->_remove_media_items( $media = 'VIDEO', $response_items );
		$sliced         = array_slice( $response_items, 0, $this->get_settings( 'count' ), true);

		foreach ( $sliced as $post ) {

			$media_url = ! empty( $post['media_url'] ) ? $post['media_url'] : '';

			if( 'CAROUSEL_ALBUM' === $post['media_type'] ) {
				$media_url = ! empty( $post['children']['data'][0]['media_url'] ) ? $post['children']['data'][0]['media_url'] : $media_url;
			}

			array_push( $data, [
				'id'        => $post['id'],
				'author'    => $post['username'],
				'image'     => $media_url,
				'permalink' => $post['permalink']
			]);
		}

		return $data;
	}

	/**
	 * Remove video items from data.
	 */
	public function _remove_media_items( $media = 'VIDEO', $data = [] ) {
		$result = [];

		foreach ( $data as $item ) {
            if ( ! empty( $item['media_type'] ) && $media === $item['media_type'] ) {
                continue;
            }

            if ( ! empty( $item['children']['data'] ) ) {
                $item['children']['data'] = $this->_remove_media_items( $media, $item['children']['data'] );

                if ( empty( $item['children']['data'] ) ) {
                    continue;
                }
            }

			$result[] = $item;
		}

		return $result;
	}

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-instagram");
		include $render;
    }
}