<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use \Elementor\Plugin as KFEl;
use \Elementor\Utils as KFUtils;
use \Elementor\Widget_Base as WB;
use \Elementor\Controls_Manager as KFCM;
use \Elementor\Repeater as KFRTR;
use \Elementor\Icons_Manager as KFIM;
use \Elementor\Group_Control_Image_Size as KFGCIS;

use \Elementor\Group_Control_Typography as KFGCT;
use \Elementor\Group_Control_Text_Shadow as KFGCTS;
use \Elementor\Group_Control_Text_Stroke as KFGCTST;

class KFW_Interactive_Image extends WB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-interactive-image';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Interactive Image', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_core_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-accordion';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'kfw', 'image', 'interactive' ];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
        return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/interactive-image';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

		wp_register_script( 'lordicon',
			KFE_CONST_URL . 'libraries/elementor/assets/js/lordicon.js',
			[],
			KFE_CONST_VERSION,
			true
		);

		wp_register_script( 'kfw-interactive-image',
			KFE_CONST_URL . 'libraries/elementor/assets/js/interactive-image' . KFE_CONST_DEBUG_SUFFIX . '.js',
			[ 'jquery', 'elementor-frontend' ],
			KFE_CONST_VERSION,
			true
		);

		wp_register_style( 'kfw-interactive-image',
			KFE_CONST_URL . 'libraries/elementor/assets/css/interactive-image' . KFE_CONST_DEBUG_SUFFIX . '.css',
			[],
			KFE_CONST_VERSION
		);

    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'kfw-interactive-image' ];
	}

	/**
	 * Get script dependencies.
	 *
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @return array Element scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'lordicon', 'kfw-interactive-image' ];
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
		$this->_register_items_section();
        $this->_register_settings_section();

		$this->_register_section_style_section();
        $this->_register_separator_style_section();
        $this->_register_title_icon_style_section();
        $this->_register_title_style_section();
		$this->_register_sub_title_style_section();
    }

    public function _register_items_section() {
		$repeater = new KFRTR();

		$this->start_controls_section( 'section_items', [
			'label' => esc_html__( 'Items', 'onnat-extra' )
		]);
            $repeater->start_controls_tabs('accordion_tabs');
                $repeater->start_controls_tab('tab_title',[
                    'label' => esc_html__( 'Title', 'onnat-extra' ),
                ]);
                    $repeater->add_control('title',[
                        'label'       => esc_html__( 'Title', 'onnat-extra' ),
                        'type'        => KFCM::TEXT,
                        'dynamic'     => [ 'active' => true ],
                        'default'     => esc_html__( 'Item Title' , 'onnat-extra' ),
                        'label_block' => true,
                    ]);
                    $repeater->add_control('sub_title',[
                        'label'       => esc_html__( 'Sub Title', 'onnat-extra' ),
                        'type'        => KFCM::TEXT,
                        'dynamic'     => [ 'active' => true, ],
                        'placeholder' => esc_html__( 'Sub Title', 'onnat-extra' ),
                        'label_block' => true,
                    ]);
                    $repeater->add_control('icons_settings_des', [
                        'type'            => KFCM::RAW_HTML,
                        'content_classes' => 'kfw-elementor-editor-notice-ctrl',
                        'raw'             => sprintf(
                            esc_html__('Icon settings reflects only if %1$s Use Title Icon option %2$s is set to %1$sYES%2$s.', 'onnat-extra'),
                            '<b> <i> ', '<i> </b>'
                        ),
                    ]);
                    $repeater->add_control('icon_type',[
                        'label'   => esc_html__( 'Icon Type', 'onnat-extra' ),
                        'type'    => KFCM::CHOOSE,
                        'options' => [
                            'blank'             => [
                                'title' => esc_html__('Blank', 'onnat-extra'),
                                'icon'  => '',
                            ],
                            'icon'              => [
                                'title' => esc_html__('Icon', 'onnat-extra'),
                                'icon'  => 'eicon-plus-circle-o',
                            ],
                            'image'             => [
                                'title' => esc_html__('Image', 'onnat-extra'),
                                'icon'  => 'eicon-image',
                            ],
                            'lordicon_media'    => [
                                'title' => esc_html__('Upload Lordicon From Media', 'onnat-extra'),
                                'icon'  => 'eicon-library-upload',
                            ],
                            'lordicon_external' => [
                                'title' => esc_html__('Upload Lordicon From Extrnal', 'onnat-extra'),
                                'icon'  => 'eicon-editor-external-link',
                            ],
                        ],
                        'default' => 'blank',
                        'toggle'  => false
                    ]);
                    $repeater->add_control( 'icon', [
                        'label'     => esc_html__( 'Icon', 'onnat-extra' ),
                        'type'      => KFCM::ICONS,
                        'default'   => [
                            'value'   => 'fas fa-heart',
                            'library' => 'fa-solid',
                        ],
                        'condition' => [
                            'icon_type' => 'icon'
                        ],
                    ]);
                    $repeater->add_control( 'image', [
                        'label'     => esc_html__( 'Image', 'onnat-extra' ),
                        'type'      => KFCM::MEDIA,
                        'dynamic'   => [ 'active' => true ],
                        'default'   => [
                            'url' => \Elementor\Utils::get_placeholder_image_src()
                        ],
                        'condition' => [
                            'icon_type' => 'image'
                        ],
                    ]);
                    $repeater->add_control('lordicon_des', [
                        'type'            => KFCM::RAW_HTML,
                        'content_classes' => 'kfw-elementor-editor-notice-ctrl',
                        'raw'             => sprintf(
                            esc_html__('You can get lordicon code from %1$s ', 'onnat-extra'),
                            sprintf('<a href="%1$s" target="_blank">%2$s</a>', esc_url( 'https://lordicon.com/' ), esc_html_x( 'Lordicon.com', 'link text for lordicon.com', 'onnat-extra' ) )
                        ),
                        'condition'       => [
                            'icon_type' => [ 'lordicon_media', 'lordicon_external' ]
                        ],
                    ]);
                    $repeater->add_control('lordicon_media', [
                        'label'      => esc_html__( 'Upload JSON File', 'onnat-extra' ),
                        'type'       => KFCM::MEDIA,
                        'media_type' => 'application/json',
                        'default'    => [
                            'url' => KFE_CONST_URL . 'libraries/elementor/assets/json/lordicon-clock-icon.json'
                        ],
                        'condition'  => [
                            'icon_type' => 'lordicon_media'
                        ]
                    ]);
                    $repeater->add_control('lordicon_external', [
                        'label'       => esc_html__( 'Upload JSON File', 'onnat-extra' ),
                        'type'        => KFCM::URL,
                        'dynamic'     => [ 'active' => true, ],
                        'placeholder' => esc_html__( 'Enter your URL', 'onnat-extra' ),
                        'options'     => [ 'url' ],
                        'default'     => [
                            'url' => KFE_CONST_URL . 'libraries/elementor/assets/json/lordicon-clock-icon.json'
                        ],
                        'condition'   => [
                            'icon_type' => 'lordicon_external'
                        ]
                    ]);
                $repeater->end_controls_tab();
                $repeater->start_controls_tab('tab_image',[
                    'label' => esc_html__( 'Image', 'onnat-extra' ),
                ]);
                    $repeater->add_control( 'main_image', [
                        'label'   => esc_html__('Main Image', 'onnat-extra'),
                        'type'    => KFCM::MEDIA,
                        'default' => [
                            'url' => KFUtils::get_placeholder_image_src(),
                            'id'  => -1
                        ],
                        'dynamic' => [
                            'active' => true,
                        ],
                    ]);
                $repeater->end_controls_tab();
            $repeater->end_controls_tabs();

            $this->add_control('items', [
                'type'    => KFCM::REPEATER,
                'fields'  => $repeater->get_controls(),
                'default' => [
                    [
						'title'  => 'Item #1',
                    ],
                    [
                        'title'  => 'Item #2',
                    ],
                    [
                        'title'  => 'Item #3',
                    ],
                    [
                        'title'  => 'Item #4',
                    ],
                    [
                        'title'  => 'Item #5',
                    ],
                ]
            ]);
		$this->end_controls_section();
    }

    public function _register_settings_section() {
		$this->start_controls_section( 'section_settings', [
			'label' => esc_html__( 'Settings', 'onnat-extra' )
		]);
            $this->add_control('type',[
                'label'   => esc_html__( 'Type', 'onnat-extra' ),
                'type'    => KFCM::CHOOSE,
                'toggle'  => false,
                'default' => 'type-1',
                'options' => [
                    'type-1' => [
                        'title' => esc_html__('Type 1', 'onnat-extra'),
                        'icon'  => 'kfw-ctrl-icon kfw-ctrl-type-1-icon'
                    ],
                    'type-2' => [
                        'title' => esc_html__('Type 2', 'onnat-extra'),
                        'icon'  => 'kfw-ctrl-icon kfw-ctrl-type-2-icon'
                    ],
                ],
                'prefix_class' => 'kfw-interactive-image-type kfw-interactive-image-',
            ]);
            $this->add_group_control( KFGCIS::get_type(), [
                'name'           => 'image_size',
                'fields_options' => [
                    'size' => [
                        'label'       => esc_html__('Main Image Resolution', 'onnat-extra'),
                        'label_block' => true,
                    ],
                ],
                'default'        => 'full',
            ]);
            $this->add_control( 'trigger', [
                'label'   => esc_html__( 'Selection On', 'onnat-extra' ),
                'type'    => KFCM::CHOOSE,
                'options' => [
                    'click'     => [
                        'title' => esc_html__('Mouse Click', 'onnat-extra'),
                        'icon'  => 'eicon-click',
                    ],
                    'mouseover'  => [
                        'title' => esc_html__('Mouse Hover', 'onnat-extra'),
                        'icon'  => 'eicon-cursor-move',
                    ],
                ],
                'default'      => 'click',
                'prefix_class' => 'kfw-interactive-image-trigger-on-',
                'toggle'       => false,
            ]);
            $this->add_control('active_item', [
                'label'   => esc_html__( 'Active Item Number', 'onnat-extra' ),
                'type'    => KFCM::NUMBER,
                'min'     => 1,
                'max'     => 20,
                'step'    => 1,
                'default' => 1,
            ]);
            $this->add_control('icon_settings_heading',[
                'label' => esc_html__( 'ICON SETTINGS', 'onnat-extra' ),
                'type'  => KFCM::HEADING,
            ]);
            $this->add_control('icon_settings_heading_after_hr', [
                'type' => KFCM::DIVIDER,
            ]);
            $this->add_control('use_title_icon',[
                'label'        => esc_html__( 'Use Title Icon?', 'onnat-extra' ),
                'type'         => KFCM::SWITCHER,
                'return_value' => 'yes',
                'default'      => 'yes'
            ]);
			$this->add_control('icon_position', [
				'label'     => esc_html__( 'Icon Position', 'onnat-extra' ),
				'type'      => KFCM::CHOOSE,
				'options'   => [
					'left'   => [
						'title' => esc_html__( 'Left', 'onnat-extra' ),
						'icon'  => 'eicon-h-align-left',
					],
					'top'    => [
						'title' => esc_html__( 'Top', 'onnat-extra' ),
						'icon'  => 'eicon-v-align-top',
					],
				],
				'default'      => 'left',
				'toggle'       => false,
				'prefix_class' => 'kfw-interactive-image-icon-position-',
				'condition'    => [
                    'use_title_icon' => 'yes'
                ]
			]);
		$this->end_controls_section();
    }

    public function _register_section_style_section() {
		$this->start_controls_section( 'section_accordion_style', [
            'label' => esc_html__( 'Sections', 'onnat-extra' ),
            'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_control('heading_0',[
                'label' => esc_html__( 'CONTENT SECTIONS', 'onnat-extra' ),
                'type'  => KFCM::HEADING,
            ]);
            $this->add_control('hr_0', [
                'type' => KFCM::DIVIDER,
            ]);
            $this->add_responsive_control( 'items_gap', [
                'label'      => esc_html__('Items Gap', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'size_units' => [ 'px' ],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 200,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-elementor-inter-img-content-item:not(:last-child)' => 'margin-bottom:{{SIZE}}{{UNIT}};',
                ],
            ]);
            $this->add_control( 'section_width', [
                'label'      => esc_html__('Section Width', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'size_units' => [ '%' ],
                'range'      => [
                    '%' => [
                        'min' => 0,
                        'max' => 80,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-inter-img-section-wrap.kfw-content-section-wrap' => 'width:{{SIZE}}%;',
                    '{{WRAPPER}} .kfw-inter-img-section-wrap.kfw-img-section-wrap' => 'width:calc(100% - {{SIZE}}%);',
                ],
            ]);
            $this->add_control('heading_1',[
                'label' => esc_html__( 'IMAGE SECTIONS', 'onnat-extra' ),
                'type'  => KFCM::HEADING,
            ]);
            $this->add_control('hr_1', [
                'type' => KFCM::DIVIDER,
            ]);
            $this->add_control( 'img_position', [
                'label'      => esc_html__('Image Position', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'size_units' => [ 'px' ],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-elementor-inter-img-image-item img' => 'top:{{SIZE}}{{UNIT}};',
                ],
            ]);
            $this->add_responsive_control( 'img_border_radius', [
                'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-img-section-wrap .kfw-elementor-inter-img-image-item img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ]
            ]);
		$this->end_controls_section();
    }

    public function _register_separator_style_section() {
        $type_condition = [
            'type' => 'type-1'
        ];

		$this->start_controls_section( 'section_separator_style', [
            'label'     => esc_html__( 'Separator', 'onnat-extra' ),
            'tab'       => KFCM::TAB_STYLE,
            'condition' => $type_condition,
		]);
        $this->start_controls_tabs( 'separator_customize_tabs', [
            'condition' => $type_condition,
        ] );
            $this->start_controls_tab( 'separator_normal_state', [
                'label' => esc_html__( 'Normal', 'onnat-extra' ),
            ]);
                $this->add_control( 'separator_width', [
                    'label'      => esc_html__('Separator Width', 'onnat-extra' ),
                    'type'       => KFCM::SLIDER,
                    'size_units' => [ 'px' ],
                    'range'      => [
                        'px' => [
                            'min' => 0,
                            'max' => 300,
                        ],
                    ],
                    'selectors'  => [
                        '{{WRAPPER}} .kfw-elementor-inter-img-content-item:before' => 'width:{{SIZE}}{{UNIT}};',
                    ],
                ]);
                $this->add_control( 'separator_color', [
                    'label'     => esc_html__('Color', 'onnat-extra' ),
                    'type'      => KFCM::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .kfw-elementor-inter-img-content-item:before' => 'background: {{VALUE}};'
                    ],
                ]);
            $this->end_controls_tab();
            $this->start_controls_tab( 'separator_active_state', [
                'label' => esc_html__( 'Active', 'onnat-extra' ),
            ]);
                $this->add_control( 'separator_active_width', [
                    'label'      => esc_html__('Separator Width', 'onnat-extra' ),
                    'type'       => KFCM::SLIDER,
                    'size_units' => [ 'px' ],
                    'range'      => [
                        'px' => [
                            'min' => 0,
                            'max' => 300,
                        ],
                    ],
                    'selectors'  => [
                        '{{WRAPPER}} .kfw-elementor-inter-img-content-item.kfw-active-item:before' => 'width:{{SIZE}}{{UNIT}};',
                    ],
                ]);
                $this->add_control( 'separator_active_color', [
                    'label'     => esc_html__('Color', 'onnat-extra' ),
                    'type'      => KFCM::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .kfw-elementor-inter-img-content-item.kfw-active-item:before' => 'background: {{VALUE}};'
                    ],
                ]);
            $this->end_controls_tab();
        $this->end_controls_tabs();

		$this->end_controls_section();
    }

    public function _register_title_icon_style_section() {
		$this->start_controls_section( 'section_icon_style', [
			'label' => esc_html__( 'Title Icon', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_responsive_control( 'icon_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-inter-img-title-icon-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'icon_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-inter-img-title-icon-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);

            /**
             * Style: Icon Type
             */
                $this->add_control('icon_type_style_before_hr', [
                    'type' => KFCM::DIVIDER,
                ]);
                $this->add_control('icon_type_style_heading',[
					'label' => esc_html__( 'Icon Type Style', 'onnat-extra' ),
					'type'  => KFCM::HEADING,
				]);
				$this->add_control('icon_type_style_after_hr', [
					'type' => KFCM::DIVIDER,
				]);
                $this->add_responsive_control( 'icon_type_size', [
                    'label'      => esc_html__('Size', 'onnat-extra' ),
                    'type'       => KFCM::SLIDER,
                    'size_units' => [ 'px' ],
                    'range'      => [
                        'px' => [
                            'min' => 0,
                            'max' => 200,
                        ],
                    ],
                    'selectors'  => [
                        '{{WRAPPER}} .kfw-inter-img-title-icon-wrap.kfw-inter-img-title-icon-type .kfw-elementor-icon-wrap svg, {{WRAPPER}} .kfw-inter-img-title-icon-wrap.kfw-inter-img-title-icon-type .kfw-elementor-icon-wrap i' => 'font-size: {{SIZE}}{{UNIT}};'
                    ],
                ]);
				$this->add_control( 'icon_color_settings', [
                    'label'        => esc_html__('Colors', 'onnat-extra'),
					'type'         => KFCM::POPOVER_TOGGLE,
					'return_value' => 'yes',
				]);
                $this->start_popover();
                    $this->add_control( 'icon_type_color', [
                        'label'     => esc_html__('Normal Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => [
                            '{{WRAPPER}} .kfw-inter-img-title-icon-wrap.kfw-inter-img-title-icon-type .kfw-elementor-icon-wrap svg' => 'fill:{{VALUE}};',
                            '{{WRAPPER}} .kfw-inter-img-title-icon-wrap.kfw-inter-img-title-icon-type .kfw-elementor-icon-wrap i' => 'color:{{VALUE}};',
                        ],
                    ]);
                    $this->add_control( 'icon_type_hover_color', [
                        'label'     => esc_html__('Hover Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => [
                            '{{WRAPPER}} .kfw-elementor-inter-img-content-item:hover .kfw-inter-img-title-icon-wrap.kfw-inter-img-title-icon-type .kfw-elementor-icon-wrap svg' => 'fill:{{VALUE}};',
                            '{{WRAPPER}} .kfw-elementor-inter-img-content-item:hover .kfw-inter-img-title-icon-wrap.kfw-inter-img-title-icon-type .kfw-elementor-icon-wrap i' => 'color:{{VALUE}};',
                        ],
                    ]);
                    $this->add_control( 'icon_type_active_color', [
                        'label'     => esc_html__('Active Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => [
                            '{{WRAPPER}} .kfw-elementor-inter-img-content-item.kfw-active-item .kfw-inter-img-title-icon-wrap.kfw-inter-img-title-icon-type .kfw-elementor-icon-wrap svg' => 'fill:{{VALUE}};',
                            '{{WRAPPER}} .kfw-elementor-inter-img-content-item.kfw-active-item .kfw-inter-img-title-icon-wrap.kfw-inter-img-title-icon-type .kfw-elementor-icon-wrap i' => 'color:{{VALUE}};',
                        ],
                    ]);
                $this->end_popover();

            /**
             * Style: Image Type
             */
                $this->add_control('img_type_style_before_hr', [
                    'type' => KFCM::DIVIDER,
                ]);
                $this->add_control('img_type_style_heading',[
                    'label' => esc_html__( 'Image Type Style', 'onnat-extra' ),
                    'type'  => KFCM::HEADING,
                ]);
                $this->add_control('img_type_style_after_hr', [
                    'type' => KFCM::DIVIDER,
                ]);
                $this->add_responsive_control( 'img_type_size', [
                    'label'      => esc_html__('Size', 'onnat-extra' ),
                    'type'       => KFCM::SLIDER,
                    'size_units' => ['px'],
                    'range'      => [
                        'px' => [
                            'min' => 5,
                            'max' => 200,
                        ],
                    ],
                    'selectors'  => [
                        '{{WRAPPER}} .kfw-inter-img-title-icon-wrap.kfw-inter-img-title-image-type .kfw-elementor-icon-wrap img' => 'max-width:{{SIZE}}px;'
                    ],
                ]);

            /**
             * Style: Lordicon Type
             */
                $this->add_control('lordicon_type_style_before_hr', [
                    'type' => KFCM::DIVIDER,
                ]);
                $this->add_control('lordicon_type_style_heading',[
                    'label' => esc_html__( 'LordIcon Type Style', 'onnat-extra' ),
                    'type'  => KFCM::HEADING,
                ]);
                $this->add_control('lordicon_type_style_after_hr', [
                    'type' => KFCM::DIVIDER,
                ]);
                $this->add_responsive_control( 'lordicon_type_size', [
                    'label'      => esc_html__('Size', 'onnat-extra' ),
                    'type'       => KFCM::SLIDER,
                    'size_units' => [ 'px' ],
                    'range'      => [
                        'px' => [
                            'min' => 1,
                            'max' => 300,
                        ],
                    ],
                    'selectors'  => [
                        '
                        {{WRAPPER}} .kfw-inter-img-title-icon-wrap.kfw-inter-img-title-lordicon-type .kfw-elementor-icon-wrap,
                        {{WRAPPER}} .kfw-inter-img-title-icon-wrap.kfw-inter-img-title-lordicon-type .kfw-elementor-lordicon' => 'width:{{SIZE}}{{UNIT}};height:{{SIZE}}{{UNIT}};'
                    ]
                ]);
				$this->add_control( 'lordicon_color_settings', [
                    'label'        => esc_html__('Colors', 'onnat-extra'),
					'type'         => KFCM::POPOVER_TOGGLE,
					'return_value' => 'yes',
				]);
                $this->start_popover();
                    $this->add_control('lordicon_primary_color',[
                        'label'     => __( 'Primary Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'default'   => '#121331',
                    ]);
                    $this->add_control('lordicon_secondary_color',[
                        'label'   => __( 'Secondary Color', 'onnat-extra' ),
                        'type'    => KFCM::COLOR,
                        'default' => '#121331',
                    ]);
                    $this->add_control('lordicon_tertiary_color',[
                        'label'   => __( 'Tertiary Color', 'onnat-extra' ),
                        'type'    => KFCM::COLOR,
                        'default' => '#0816A8',
                    ]);
                $this->end_popover();
		$this->end_controls_section();
    }

    public function _register_title_style_section() {
		$this->start_controls_section( 'section_title_style', [
			'label' => esc_html__( 'Title', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_group_control(KFGCT::get_type(), [
                'name'     => 'title_typo',
                'label'    => esc_html__( 'Typography', 'onnat-extra' ),
                'selector' => '{{WRAPPER}} .kfw-inter-img-title',
            ]);
            $this->add_responsive_control( 'title_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-inter-img-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'title_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-inter-img-title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->start_controls_tabs( 'title_customize_tabs' );
                $this->start_controls_tab( 'title_normal_state', [
                    'label' => esc_html__( 'Normal', 'onnat-extra' ),
                ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'     => 'title_txt_stroke',
                        'selector' => '{{WRAPPER}} .kfw-inter-img-title',
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'     => 'title_txt_shadow',
                        'selector' => '{{WRAPPER}} .kfw-inter-img-title',
                    ]);
                $this->end_controls_tab();
                $this->start_controls_tab( 'title_hover_state', [
                    'label' => esc_html__( 'Hover', 'onnat-extra' ),
                ]);
                    $this->add_control( 'title_hover_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kfw-inter-img-title:before' => '-webkit-text-fill-color: {{VALUE}};' ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'     => 'title_hover_txt_shadow',
                        'selector' => '{{WRAPPER}} .kfw-elementor-inter-img-content-item:hover .kfw-inter-img-title',
                    ]);
                $this->end_controls_tab();
                $this->start_controls_tab( 'title_active_state', [
                    'label' => esc_html__( 'Active', 'onnat-extra' ),
                ]);
                    $this->add_control( 'title_active_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}}.kfw-interactive-image-trigger-on-click .kfw-elementor-inter-img-content-item.kfw-active-item .kfw-inter-img-title' => 'color: {{VALUE}};' ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'     => 'title_active_txt_shadow',
                        'selector' => '{{WRAPPER}}.kfw-interactive-image-trigger-on-click .kfw-elementor-inter-img-content-item.kfw-active-item .kfw-inter-img-title',
                    ]);
                $this->end_controls_tab();
            $this->end_controls_tabs();
		$this->end_controls_section();
    }

    public function _register_sub_title_style_section() {
		$this->start_controls_section( 'section_sub_title_style', [
			'label' => esc_html__( 'Sub Title', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_group_control(KFGCT::get_type(), [
                'name'     => 'sub_title_typo',
                'label'    => esc_html__( 'Typography', 'onnat-extra' ),
                'selector' => '{{WRAPPER}} .kfw-inter-img-sub-title',
            ]);
            $this->add_responsive_control( 'sub_title_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-inter-img-sub-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'sub_title_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-inter-img-sub-title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->start_controls_tabs( 'sub_title_customize_tabs' );
                $this->start_controls_tab( 'sub_title_normal_state', [
                    'label' => esc_html__( 'Normal', 'onnat-extra' ),
                ]);
                    $this->add_control( 'sub_title_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kfw-inter-img-sub-title' => 'color: {{VALUE}};' ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'     => 'sub_title_txt_shadow',
                        'selector' => '{{WRAPPER}} .kfw-inter-img-sub-title',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'     => 'sub_title_txt_stroke',
                        'selector' => '{{WRAPPER}} .kfw-inter-img-sub-title',
                    ]);
                $this->end_controls_tab();
                $this->start_controls_tab( 'sub_title_hover_state', [
                    'label' => esc_html__( 'Hover', 'onnat-extra' ),
                ]);
                    $this->add_control( 'sub_title_hover_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kfw-elementor-inter-img-content-item:hover .kfw-inter-img-sub-title' => 'color: {{VALUE}};' ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'     => 'sub_title_hover_txt_shadow',
                        'selector' => '{{WRAPPER}} .kfw-elementor-inter-img-content-item:hover .kfw-inter-img-sub-title',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'     => 'sub_title_hover_txt_stroke',
                        'selector' => '{{WRAPPER}} .kfw-elementor-inter-img-content-item:hover .kfw-inter-img-sub-title',
                    ]);
                $this->end_controls_tab();
                $this->start_controls_tab( 'sub_title_active_state', [
                    'label' => esc_html__( 'Active', 'onnat-extra' ),
                ]);
                    $this->add_control( 'sub_title_active_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kfw-elementor-inter-img-content-item.kfw-active-item .kfw-inter-img-sub-title' => 'color: {{VALUE}};' ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'     => 'sub_title_active_txt_shadow',
                        'selector' => '{{WRAPPER}} .kfw-elementor-inter-img-content-item.kfw-active-item .kfw-inter-img-sub-title',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'     => 'sub_title_active_txt_stroke',
                        'selector' => '{{WRAPPER}} .kfw-elementor-inter-img-content-item.kfw-active-item .kfw-inter-img-sub-title',
                    ]);
                $this->end_controls_tab();
            $this->end_controls_tabs();
		$this->end_controls_section();
    }

	/**
	 * Get attachment image src.
	 */
	public static function _get_image_src( $attachment_id, $image_size_key, array $settings ) {
        return KFGCIS::get_attachment_image_src( $attachment_id, $image_size_key, $settings );
	}

	public function _render_icon( $icon , $echo = true ) {
		$icon = array_filter( $icon );

		if( !empty( $icon ) ) {
            echo '<div class="kfw-inter-img-title-icon-wrap kfw-inter-img-title-icon-type">';
                echo '<span class="kfw-elementor-icon-wrap">';
                    KFIM::render_icon( $icon, [ 'aria-hidden' => 'true' ] );
                echo '</span>';
            echo '</div>';
		}
    }

    public function _render_img( $settings = [] ) {
        if( !empty( $settings['url'] ) ) {
            $image_src = $settings['url'];

			$image_title = $this->_get_image_title( $settings );
			$image_title = empty( $image_title ) ? $this->get_title() : $image_title;

			$image_alt = $this->_get_image_alt( $settings );
			$image_alt = empty( $image_alt ) ? $image_title : $image_title;

            echo '<div class="kfw-inter-img-title-icon-wrap kfw-inter-img-title-image-type">';
                echo '<span class="kfw-elementor-icon-wrap">';
                    echo '<img alt="'. esc_attr( $image_alt ) .'" title="'. esc_attr( $image_title ) .'" src="'. esc_url( $image_src ) .'"/>';
                echo '</span>';
            echo '</div>';
        }
    }

	public function _get_image_title( $attachment ) {
		if ( empty( $attachment['id'] ) ) {
			return '';
		}

		return get_the_title( $attachment['id'] );
	}

	public function _get_image_alt( $instance ) {
		if ( empty( $instance['id'] ) ) {
			// For `Insert From URL` images.
			return isset( $instance['alt'] ) ? trim( strip_tags( $instance['alt'] ) ) : '';
		}

		$attachment_id = $instance['id'];
		if ( ! $attachment_id ) {
			return '';
		}

		$attachment = get_post( $attachment_id );
		if ( ! $attachment ) {
			return '';
		}

		$alt = get_post_meta( $attachment_id, '_wp_attachment_image_alt', true );
		if ( ! $alt ) {
			$alt = $attachment->post_excerpt;
			if ( ! $alt ) {
				$alt = $attachment->post_title;
			}
		}
		return trim( strip_tags( $alt ) );
	}

	/**
	 * Generate setting json
	 */
	protected function _generate_setting_json( array $settings ) {
		extract( $settings );

		$options = [
			'id'             => 'kfw-ele-widget-interactive-image-'.esc_attr( $this->get_id() ),
			'trigger'        => $trigger,
			'active_item'    => true,
			'active_item_no' => !empty( $active_item ) ? $active_item : 1,
			'total_items'    => count( $items ),
		];

		return wp_json_encode( $options );
	}

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-interactive-image");
		include $render;
    }
}