<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use \Elementor\Widget_Base as KFWB;
use \Elementor\Controls_Manager as KFCM;
use \Elementor\Icons_Manager as KFIM;

use \Elementor\Repeater as KFRTR;
use \Elementor\Group_Control_Typography as KFGCT;
use \Elementor\Group_Control_Border as KFGCBR;
use \Elementor\Group_Control_Background as KFGCBG;
use \Elementor\Group_Control_Box_Shadow as KFGCBS;
use \Elementor\Group_Control_Text_Shadow as KFGCTS;

class KFW_Pricing_Table_Basic extends KFWB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-pricing-table-basic';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Pricing Table : Basic', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_core_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-price-table kfw-pricing-table-basic';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'kfw', 'kin', 'pricing', 'table', 'basic'];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
		return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/pricing-table-basic';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

		wp_register_script( 'kfw-pricing-table-basic',
			KFE_CONST_URL . 'libraries/elementor/assets/js/pricing-table-basic' . KFE_CONST_DEBUG_SUFFIX . '.js',
			[ 'jquery', 'elementor-frontend' ],
			KFE_CONST_VERSION,
			true
		);

		wp_register_style( 'kfw-pricing-table-basic',
			KFE_CONST_URL . 'libraries/elementor/assets/css/pricing-table-basic' . KFE_CONST_DEBUG_SUFFIX . '.css',
			[],
			KFE_CONST_VERSION
		);
    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'kfw-pricing-table-basic' ];
	}

	/**
	 * Get script dependencies.
	 *
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @return array Element scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'jquery-numerator', 'kfw-pricing-table-basic' ];
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
		$this->_register_titles_section();
		$this->_register_pricing_section();
		$this->_register_features_section();
		$this->_register_ribbon_section();
		$this->_register_button_section();

		$this->_register_table_style_section();
		$this->_register_title_style_section();
		$this->_register_pricing_style_section();
		$this->_register_features_style_section();
		$this->_register_ribbon_style_section();
		$this->_register_button_style_section();
	}

	public function _register_titles_section() {
		$this->start_controls_section( 'section_title', [
			'label' => esc_html__( 'Title & Sub Title', 'onnat-extra' )
		]);
			$this->add_control( 'title', [
				'label'       => esc_html__( 'Title', 'onnat-extra' ),
				'type'        => KFCM::TEXT,
				'label_block' => true,
				'default'     => esc_html__( 'Title', 'onnat-extra' ),
			]);
			$this->add_control( 'show_sub_title', [
				'label'        => esc_html__('Show Sub Title','onnat-extra' ),
				'type'         => KFCM::SWITCHER,
				'label_on'     => esc_html__( 'On', 'onnat-extra' ),
				'label_off'    => esc_html__( 'Off', 'onnat-extra' ),
				'return_value' => 'yes',
				'default'      => 'no'
			]);
			$this->add_control( 'sub_title', [
				'label'       => esc_html__( 'Sub Title', 'onnat-extra' ),
				'type'        => KFCM::TEXT,
				'label_block' => true,
				'default'     => esc_html__( 'Sub Title', 'onnat-extra' ),
				'condition'   => [ 'show_sub_title' => 'yes' ]
			]);
		$this->end_controls_section();
	}

	public function _register_pricing_section() {
		$this->start_controls_section( 'section_pricing', [
			'label' => esc_html__( 'Pricing', 'onnat-extra' )
		]);
			$this->add_control( 'currency_heading', [
				'label' => esc_html__( 'Currency', 'onnat-extra' ),
				'type'  => KFCM::HEADING,
			]);
			$this->add_control( 'currency_symbol', [
				'label'   => esc_html__( 'Symbol', 'onnat-extra' ),
				'type'    => KFCM::SELECT,
				'options' => [
					''             => esc_html__( 'None', 'onnat-extra' ),
					'dollar'       => '&#36; ' . _x( 'Dollar', 'Currency Symbol', 'onnat-extra' ),
					'euro'         => '&#128; ' . _x( 'Euro', 'Currency Symbol', 'onnat-extra' ),
					'baht'         => '&#3647; ' . _x( 'Baht', 'Currency Symbol', 'onnat-extra' ),
					'franc'        => '&#8355; ' . _x( 'Franc', 'Currency Symbol', 'onnat-extra' ),
					'guilder'      => '&fnof; ' . _x( 'Guilder', 'Currency Symbol', 'onnat-extra' ),
					'krona'        => 'kr ' . _x( 'Krona', 'Currency Symbol', 'onnat-extra' ),
					'lira'         => '&#8356; ' . _x( 'Lira', 'Currency Symbol', 'onnat-extra' ),
					'indian_rupee' => '&#8377; ' . _x( 'Rupee (Indian)', 'Currency Symbol', 'onnat-extra' ),
					'peseta'       => '&#8359 ' . _x( 'Peseta', 'Currency Symbol', 'onnat-extra' ),
					'peso'         => '&#8369; ' . _x( 'Peso', 'Currency Symbol', 'onnat-extra' ),
					'pound'        => '&#163; ' . _x( 'Pound Sterling', 'Currency Symbol', 'onnat-extra' ),
					'real'         => 'R$ ' . _x( 'Real', 'Currency Symbol', 'onnat-extra' ),
					'ruble'        => '&#8381; ' . _x( 'Ruble', 'Currency Symbol', 'onnat-extra' ),
					'rupee'        => '&#8360; ' . _x( 'Rupee', 'Currency Symbol', 'onnat-extra' ),
					'shekel'       => '&#8362; ' . _x( 'Shekel', 'Currency Symbol', 'onnat-extra' ),
					'yen'          => '&#165; ' . _x( 'Yen/Yuan', 'Currency Symbol', 'onnat-extra' ),
					'won'          => '&#8361; ' . _x( 'Won', 'Currency Symbol', 'onnat-extra' ),
					'custom'       => esc_html__( 'Custom', 'onnat-extra' ),
				],
				'default' => 'dollar',
			]);
			$this->add_control( 'currency_symbol_custom', [
				'label'     => esc_html__( 'Symbol', 'onnat-extra' ),
				'type'      => KFCM::TEXT,
				'condition' => [ 'currency_symbol' => 'custom' ]
			]);
			$this->add_control( 'currency_symbol_format', [
				'label'   => esc_html__( 'Format', 'onnat-extra' ),
				'type'    => KFCM::CHOOSE,
				'options' => [
					'sub'  => [
						'title' =>  esc_html__( 'Subscript', 'onnat-extra' ),
						'icon'  => 'eicon-v-align-bottom'
					],
					'span' => [
						'title' =>  esc_html__( 'Unset', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-justify'
					],
					'sup'  => [
						'title' =>  esc_html__( 'Superscript', 'onnat-extra' ),
						'icon'  => 'eicon-v-align-top'
					],
				],
				'default' => 'sup',
				'toggle'  => false,
			]);
			$this->add_control( 'price', [
				'label'     => esc_html__( 'Price', 'onnat-extra' ),
				'type'      => KFCM::TEXT,
				'default'   => '69.99',
				'separator' => 'before'
			]);
			$this->add_control( 'price_format', [
				'label'   => esc_html__( 'Fractional Format', 'onnat-extra' ),
				'type'    => KFCM::CHOOSE,
				'options' => [
					'sub'  => [
						'title' =>  esc_html__( 'Subscript', 'onnat-extra' ),
						'icon'  => 'eicon-v-align-bottom'
					],
					'span' => [
						'title' =>  esc_html__( 'Unset', 'onnat-extra' ),
						'icon'  => ' eicon-text-align-justify'
					],
					'sup'  => [
						'title' =>  esc_html__( 'Superscript', 'onnat-extra' ),
						'icon'  => 'eicon-v-align-top'
					],
				],
				'default' => 'sup',
				'toggle'  => false,
			]);
			$this->add_control( 'price_format_help', [
				'type'            => KFCM::RAW_HTML,
				'raw'             => esc_html__( 'This will add a subscript or superscript design to the fractional portion of the price.', 'onnat-extra' ),
				'content_classes' => 'elementor-control-field-description kfw-elementor-editor-ctrl-remove-margin-top',
				'separator'       => 'none'
			]);
			$this->add_control( 'period', [
				'label'   => esc_html__( 'Period', 'onnat-extra' ),
				'type'    => KFCM::TEXT,
				'default' => '/mo',
			]);
		$this->end_controls_section();
	}

	public function _register_features_section() {
		$repeater = new KFRTR();

		$this->start_controls_section( 'section_features', [
			'label' => esc_html__( 'Features', 'onnat-extra' )
		]);
			$repeater->add_control( 'label', [
				'label'   => esc_html__( 'Label', 'onnat-extra' ),
				'type'    => KFCM::TEXT,
				'default' => esc_html__( 'List Item', 'onnat-extra' ),
			]);
			$repeater->add_control( 'bold', [
				'label'   => esc_html__( 'Bold', 'onnat-extra' ),
				'type'    => KFCM::SWITCHER,
				'default' => 'no',
			]);
			$repeater->add_control( 'icon', [
				'label'       => esc_html__( 'Icon', 'onnat-extra' ),
				'type'        => KFCM::ICONS,
				'skin'        => 'inline',
				'label_block' => false,
				'default'     => [
					'value'   => 'fa fa-check',
					'library' => 'fa-solid'
				]
			]);
			$repeater->add_control( 'icon_color', [
				'label'     => esc_html__('Icon Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} {{CURRENT_ITEM}} .kfw-ele-pricing-table-features-list-item-label i'   => 'color: {{VALUE}};',
					'{{WRAPPER}} {{CURRENT_ITEM}} .kfw-ele-pricing-table-features-list-item-label svg' => 'fill: {{VALUE}};'
				],
				'condition' => [ 'icon[value]!' => '' ]
			]);
			$repeater->add_control( 'value', [
				'label' => esc_html__( 'Value', 'onnat-extra' ),
				'type'  => KFCM::TEXT,
			]);
			$repeater->add_control( 'status', [
				'label'   => esc_html__( 'Active status', 'onnat-extra' ),
				'type'    => KFCM::SWITCHER,
				'default' => 'no',
			]);

			$this->add_control( 'features', [
				'type'        => KFCM::REPEATER,
				'fields'      => $repeater->get_controls(),
				'default'     => [
					[
						'label'  => esc_html__( 'List Item #1', 'onnat-extra' ),
						'bold'   => 'no',
						'icon'   => [ 'value' => 'fa fa-check', 'library' => 'fa-brands' ],
						'value'  => '01',
						'status' => 'yes'
					],
					[
						'label'  => esc_html__( 'List Item #2', 'onnat-extra' ),
						'bold'   => 'no',
						'icon'   => [ 'value' => 'fa fa-check', 'library' => 'fa-brands' ],
						'value'  => '02',
						'status' => 'yes'
					],
					[
						'label'  => esc_html__( 'List Item #3', 'onnat-extra' ),
						'bold'   => 'no',
						'icon'   => [ 'value' => 'fas fa-times', 'library' => 'fa-brands' ],
						'value'  => '03',
						'status' => 'no'
					],
					[
						'label'  => esc_html__( 'List Item #4', 'onnat-extra' ),
						'bold'   => 'no',
						'icon'   => [ 'value' => 'fa fa-check', 'library' => 'fa-brands' ],
						'value'  => '04',
						'status' => 'yes'
					],
				],
				'title_field' => '{{{ elementor.helpers.renderIcon( this, icon, {}, "i", "panel" ) || \'<i class="{{ icon }}" aria-hidden="true"></i>\' }}} {{{ label }}}',
			]);

		$this->end_controls_section();
	}

    public function _register_ribbon_section() {
		$this->start_controls_section( 'section_ribbon', [
			'label' => esc_html__( 'Ribbon', 'onnat-extra' ),
		]);
			$this->add_control( 'show_ribbon', [
				'label'        => esc_html__('Show Ribbon','onnat-extra' ),
				'type'         => KFCM::SWITCHER,
				'label_on'     => esc_html__( 'On', 'onnat-extra' ),
				'label_off'    => esc_html__( 'Off', 'onnat-extra' ),
				'return_value' => 'yes',
				'default'      => 'yes'
			]);
			$this->add_control( 'ribbon_text', [
				'label'     => esc_html__( 'Text', 'onnat-extra' ),
				'type'      => KFCM::TEXT,
				'default'   => esc_html__( 'New', 'onnat-extra' ),
				'condition' => [ 'show_ribbon' => 'yes' ]
			]);
			$this->add_control( 'ribbon_style', [
				'label'     => esc_html__( 'Style', 'onnat-extra' ),
				'type'      => KFCM::SELECT,
				'options'   => [
					'corner-1' => esc_html__( 'Corner 1', 'onnat-extra' ),
					'corner-2' => esc_html__( 'Corner 2', 'onnat-extra' ),
					'flag-1'   => esc_html__( 'Flag 1', 'onnat-extra' ),
					'flag-2'   => esc_html__( 'Flag 2', 'onnat-extra' ),
				],
				'condition' => [ 'show_ribbon' => 'yes' ],
				'default'   => 'corner-2',
			]);
			$this->add_control( 'ribbon_alignment', [
				'label'       => esc_html__( 'Position', 'onnat-extra' ),
				'type'        => KFCM::CHOOSE,
				'label_block' => false,
				'toggle'      => false,
				'options'     => [
					'left' => [
						'title' => esc_html__( 'Left', 'onnat-extra' ),
						'icon'  => 'fa fa-align-left',
					],
					'right' => [
						'title' => esc_html__( 'Right', 'onnat-extra' ),
						'icon'  => 'fa fa-align-right',
					],
				],
				'default'     => 'right',
				'condition'   => [ 'show_ribbon' => 'yes' ]
			]);
		$this->end_controls_section();
    }

    public function _register_button_section() {

		$this->start_controls_section( 'section_button', [
			'label' => esc_html__( 'Button', 'onnat-extra' )
		]);
			$this->add_control( 'button_style', [
				'label'     => esc_html__( 'Style', 'onnat-extra' ),
				'type'      => KFCM::SELECT,
				'options'   => [
					'default' => esc_html__( 'Default', 'onnat-extra' ),
					'fill'    => esc_html__( 'Fill', 'onnat-extra' ),
					'full'    => esc_html__( 'Full', 'onnat-extra' ),
				],
				'default'   => 'default',
			]);
			$this->add_control( 'button_align', [
				'label'     => esc_html__( 'Alignment', 'onnat-extra' ),
				'type'      => KFCM::CHOOSE,
				'options'   => [
					'left'   => [
						'title' => esc_html__('Left', 'onnat-extra'),
						'icon'  => 'eicon-text-align-left',
					],
					'center' => [
						'title' => esc_html__('Center', 'onnat-extra'),
						'icon'  => 'eicon-text-align-center',
					],
					'right'  => [
						'title' => esc_html__('Right', 'onnat-extra'),
						'icon'  => 'eicon-text-align-right',
					],
				],
				'default'   => 'center',
				'condition' => [ 'button_style' => 'default' ],
			]);
			$this->add_control( 'button_text', [
				'label'   => esc_html__( 'Text', 'onnat-extra' ),
				'type'    => KFCM::TEXT,
				'default' => esc_html__( 'Button', 'onnat-extra' ),
			]);
			$this->add_control( 'button_link', [
				'label'       => esc_html__( 'Link', 'onnat-extra' ),
				'type'        => KFCM::URL,
				'placeholder' => 'https://your-link.com',
				'default'     => [ 'url' => '#' ]
			]);
			$this->add_control( 'button_icon', [
				'label'       => esc_html__( 'Icon', 'onnat-extra' ),
				'type'        => KFCM::ICONS,
				'skin'        => 'inline',
				'label_block' => false,
				'default'     => [
					'value'   => 'fa fa-check',
					'library' => 'fa-solid'
				]
			]);
			$this->add_control( 'button_icon_pos', [
				'label'     => esc_html__( 'Icon Position', 'onnat-extra' ),
				'type'      => KFCM::CHOOSE,
				'options'   => [
					'left'   => [
						'title' => esc_html__('Left', 'onnat-extra'),
						'icon'  => 'eicon-text-align-left',
					],
					'right'  => [
						'title' => esc_html__('Right', 'onnat-extra'),
						'icon'  => 'eicon-text-align-right',
					],
				],
				'default'   => 'left',
				'condition' => [
					'button_icon[value]!' => ''
				],
			]);
		$this->end_controls_section();
    }

	public function _register_table_style_section() {
		$this->start_controls_section( 'section_table_style', [
			'label' => esc_html__( 'Table', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_group_control( KFGCBG::get_type(), [
				'name'     => 'table_background',
				'label'    => esc_html__('Background', 'onnat-extra' ),
				'types'    => [ 'classic', 'gradient' ],
				'exclude'  => [ 'image' ], // to remove background image option in classic mode
				'selector' => '{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper'
			]);
			$this->add_responsive_control( 'table_padding', [
				'label'      => esc_html__( 'Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
			]);
			$this->add_responsive_control( 'table_margin', [
				'label'      => esc_html__( 'Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
			]);
			$this->add_group_control( KFGCBR::get_type(), [
				'name'           => 'table_border',
				'fields_options' => [
					'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
					'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
				],
				'selector'       => '{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper'
			]);
			$this->add_responsive_control( 'table_border_radius', [
				'label'      => esc_html__('Border Radius', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				]
			]);
			$this->add_group_control( KFGCBS::get_type(), [
				'name'     => 'table_box_shadow',
				'selector' => '{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper'
			]);
			$this->add_control( 'add_table_animation_hover', [
				'label'        => esc_html__('Hover Effect','onnat-extra' ),
				'type'         => KFCM::SWITCHER,
				'label_on'     => esc_html__( 'On', 'onnat-extra' ),
				'label_off'    => esc_html__( 'Off', 'onnat-extra' ),
				'return_value' => 'yes',
				'default'      => 'yes'
			]);
			$this->add_control( 'table_animation_hover', [
				'label'     => esc_html__( 'Animation', 'onnat-extra' ),
				'type'      => KFCM::HOVER_ANIMATION,
				'default'   => 'grow',
				'condition' => [ 'add_table_animation_hover' => 'yes', ],
			]);
		$this->end_controls_section();
    }

    public function _register_title_style_section() {
		$this->start_controls_section( 'section_title_style', [
			'label' => esc_html__( 'Title & Sub Title', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_control( 'title_heading', [
				'label' => esc_html__( 'Title', 'onnat-extra' ),
				'type'  => KFCM::HEADING,
			]);
			$this->add_control( 'title_color', [
				'label'     => esc_html__('Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-title' => 'color: {{VALUE}};'
				]
			]);
			$this->add_group_control( KFGCT::get_type(), [
				'name'        => 'title_typo',
				'description' => esc_html__( 'Typography', 'onnat-extra' ),
				'selector'    => '{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-title'
			]);
			$this->add_group_control( KFGCTS::get_type(), [
				'name'     => 'title_text_shadow',
				'selector' => '{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-title'
			]);
			$this->add_responsive_control( 'title_margin', [
				'label'      => esc_html__( 'Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
			]);
			$this->add_control( 'sub_title_heading', [
				'label'     => esc_html__( 'Sub Title', 'onnat-extra' ),
				'type'      => KFCM::HEADING,
				'condition' => [ 'show_sub_title' => 'yes' ],
				'separator' => 'before',
			]);
			$this->add_control( 'sub_title_color', [
				'label'     => esc_html__('Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-sub-title' => 'color: {{VALUE}};'
				],
				'condition' => [ 'show_sub_title' => 'yes' ]
			]);
			$this->add_group_control( KFGCT::get_type(), [
				'name'        => 'sub_title_typo',
				'description' => esc_html__( 'Typography', 'onnat-extra' ),
				'selector'    => '{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-sub-title',
				'condition'   => [ 'show_sub_title' => 'yes' ]
			]);
			$this->add_group_control( KFGCTS::get_type(), [
				'name'      => 'sub_title_text_shadow',
				'selector'  => '{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-sub-title',
				'condition' => [ 'show_sub_title' => 'yes' ]
			]);
			$this->add_responsive_control( 'sub_title_margin', [
				'label'      => esc_html__( 'Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-sub-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
				'condition'  => [ 'show_sub_title' => 'yes' ],
			]);
		$this->end_controls_section();
    }

    public function _register_pricing_style_section() {
		$this->start_controls_section( 'section_pricing_style', [
			'label' => esc_html__( 'Pricing', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_group_control( KFGCT::get_type(), [
				'name'        => 'pricing_typo',
				'description' => esc_html__( 'Typography', 'onnat-extra' ),
				'selector'    => '{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-currency',
			]);
			$this->add_control( 'pricing_color', [
				'label'     => esc_html__('Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => ['{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-currency' => 'color: {{VALUE}};' ],
			]);
			$this->add_control( 'pricing_circle_heading', [
				'label'     => esc_html__( 'Circle', 'onnat-extra' ),
				'type'      => KFCM::HEADING,
				'separator' => 'before',
			]);
			$this->add_responsive_control( 'pricing_bg_size', [
				'label'     => esc_html__( 'Size', 'onnat-extra' ),
				'type'      => KFCM::SLIDER,
				'range'     => [
					'px' => [ 'min' => 100, 'max' => 350 ],
				],
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-currency' => 'width:{{SIZE}}{{UNIT}}; height:{{SIZE}}{{UNIT}}'
				]
			]);
			$this->add_group_control( KFGCBG::get_type(), [
				'name'     => 'pricing_background',
				'label'    => esc_html__('Background', 'onnat-extra' ),
				'types'    => [ 'classic', 'gradient' ],
				'exclude'  => [ 'image' ], // to remove background image option in classic mode
				'selector' => '{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-currency'
			]);
			$this->add_responsive_control( 'pricing_margin', [
				'label'      => esc_html__( 'Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-currency' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
			]);
			$this->add_group_control( KFGCBR::get_type(), [
				'name'           => 'pricing_circle_border',
				'fields_options' => [
					'width'  => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
					'color'  => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
				],
				'selector'       => '{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-currency'
			]);
			$this->add_responsive_control( 'pricing_circle_border_radius', [
				'label'      => esc_html__('Border Radius', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-currency' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				]
			]);
		$this->end_controls_section();
    }

    public function _register_features_style_section() {
		$this->start_controls_section( 'section_features_style', [
			'label' => esc_html__( 'Features', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_control( 'feature_item_icon_heading', [
				'label' => esc_html__( 'Icon', 'onnat-extra' ),
				'type'  => KFCM::HEADING,
			]);
			$this->add_responsive_control( 'item_icon_size', [
				'label'     => esc_html__( 'Size', 'onnat-extra' ),
				'type'      => KFCM::SLIDER,
				'range'     => [
					'px' => [ 'min' => 10, 'max' => 100 ],
				],
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-features-list .kfw-elementor-icon-wrapper' => 'font-size:{{SIZE}}{{UNIT}};',
				]
			]);
			$this->add_control( 'feature_item_heading', [
				'label'     => esc_html__( 'Item', 'onnat-extra' ),
				'type'      => KFCM::HEADING,
				'separator' => 'before',
			]);
			$this->add_group_control( KFGCBG::get_type(), [
				'name'     => 'item_background',
				'label'    => esc_html__('Background', 'onnat-extra' ),
				'types'    => [ 'classic', 'gradient' ],
				'exclude'  => [ 'image' ], // to remove background image option in classic mode
				'selector' => '{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-features-list'
			]);
			$this->add_control( 'active_item_color', [
				'label'     => esc_html__('Active Text Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => ['{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-features-list-item.kfw-ele-pricing-table-features-list-active-item' => 'color: {{VALUE}};' ]
			]);
			$this->add_control( 'inactive_item_color', [
				'label'     => esc_html__('Inactive Text Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => ['{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-features-list-item.kfw-ele-pricing-table-features-list-inactive-item' => 'color: {{VALUE}};' ]
			]);
			$this->add_group_control( KFGCT::get_type(), [
				'name'        => 'item_typo',
				'description' => esc_html__( 'Typography', 'onnat-extra' ),
				'selector'    => '{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-features-list-item'
			]);
			$this->add_control( 'feature_item_divider_heading', [
				'label'     => esc_html__( 'Divider', 'onnat-extra' ),
				'type'      => KFCM::HEADING,
				'separator' => 'before',
			]);
			$this->add_control( 'show_item_divider', [
				'label'        => esc_html__('Show Divider','onnat-extra' ),
				'type'         => KFCM::SWITCHER,
				'label_on'     => esc_html__( 'On', 'onnat-extra' ),
				'label_off'    => esc_html__( 'Off', 'onnat-extra' ),
				'return_value' => 'yes',
				'default'      => 'no'
			]);
			$this->add_control( 'item_divider_style', [
				'label'     => esc_html__( 'Style', 'onnat-extra' ),
				'type'      => KFCM::SELECT,
				'options'   => [
					'solid'  => esc_html__( 'Solid', 'onnat-extra' ),
					'double' => esc_html__( 'Double', 'onnat-extra' ),
					'dotted' => esc_html__( 'Dotted', 'onnat-extra' ),
					'dashed' => esc_html__( 'Dashed', 'onnat-extra' ),
				],
				'default'   => 'solid',
				'condition' => [ 'show_item_divider' => 'yes', ],
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper ul.kfw-ele-pricing-table-features-list-has-divider li:before' => 'border-bottom-style: {{VALUE}};',
				],
			]);
			$this->add_control( 'item_divider_color', [
				'label'     => esc_html__( 'Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'default'   => '#ddd',
				'condition' => [ 'show_item_divider' => 'yes' ],
				'selectors' => [ '{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper ul.kfw-ele-pricing-table-features-list-has-divider li:before' => 'border-bottom-color: {{VALUE}};' ]
			]);
			$this->add_control( 'item_divider_weight', [
				'label'     => esc_html__( 'Weight', 'onnat-extra' ),
				'type'      => KFCM::SLIDER,
				'default'   => [ 'size' => 2, 'unit' => 'px' ],
				'range'     => [ 'px' => [ 'min' => 1, 'max' => 20 ] ],
				'condition' => [ 'show_item_divider' => 'yes', ],
				'selectors' => [ '{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper ul.kfw-ele-pricing-table-features-list-has-divider li:before' => 'border-bottom-width: {{SIZE}}{{UNIT}};' ]
			]);
			$this->add_control( 'divider_width', [
				'label'     => esc_html__( 'Width', 'onnat-extra' ),
				'type'      => KFCM::SLIDER,
				'condition' => [ 'show_item_divider' => 'yes' ],
				'selectors' => ['{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper ul.kfw-ele-pricing-table-features-list-has-divider li:before' => 'width: {{SIZE}}%;' ],
			]);
			$this->add_control( 'divider_gap', [
				'label'     => esc_html__( 'Gap', 'onnat-extra' ),
				'type'      => KFCM::SLIDER,
				'default'   => [ 'size' => 15, 'unit' => 'px', ],
				'range'     => [
					'px' => [ 'min' => 1, 'max' => 50, ]
				],
				'selectors' => ['{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper ul li' => 'padding-top: {{SIZE}}{{UNIT}}; padding-bottom: {{SIZE}}{{UNIT}}' ],
			]);
			$this->add_control( 'item_alignment', [
				'label'                => esc_html__( 'Alignment', 'onnat-extra' ),
				'type'                 => KFCM::CHOOSE,
				'separator'            => 'before',
				'options'              => [
					'left' => [
						'title' => esc_html__( 'Left', 'onnat-extra' ),
						'icon'  => 'fa fa-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'onnat-extra' ),
						'icon'  => 'fa fa-align-center',
					],
					'right' => [
						'title' => esc_html__( 'Right', 'onnat-extra' ),
						'icon'  => 'fa fa-align-right',
					],
				],
				'prefix_class' 		   => 'kfw-ele-pricing-table-features-list-align-',
				'toggle'      		   => false,
				'default' 			   => 'center',
				'selectors'            => [ '{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper ul.kfw-ele-pricing-table-features-list:not(.kfw-ele-pricing-table-features-list-has-values) li' =>  '{{VALUE}};' ],
				'selectors_dictionary' => [
					'left'   => 'text-align: left;',
					'center' => 'text-align: center;',
					'right'  => 'text-align: right;',
				],
			]);
			$this->add_responsive_control( 'item_background_padding', [
				'label'      => esc_html__( 'Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper ul.kfw-ele-pricing-table-features-list' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
			]);
			$this->add_responsive_control( 'item_background_margin', [
				'label'      => esc_html__( 'Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper ul.kfw-ele-pricing-table-features-list' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
			]);
		$this->end_controls_section();
    }

	public function _register_ribbon_style_section() {
		$this->start_controls_section( 'section_ribbon_style', [
			'label'     => esc_html__( 'Ribbon', 'onnat-extra' ),
			'tab'       => KFCM::TAB_STYLE,
			'condition' => [ 'show_ribbon' => 'yes' ]
		]);
			$this->add_control( 'ribbon_color', [
				'label'     => esc_html__('Ribbon Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-ribbon.corner-1 .kfw-ele-pricing-table-ribbon-content' => 'background: {{VALUE}};',
					'{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-ribbon.corner-2'                                       => 'background: {{VALUE}};',
					'{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-ribbon.flag-1'                                         => 'background: {{VALUE}};',
					'{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-ribbon.flag-1:after'                                   => 'border-top-color:{{VALUE}}; border-bottom-color:{{VALUE}};',
					'{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-ribbon.flag-2'                                         => 'background: {{VALUE}};',
					'{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-ribbon.flag-2:after'                                   => 'border-left-color:{{VALUE}};',
				]
			]);
			$this->add_control( 'ribbon_text_color', [
				'label'     => esc_html__('Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-ribbon-content' => 'color: {{VALUE}};'
				]
			]);
			$this->add_group_control( KFGCT::get_type(), [
				'name'        => 'ribbon_text_typo',
				'description' => esc_html__( 'Typography', 'onnat-extra' ),
				'selector'    => '{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-ribbon-content'
			]);
			$this->add_group_control( KFGCTS::get_type(), [
				'name'     => 'ribbon_text_shadow',
				'selector' => '{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-ribbon-content'
			]);
		$this->end_controls_section();
    }

    public function _register_button_style_section() {
		$this->start_controls_section( 'section_button_style', [
			'label' => esc_html__( 'Button', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_group_control( KFGCT::get_type(), [
				'name'        => 'button_typo',
				'description' => esc_html__( 'Typography', 'onnat-extra' ),
				'selector'    => '{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-button'
			]);
			$this->add_responsive_control( 'button_padding', [
				'label'      => esc_html__( 'Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
			]);
			$this->add_responsive_control( 'button_margin', [
				'label'      => esc_html__( 'Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'separator'  => 'after',
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-button' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
			]);
			$this->start_controls_tabs( 'button_customize_tabs' );
				$this->start_controls_tab( 'button_normal_state', [
					'label' => esc_html__( 'Normal', 'onnat-extra' ),
				]);
					$this->add_control( 'button_color', [
						'label'     => esc_html__('Button Text Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => ['{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-button' => 'color: {{VALUE}};' ],
					]);
					$this->add_group_control( KFGCBG::get_type(), [
						'name'     => 'button_background',
						'label'    => esc_html__('Background', 'onnat-extra' ),
						'types'    => [ 'classic', 'gradient' ],
						'exclude'  => [ 'image' ], // to remove background image option in classic mode
						'selector' => '{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-button'
					]);
					$this->add_group_control( KFGCBR::get_type(), [
						'name'           => 'button_border',
						'fields_options' => [
							'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
							'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
						],
						'selector'       => '{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-button'
					]);
					$this->add_responsive_control( 'button_border_radius', [
						'label'      => esc_html__('Border Radius', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', '%' ],
						'selectors'  => [ '{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ]
					]);
					$this->add_group_control( KFGCBS::get_type(), [
						'name'     => 'button_box_shadow',
						'selector' => '{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-button'
					]);
				$this->end_controls_tab();
				$this->start_controls_tab( 'button_hover_state', [
					'label'     => esc_html__( 'Hover', 'onnat-extra' ),
				]);
					$this->add_control( 'button_color_hover', [
						'label'     => esc_html__('Button Text Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => [ '{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-button:hover' => 'color: {{VALUE}};' ],
					]);
					$this->add_group_control( KFGCBG::get_type(), [
						'name'     => 'button_background_hover',
						'label'    => esc_html__('Background', 'onnat-extra' ),
						'types'    => [ 'classic', 'gradient' ],
						'exclude'  => [ 'image' ], // to remove background image option in classic mode
						'selector' => '{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-button:hover'
					]);
					$this->add_group_control( KFGCBR::get_type(), [
						'name'           => 'button_border_hover',
						'fields_options' => [
							'width'  => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
							'color'  => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
						],
						'selector'       => '{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-button:hover'
					]);
					$this->add_responsive_control( 'button_border_radius_hover', [
						'label'      => esc_html__('Border Radius', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', '%' ],
						'selectors'  => [
							'{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-button:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
						]
					]);
					$this->add_group_control( KFGCBS::get_type(), [
						'name'      => 'button_box_shadow_hover',
						'selector'  => '{{WRAPPER}} .kfw-ele-pricing-table-basic-wrapper .kfw-ele-pricing-table-button:hover'
					]);
					$this->add_control( 'button_animation_hover', [
						'label' => esc_html__( 'Animation', 'onnat-extra' ),
						'type'  => KFCM::HOVER_ANIMATION,
					]);
				$this->end_controls_tab();
			$this->end_controls_tabs();
		$this->end_controls_section();
    }

    public function _render_icons( $icon , $echo = true ) {
		$icon = array_filter( $icon );

		if( !empty( $icon ) ) {
			echo '<span class="kfw-elementor-icon-wrapper">';
				KFIM::render_icon( $icon, [ 'aria-hidden' => 'true' ] );
			echo '</span>';
		}
	}

	public function _ribbon( $settings ) {
		if( $settings['show_ribbon'] == 'yes' && $settings['ribbon_text'] ) {

			$this->add_render_attribute( 'ribbon-wrapper', [
				'class' => [
					'kfw-ele-pricing-table-ribbon',
					esc_attr( $settings['ribbon_style'] ),
					esc_attr( $settings['ribbon_alignment'] )
				]
			]);

			echo '<div '.$this->get_render_attribute_string( 'ribbon-wrapper' ).'>';
				echo '<div class="kfw-ele-pricing-table-ribbon-content">'.esc_html( $settings['ribbon_text'] ).'</div>';
			echo '</div>';
		}
	}

	public function _titles( $settings ) {

		if( $settings['title'] || ( $settings['show_sub_title'] == 'yes' && $settings['sub_title'] ) ) {

			echo '<div class="kfw-ele-pricing-table-header">';
				if( !empty( $settings['title'] ) ) {
					printf( '<span class="kfw-ele-pricing-table-title">%s</span>', esc_html( $settings['title'] ) );
				}

				if( $settings['show_sub_title'] == 'yes' && !empty( $settings['sub_title'] ) ) {
					printf( '<span class="kfw-ele-pricing-table-sub-title">%s</span>', esc_html( $settings['sub_title'] ) );
				}
			echo '</div>';
		}
	}

	public function _button( $settings ) {
		$this->add_render_attribute( 'cta-wrapper', [
			'class' => array_filter([
				'kfw-ele-pricing-table-button-wrapper',
				'kfw-ele-pricing-table-button-style-' . esc_attr( $settings['button_style'] ),
				'kfw-ele-pricing-table-button-align-' . esc_attr( $settings['button_align'] ),
				!empty( trim( $settings['button_text'] ) ) ? 'kfw-ele-pricing-table-button-has-txt' : ''
			])
		]);

		echo '<div '.$this->get_render_attribute_string( 'cta-wrapper' ).'>';

			$this->add_render_attribute( 'cta', [
				'class' => array_filter([
					'kfw-ele-pricing-table-button',
					'kfw-ele-pricing-table-button-icon-align-' . esc_attr( $settings['button_icon_pos'] ),
					!empty( $settings['button_animation_hover'] ) ? 'elementor-animation-'.esc_attr( $settings['button_animation_hover'] ): '',
				])
			]);

			if ( ! empty( $settings['button_link']['url'] ) ) {
				$this->add_render_attribute( 'cta', 'href', esc_url( $settings['button_link']['url'] ) );

				if ( ! empty( $settings['button_link']['is_external'] ) ) {
					$this->add_render_attribute( 'cta', 'target', '_blank' );
				}

				if ( ! empty( $settings['button_link']['nofollow'] ) ) {
					$this->add_render_attribute( 'cta', 'rel', 'nofollow' );
				}
			}

			echo '<a '.$this->get_render_attribute_string( 'cta' ).'>';

				if( $settings['button_icon_pos'] === 'left' ) {
					$this->_render_icons( $settings['button_icon']);
				}

				echo esc_html( $settings['button_text'] );

				if( $settings['button_icon_pos'] === 'right' ) {
					$this->_render_icons( $settings['button_icon']);
				}

			echo '</a>';

		echo '</div>';
	}

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-pricing-table-basic");
		include $render;
    }
}