<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use Elementor\Widget_Base as KFWB;
use Elementor\Controls_Manager as KFCM;
use Elementor\Group_Control_Typography as KFGCT;
use \Elementor\Group_Control_Text_Shadow as KFGCTS;
use \Elementor\Group_Control_Text_Stroke as KFGCTST;
use Elementor\Icons_Manager as KFIM;
use Elementor\Group_Control_Border as KFGCBR;
use Elementor\Group_Control_Box_Shadow as KFGCBS;

class KFW_Progressbar_Circle extends KFWB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-progress-bar-circle';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Progress Bar Circle', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_core_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-counter-circle';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'progress', 'bar', 'circle' ];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
        return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/progressbar-circle';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

		wp_register_script( 'jquery.asPieProgress',
			KFE_CONST_URL . 'libraries/elementor/assets/js/jquery.kfw.asPieProgress' . KFE_CONST_DEBUG_SUFFIX . '.js',
			[ 'jquery' ],
			KFE_CONST_VERSION,
			true
		);

		wp_register_script( 'kfw-progressbar-circle',
			KFE_CONST_URL . 'libraries/elementor/assets/js/progressbar-circle' . KFE_CONST_DEBUG_SUFFIX . '.js',
			[ 'jquery', 'jquery.asPieProgress', 'elementor-frontend' ],
			KFE_CONST_VERSION,
			true
		);

		wp_register_style( 'kfw-progressbar-circle',
			KFE_CONST_URL . 'libraries/elementor/assets/css/progressbar-circle' . KFE_CONST_DEBUG_SUFFIX . '.css',
			[],
			KFE_CONST_VERSION
		);
    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'kfw-progressbar-circle' ];
	}

	/**
	 * Get script dependencies.
	 *
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @return array Element scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'kfw-progressbar-circle' ];
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
		$this->_register_title_section();
		$this->_register_bar_section();

		$this->_register_top_section();
		$this->_register_middle_section();
		$this->_register_bottom_section();

		$this->_register_title_style_section();
		$this->_register_bar_style_section();

		$this->_register_before_text_style_section();
		$this->_register_middle_section_style_section();
		$this->_register_after_text_style_section();
	}

	public function _register_title_section() {
		$this->start_controls_section( 'section_title', [
			'label' => esc_html__( 'Title', 'onnat-extra' )
		]);
			$this->add_control('title',[
				'label'       => esc_html__( 'Title', 'onnat-extra' ),
				'type'        => KFCM::TEXT,
				'dynamic'     => [ 'active' => true, ],
				'placeholder' => esc_html__( 'Enter Name', 'onnat-extra' ),
				'default'     => esc_html__( 'Circle Progress Bar', 'onnat-extra' ),
				'label_block' => true,
			]);
			$this->add_control('title_pos',[
				'label'        => esc_html__( 'Title Position?', 'onnat-extra' ),
				'type'         => KFCM::SWITCHER,
				'label_on'     => esc_html__( 'Top', 'onnat-extra' ),
				'label_off'    => esc_html__( 'Bottom', 'onnat-extra' ),
				'return_value' => 'yes',
				'default'      => 'yes',
				'condition'    => [
					'title!' => ''
				]
			]);

		$this->end_controls_section();
	}

	public function _register_bar_section() {
		$this->start_controls_section( 'section_bar', [
			'label' => esc_html__( 'Bar', 'onnat-extra' )
		]);
			$this->add_control('bar_val_type',[
				'label'              => esc_html__( 'Value Type', 'onnat-extra' ),
				'type'               => KFCM::CHOOSE,
				'options'            => [
					'percent'  => [
						'title' => esc_html__('Percentage', 'onnat-extra'),
						'icon'  => 'eicon-flip'
					],
					'absolute' => [
						'title' => esc_html__('Absolute', 'onnat-extra'),
						'icon' => 'eicon-custom'
					],
				],
				'frontend_available' => true,
				'default'            => 'percent',
				'toggle'             => false
			]);
			$this->add_control('percent',[
				'label'     => esc_html__( 'Percentage Value', 'onnat-extra' ),
				'type'      => KFCM::NUMBER,
				'default'   => 50,
				'min'       => 0,
				'max'       => 100,
				'condition' => [
					'bar_val_type' => 'percent'
				]
			]);
			$this->add_control('abs_fill_val',[
				'label'     => esc_html__( 'Fill Value', 'onnat-extra' ),
				'type'      => KFCM::NUMBER,
				'default'   => 50,
				'min'       => 0,
				'condition' => [
					'bar_val_type' => 'absolute'
				]
			]);
			$this->add_control('abs_max_val',[
				'label'     => esc_html__( 'Max Value', 'onnat-extra' ),
				'type'      => KFCM::NUMBER,
				'default'   => 100,
				'min'       => 0,
				'condition' => [
					'bar_val_type' => 'absolute'
				]
			]);
			$this->add_control('show',[
				'label'              => esc_html__( 'Show ?', 'onnat-extra' ),
				'type'               => KFCM::SWITCHER,
				'label_on'           => esc_html__( ' % ', 'onnat-extra' ),
				'label_off'          => esc_html__( 'Max', 'onnat-extra' ),
				'description'        => esc_html__( 'Show % symbol or Max value.', 'onnat-extra' ),
				'return_value'       => 'yes',
				'default'            => 'yes',
				'frontend_available' => true,
			]);
			$this->add_control( 'bar_speed', [
				'label'      => esc_html__('Speed', 'onnat-extra' ),
				'type'       => KFCM::SLIDER,
				'size_units' => [''],
				'default'    => [
					'unit' => '',
					'size' => 15,
				],
				'range'      => [
					'' => [
						'min' => 10,
						'max' => 1000,
						'step' => 5,
					],
				],
			]);
		$this->end_controls_section();
	}

	public function _register_top_section() {
		$this->start_controls_section( 'section_top', [
			'label' => esc_html__( 'Top Section', 'onnat-extra' )
		]);
			$this->add_control('top_type',[
				'label'   => esc_html__( 'Type', 'onnat-extra' ),
				'type'    => KFCM::CHOOSE,
				'options' => [
					'blank' => [
						'title' => esc_html__('Blank', 'onnat-extra'),
						'icon'  => '',
					],
					'icon'  => [
						'title' => esc_html__('Icon', 'onnat-extra'),
						'icon'  => 'eicon-plus-circle-o',
					],
					'text'  => [
						'title' => esc_html__('Text', 'onnat-extra'),
						'icon'  => 'eicon-animation-text',
					],
				],
				'default' => 'blank',
				'toggle'  => false
			]);
			$this->add_control( 'top_icon', [
				'label'     => esc_html__( 'Icon', 'onnat-extra' ),
				'type'      => KFCM::ICONS,
				'default'   => [ '' ],
				'condition' => [ 'top_type' => 'icon' ],
			]);
			$this->add_control( 'top_text', [
				'label'       => esc_html__( 'Text', 'onnat-extra' ),
				'type'        => KFCM::TEXT,
				'dynamic'     => [
					'active' => true,
				],
				'condition'   => [ 'top_type' => 'text' ],
				'placeholder' => esc_html__( 'Enter Top Text', 'onnat-extra' ),
			]);
		$this->end_controls_section();
	}

	public function _register_middle_section() {
		$this->start_controls_section( 'section_middle', [
			'label' => esc_html__( 'Middle Section', 'onnat-extra' )
		]);
			$this->add_control('middle_type',[
				'label'   => esc_html__( 'Type', 'onnat-extra' ),
				'type'    => KFCM::CHOOSE,
				'options' => [
					'blank' => [
						'title' => esc_html__('Blank', 'onnat-extra'),
						'icon'  => '',
					],
					'val'  => [
						'title' => esc_html__('Value', 'onnat-extra'),
						'icon'  => 'eicon-plus',
					],
					'text'  => [
						'title' => esc_html__('Text', 'onnat-extra'),
						'icon'  => 'eicon-animation-text',
					],
				],
				'default' => 'val',
				'toggle'  => false
			]);
			$this->add_control( 'middle_text', [
				'label'       => esc_html__( 'Text', 'onnat-extra' ),
				'type'        => KFCM::TEXT,
				'dynamic'     => [
					'active' => true,
				],
				'condition'   => [ 'middle_type' => 'text' ],
				'placeholder' => esc_html__( 'Enter Middle Text', 'onnat-extra' ),
			]);
		$this->end_controls_section();
	}

	public function _register_bottom_section() {
		$this->start_controls_section( 'section_bottom', [
			'label' => esc_html__( 'Bottom Section', 'onnat-extra' )
		]);
			$this->add_control('bottom_type',[
				'label'   => esc_html__( 'Type', 'onnat-extra' ),
				'type'    => KFCM::CHOOSE,
				'options' => [
					'blank' => [
						'title' => esc_html__('Blank', 'onnat-extra'),
						'icon'  => '',
					],
					'icon'  => [
						'title' => esc_html__('Icon', 'onnat-extra'),
						'icon'  => 'eicon-plus-circle-o',
					],
					'text'  => [
						'title' => esc_html__('Text', 'onnat-extra'),
						'icon'  => 'eicon-animation-text',
					],
				],
				'default' => 'blank',
				'toggle'  => false
			]);
			$this->add_control( 'bottom_icon', [
				'label'     => esc_html__( 'Icon', 'onnat-extra' ),
				'type'      => KFCM::ICONS,
				'default'   => [ '' ],
				'condition' => [ 'bottom_type' => 'icon' ],
			]);
			$this->add_control( 'bottom_text', [
				'label'       => esc_html__( 'Text', 'onnat-extra' ),
				'type'        => KFCM::TEXT,
				'dynamic'     => [
					'active' => true,
				],
				'condition'   => [ 'bottom_type' => 'text' ],
				'placeholder' => esc_html__( 'Enter Bottom Text', 'onnat-extra' ),
			]);
		$this->end_controls_section();
	}

	public function _register_bar_style_section() {
		$this->start_controls_section( 'section_bar_style',[
			'label' => esc_html__( 'Bar', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_responsive_control( 'bar_max_size', [
				'label'      => esc_html__('Size', 'onnat-extra' ),
				'type'       => KFCM::SLIDER,
				'size_units' => ['px'],
				'range'      => [
					'px' => [
						'min' => 100,
						'max' => 999,
					],
				],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-pbar-circle' => 'width:{{SIZE}}px;'
				],
			]);
			$this->add_control('bar_size',[
				'label'              => esc_html__( 'Border Size', 'onnat-extra' ),
				'type'               => KFCM::NUMBER,
				'frontend_available' => true,
				'min'                => 1,
				'max'                => 25,
				'step'               => 1,
				'default'            => 5
			]);
			$this->add_control('bar_style',[
				'label'   => esc_html__( 'Bar Style', 'onnat-extra' ),
				'type'    => KFCM::CHOOSE,
				'options' => [
					'kfw-ele-widget-pbar-circle-round'  => [
						'title' => esc_html__('Rounded', 'onnat-extra'),
						'icon'  => 'eicon-circle'
					],
					'kfw-ele-widget-pbar-circle-square' => [
						'title' => esc_html__('Square', 'onnat-extra'),
						'icon' => 'eicon-square'
					],
				],
				'default' => 'kfw-ele-widget-pbar-circle-round',
				'toggle'  => false,
			]);
			$this->add_control('track_color',[
				'label' => esc_html__( 'Track Color', 'onnat-extra' ),
				'type'  => KFCM::COLOR,
			]);
			$this->add_control('bar_color',[
				'label' => esc_html__( 'Bar Color', 'onnat-extra' ),
				'type'  => KFCM::COLOR,
			]);
			$this->add_responsive_control( 'pbar_padding',[
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-pbar-circle' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
		$this->end_controls_section();
	}

	public function _register_before_text_style_section() {
		$this->start_controls_section( 'section_before_text_style',[
			'label'     => esc_html__( 'Top Section', 'onnat-extra' ),
			'tab'       => KFCM::TAB_STYLE,
			'condition' => [ 'top_type!' => 'blank' ],
		]);
			$this->add_group_control(KFGCT::get_type(), [
				'name'      => 'before_text_typo',
				'label'     => esc_html__( 'Typography', 'onnat-extra' ),
				'selector'  => '{{WRAPPER}} .kfw-ele-widget-pbar-circle-content-before',
				'condition' => [
					'top_type' => 'text'
				],
			]);
			$this->add_control( 'before_text_color', [
				'label'     => esc_html__('Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'condition' => [ 'top_type' => 'text' ],
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-pbar-circle-content-before' => 'color:{{VALUE}};'
				]
			]);
			$this->add_responsive_control( 'before_icon_size', [
				'label'     => esc_html__( 'Icon Size', 'onnat-extra' ),
				'type'      => KFCM::SLIDER,
				'range'     => [ 'px' => [  'min' => 5, 'max' => 200  ] ],
				'condition' => [
					'top_type' => 'icon'
				],
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-pbar-circle-content-before-icon .kfw-ele-widget-pbar-circle-icon-wrapper' => 'font-size:{{SIZE}}{{UNIT}};'
				]
			]);
			$this->add_control( 'before_icon_normal_color', [
				'label'     => esc_html__('Normal Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'condition' => [ 'top_type' => 'icon' ],
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-pbar-circle-content-before-icon .kfw-ele-widget-pbar-circle-icon-wrapper i'   => 'color:{{VALUE}};',
					'{{WRAPPER}} .kfw-ele-widget-pbar-circle-content-before-icon .kfw-ele-widget-pbar-circle-icon-wrapper svg' => 'fill:{{VALUE}};'
				]
			]);
			$this->add_control( 'before_icon_hover_color', [
				'label'     => esc_html__('Hover Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'condition' => [ 'top_type' => 'icon' ],
				'selectors' => [
					'{{WRAPPER}}:hover .kfw-ele-widget-pbar-circle-content-before-icon .kfw-ele-widget-pbar-circle-icon-wrapper i'   => 'color:{{VALUE}};',
					'{{WRAPPER}}:hover .kfw-ele-widget-pbar-circle-content-before-icon .kfw-ele-widget-pbar-circle-icon-wrapper svg' => 'fill:{{VALUE}};',
				]
			]);
			$this->add_group_control( KFGCTS::get_type(), [
				'name'      => 'before_text_shadow',
				'selector'  => '{{WRAPPER}} .kfw-ele-widget-pbar-circle-content-before',
				'condition' => [
					'top_type' => 'text'
				],
			]);
			$this->add_group_control( KFGCTST::get_type(), [
				'name'      => 'before_text_stroke',
				'selector'  => '{{WRAPPER}} .kfw-ele-widget-pbar-circle-content-before',
				'condition' => [
					'top_type' => 'text'
				],
			]);
			$this->add_responsive_control( 'before_text_padding',[
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-pbar-circle-content-before' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_responsive_control( 'before_text_margin',[
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-pbar-circle-content-before' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
		$this->end_controls_section();
	}

	public function _register_after_text_style_section() {
		$this->start_controls_section( 'section_after_text_style',[
			'label'     => esc_html__( 'Bottom Section', 'onnat-extra' ),
			'tab'       => KFCM::TAB_STYLE,
			'condition' => [ 'bottom_type!' => 'blank' ],
		]);
			$this->add_group_control(KFGCT::get_type(), [
				'name'      => 'after_text_typo',
				'label'     => esc_html__( 'Typography', 'onnat-extra' ),
				'selector'  => '{{WRAPPER}} .kfw-ele-widget-pbar-circle-content-after',
				'condition' => [
					'bottom_type' => 'text'
				],
			]);
			$this->add_control( 'after_text_color', [
				'label'     => esc_html__('Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-pbar-circle-content-after' => 'color:{{VALUE}};'
				],
				'condition' => [
					'bottom_type' => 'text'
				],
			]);
			$this->add_responsive_control( 'after_icon_size', [
				'label'     => esc_html__( 'Icon Size', 'onnat-extra' ),
				'type'      => KFCM::SLIDER,
				'range'     => [ 'px' => [  'min' => 5, 'max' => 200  ] ],
				'condition' => [
					'bottom_type' => 'icon'
				],
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-pbar-circle-content-after-icon .kfw-ele-widget-pbar-circle-icon-wrapper' => 'font-size:{{SIZE}}{{UNIT}};'
				]
			]);
			$this->add_control( 'after_icon_normal_color', [
				'label'     => esc_html__('Normal Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'condition' => [ 'bottom_type' => 'icon' ],
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-pbar-circle-content-after-icon .kfw-ele-widget-pbar-circle-icon-wrapper i'   => 'color:{{VALUE}};',
					'{{WRAPPER}} .kfw-ele-widget-pbar-circle-content-after-icon .kfw-ele-widget-pbar-circle-icon-wrapper svg' => 'fill:{{VALUE}};',
				]
			]);
			$this->add_control( 'after_icon_hover_color', [
				'label'     => esc_html__('Hover Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'condition' => [ 'bottom_type' => 'icon' ],
				'selectors' => [
					'{{WRAPPER}}:hover .kfw-ele-widget-pbar-circle-content-after-icon .kfw-ele-widget-pbar-circle-icon-wrapper i'   => 'color:{{VALUE}};',
					'{{WRAPPER}}:hover .kfw-ele-widget-pbar-circle-content-after-icon .kfw-ele-widget-pbar-circle-icon-wrapper svg' => 'fill:{{VALUE}};',
				]
			]);
			$this->add_group_control( KFGCTS::get_type(), [
				'name'      => 'after_text_shadow',
				'selector'  => '{{WRAPPER}} .kfw-ele-widget-pbar-circle-content-after',
				'condition' => [
					'bottom_type' => 'text'
				],
			]);
			$this->add_group_control( KFGCTST::get_type(), [
				'name'      => 'after_text_stroke',
				'selector'  => '{{WRAPPER}} .kfw-ele-widget-pbar-circle-content-after',
				'condition' => [
					'bottom_type' => 'text'
				],
			]);
			$this->add_responsive_control( 'after_text_padding',[
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-pbar-circle-content-after' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_responsive_control( 'after_text_margin',[
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-pbar-circle-content-after' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
		$this->end_controls_section();
	}

	public function _register_middle_section_style_section() {
		$this->start_controls_section( 'section_middle_section_style',[
			'label'     => esc_html__( 'Middle Section', 'onnat-extra' ),
			'tab'       => KFCM::TAB_STYLE,
			'condition' => [ 'middle_type!' => 'blank' ],
		]);
			$this->add_group_control(KFGCT::get_type(), [
				'name'      => 'middle_text_typo',
				'label'     => esc_html__( 'Typography', 'onnat-extra' ),
				'selector'  => '{{WRAPPER}} .kfw-ele-widget-pbar-circle-content-middle, {{WRAPPER}} .kfw-ele-widget-pbar-circle-number',
				'condition' => [ 'middle_type!' => 'blank' ],
			]);
			$this->add_control( 'middle_text_color', [
				'label'     => esc_html__('Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'condition' => [ 'middle_type!' => 'blank' ],
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-pbar-circle-content-middle, {{WRAPPER}} .kfw-ele-widget-pbar-circle-number, {{WRAPPER}} .kfw-ele-widget-pbar-circle-number-meta span' => 'color:{{VALUE}};'
				]
			]);
			$this->add_control('symbol_heading',[
				'label'     => esc_html__( '% Symbol Settings', 'onnat-extra' ),
				'type'      => KFCM::HEADING,
				'separator' => 'before',
				'condition' => [
					'show'        => 'yes',
					'middle_type' => 'val'
				]
			]);
			$this->add_responsive_control( 'symbol_size',[
				'label'      => esc_html__( 'Font Size', 'onnat-extra' ),
				'type'       => KFCM::SLIDER,
				'size_units' => [ 'px', 'em' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-pbar-circle-number-meta span' =>'font-size: {{SIZE}}{{UNIT}}'
				],
				'condition'  => [
					'show'        => 'yes',
					'middle_type' => 'val'
				]
			]);
			$this->add_responsive_control( 'symbol_alignment',[
				'label'                => esc_html__( 'Alignment', 'onnat-extra' ),
				'type'                 => KFCM::CHOOSE,
				'options'              => [
					'top'    => [
						'title' => esc_html__('Top','onnat-extra'),
						'icon'  => 'eicon-v-align-top'
					],
					'center' => [
						'title' => esc_html__('Center','onnat-extra'),
						'icon'  => 'eicon-v-align-middle'
					],
					'bottom' => [
						'title' => esc_html__('Bottom','onnat-extra'),
						'icon'  => 'eicon-v-align-bottom'
					],
				],
				'default'              => 'center',
				'tablet_default'       => 'center',
				'mobile_default'       => 'center',
				'toggle'               => false,
				'selectors_dictionary' => [
					'top'    => 'vertical-align:top;',
					'center' => 'vertical-align:middle;',
					'bottom' => 'vertical-align:bottom;',
				],
				'selectors'            => [
					'{{WRAPPER}} .kfw-ele-widget-pbar-circle-number-meta span' => '{{VALUE}};'
				],
				'condition'            => [
					'show'        => 'yes',
					'middle_type' => 'val'
				],
			]);
		$this->end_controls_section();
	}

	public function _register_title_style_section() {
		$this->start_controls_section( 'section_title_style',[
			'label'     => esc_html__( 'Title', 'onnat-extra' ),
			'tab'       => KFCM::TAB_STYLE,
			'condition' => [
				'title!' => ''
			]
		]);
			$this->add_group_control(KFGCT::get_type(), [
				'name'      => 'title_typo',
				'label'     => esc_html__( 'Typography', 'onnat-extra' ),
				'selector'  => '{{WRAPPER}} .kfw-ele-widget-pbar-circle-title',
				'condition' => [
					'title!' => ''
				]
			]);
			$this->add_control( 'title_color', [
				'label'     => esc_html__('Typo Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-pbar-circle-title' => 'color:{{VALUE}};'
				],
				'condition' => [
					'title!' => ''
				],
			]);
			$this->add_control( 'title_bg_color', [
				'label'     => esc_html__('Background Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-pbar-circle-title' => 'background-color:{{VALUE}};'
				],
				'condition' => [
					'title!' => ''
				],
			]);
			$this->add_responsive_control( 'title_padding',[
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-pbar-circle-title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
				'condition'  => [
					'title!' => ''
				],
			]);
			$this->add_responsive_control( 'title_margin',[
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-pbar-circle-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
				'condition'  => [
					'title!' => ''
				],
			]);
			$this->add_group_control( KFGCBR::get_type(), [
				'name'     => 'title_border',
				'selector' => '{{WRAPPER}} .kfw-ele-widget-pbar-circle-title'
			]);
			$this->add_responsive_control( 'title_border_radius', [
				'label'      => esc_html__('Border Radius', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-pbar-circle-title' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_group_control( KFGCBS::get_type(), [
				'name'     => 'title_box_shadow',
				'selector' => '{{WRAPPER}} .kfw-ele-widget-pbar-circle-title'
			]);


		$this->end_controls_section();
	}

	/**
	 * Generate setting json
	 */
	public function _generate_setting_json() {
		$settings = $this->get_settings();
		extract( $settings );

		$options = [
			'min'     => 0,
			'barsize' => $bar_size,
			'easing'  => 'linear',
		];

		$options['speed'] = !empty( $bar_speed['size'] ) ? $bar_speed['size'] : 15;

		if( !empty( $track_color ) ) {
			$options['trackcolor'] = $track_color;
		}

		if( !empty( $bar_color ) ) {
			$options['barcolor'] = $bar_color;
		}

		if( $bar_val_type === 'percent' ) {
			$options[ 'goal' ] = $percent;
			$options[ 'max' ]     = 100;
		} else {
			$options[ 'goal' ] = $abs_fill_val;
			$options[ 'max' ]  = $abs_max_val;
		}

		return wp_json_encode( $options );
	}

	public function _bar_title() {
		$settings = $this->get_settings();
		extract( $settings );

		if( !empty( $title ) ) {

			$this->add_render_attribute( 'title-wrapper', [
				'class' => array_filter([
					'kfw-ele-widget-pbar-circle-title',
				])
			]);

			echo '<div '.$this->get_render_attribute_string( 'title-wrapper' ).'>' . $title . '</div>';
		}
	}

	/**
	 * Render icon from icon manger
	 */
    public function _render_icons( $icon , $echo = true ) {
		echo '<span class="kfw-ele-widget-pbar-circle-icon-wrapper">';
			KFIM::render_icon( $icon, [ 'aria-hidden' => 'true' ] );
		echo '</span>';
	}

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-progressbar-circle");
		include $render;
    }
}