<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use Elementor\Widget_Base as KFWB;
use Elementor\Controls_Manager as KFCM;
use Elementor\Group_Control_Typography as KFGCT;
use Elementor\Group_Control_Border as KFGCBR;
use Elementor\Group_Control_Background as KFGCBG;
use Elementor\Group_Control_Box_Shadow as KFGCBS;
use \Elementor\Group_Control_Text_Shadow as KFGCTS;
use \Elementor\Group_Control_Text_Stroke as KFGCTST;

use Elementor\Icons_Manager as KFIM;


class KFW_Progressbar_Line extends KFWB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-progress-bar-line';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Progress Bar Line', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_core_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-skill-bar';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'progress', 'bar', 'line' ];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
        return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/progressbar-line';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

		wp_register_script( 'kfw-progressbar-line',
			KFE_CONST_URL . 'libraries/elementor/assets/js/progressbar-line' . KFE_CONST_DEBUG_SUFFIX . '.js',
			[ 'jquery', 'elementor-frontend' ],
			KFE_CONST_VERSION,
			true
		);

		wp_register_style( 'kfw-progressbar-line',
			KFE_CONST_URL . 'libraries/elementor/assets/css/progressbar-line' . KFE_CONST_DEBUG_SUFFIX . '.css',
			[],
			KFE_CONST_VERSION
		);
    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'kfw-progressbar-line' ];
	}

	/**
	 * Get script dependencies.
	 *
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @return array Element scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'kfw-progressbar-line' ];
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
		$this->_register_bar_section();
		$this->_register_primary_bar_style_section();
		$this->_register_secondary_bar_style_section();
		$this->_register_text_style_section();
		$this->_register_val_style_section();
	}

	public function _register_bar_section() {
		$this->start_controls_section( 'section_bar', [
			'label' => esc_html__( 'Bar', 'onnat-extra' )
		]);
			$this->add_control('bar_type',[
				'label'              => esc_html__( 'Type', 'onnat-extra' ),
				'type'               => KFCM::SELECT,
				'options'            => [
					'inside-the-bar-style'          => esc_html__('Inside the bar', 'onnat-extra'),
					'inside-the-bar-centered-style' => esc_html__('Inside the bar - centered', 'onnat-extra'),
					'inside-the-empty-bar-style'    => esc_html__('Inside the empty bar', 'onnat-extra'),
					'on-the-right-style'            => esc_html__('On the right', 'onnat-extra'),
					'show-as-tooltip-style'         => esc_html__('Show as tooltip 1', 'onnat-extra'),
					'show-as-tooltip-alt-style'     => esc_html__('Show as tooltip 2', 'onnat-extra'),
				],
				'default'            => 'show-as-tooltip-style',
				'style_transfer'     => true,
				'prefix_class'       => 'kfw-ele-pbar-line-',
				'frontend_available' => true
			]);
			$this->add_control('name',[
				'label'       => esc_html__( 'Text', 'onnat-extra' ),
				'type'        => KFCM::TEXT,
				'dynamic'     => [ 'active' => true, ],
				'placeholder' => esc_html__( 'Enter Name', 'onnat-extra' ),
				'default'     => esc_html__( 'Progress Bar', 'onnat-extra' ),
				'label_block' => true,
			]);
			$this->add_control('name_pos',[
				'label'        => esc_html__( 'Text Position?', 'onnat-extra' ),
				'type'         => KFCM::SWITCHER,
				'label_on'     => esc_html__( 'Top', 'onnat-extra' ),
				'label_off'    => esc_html__( 'Bottom', 'onnat-extra' ),
				'return_value' => 'yes',
				'default'      => 'yes',
				'condition'    => [
					'name!' => ''
				]
			]);
			$this->add_control('bar_style',[
				'label'              => esc_html__( 'Style', 'onnat-extra' ),
				'type'               => KFCM::SELECT,
				'default'            => 'kfw-ele-pbar-line-default-bar',
				'options'            => [
					'kfw-ele-pbar-line-default-bar'          => esc_html__('Default', 'onnat-extra'),
					'kfw-ele-pbar-line-striped-bar'          => esc_html__('Striped', 'onnat-extra'),
					'kfw-ele-pbar-line-animated-striped-bar' => esc_html__('Striped With Animation', 'onnat-extra'),
				],
				'style_transfer'     => true,
				'frontend_available' => true,
			]);
			$this->add_control('bar_val_type',[
				'label'   => esc_html__( 'Value Type', 'onnat-extra' ),
				'type'    => KFCM::CHOOSE,
				'options' => [
					'percent'  => [
						'title' => esc_html__('Percentage', 'onnat-extra'),
						'icon'  => 'eicon-flip'
					],
					'absolute' => [
						'title' => esc_html__('Absolute', 'onnat-extra'),
						'icon' => 'eicon-custom'
					],
				],
				'default' => 'percent',
				'toggle'  => false
			]);
			$this->add_control('percent',[
				'label'     => esc_html__( 'Percentage Value', 'onnat-extra' ),
				'type'      => KFCM::NUMBER,
				'default'   => 50,
				'min'       => 0,
				'max'       => 100,
				'condition' => [
					'bar_val_type' => 'percent'
				]
			]);
			$this->add_control('abs_fill_val',[
				'label'     => esc_html__( 'Fill Value', 'onnat-extra' ),
				'type'      => KFCM::NUMBER,
				'default'   => 50,
				'min'       => 0,
				'condition' => [
					'bar_val_type' => 'absolute'
				]
			]);
			$this->add_control('abs_max_val',[
				'label'     => esc_html__( 'Max Value', 'onnat-extra' ),
				'type'      => KFCM::NUMBER,
				'default'   => 100,
				'min'       => 0,
				'condition' => [
					'bar_val_type' => 'absolute'
				]
			]);
			$this->add_control('show',[
				'label'        => esc_html__( 'Show ?', 'onnat-extra' ),
				'type'         => KFCM::SWITCHER,
				'label_on'     => esc_html__( ' % ', 'onnat-extra' ),
				'label_off'    => esc_html__( 'Max', 'onnat-extra' ),
				'description'  => esc_html__( 'Show % symbol or Max value in a bar.', 'onnat-extra' ),
				'return_value' => 'yes',
				'default'      => 'yes',
			]);
			$this->add_control('anim_heading',[
				'label'     => esc_html__( 'Animation Settings', 'onnat-extra' ),
				'type'      => KFCM::HEADING,
				'separator' => 'before',
			]);
			$this->add_control( 'anim_speed', [
				'label'     => esc_html__( 'Speed (s)', 'onnat-extra' ),
				'type'      => KFCM::SLIDER,
				'range'     => [
					'px' => [
						'min'  => 0.1,
						'max'  => 10,
						'step' => 0.1,
					]
				],
				'default'   => [
					'size' => 1
				],
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-pbar-line-bar' => 'transition-duration:{{SIZE}}s;'
				],
            ]);
			$this->add_control('anim_type',[
				'label'   => esc_html__( 'Type', 'onnat-extra' ),
				'type'    => KFCM::SELECT,
				'default' => 'kfw-ele-widget-pbar-line-bar-back-anim',
				'options' => [
					'kfw-ele-widget-pbar-line-bar-back-anim'        => esc_html__( 'Back', 'onnat-extra' ),
					'kfw-ele-widget-pbar-line-bar-circ-anim'        => esc_html__( 'Circ', 'onnat-extra' ),
					'kfw-ele-widget-pbar-line-bar-cubic-anim'       => esc_html__( 'Cubic', 'onnat-extra' ),
					'kfw-ele-widget-pbar-line-bar-ease-anim'        => esc_html__( 'Ease', 'onnat-extra' ),
					'kfw-ele-widget-pbar-line-bar-ease-in-anim'     => esc_html__( 'Ease In', 'onnat-extra' ),
					'kfw-ele-widget-pbar-line-bar-ease-in-out-anim' => esc_html__( 'Ease In Out', 'onnat-extra' ),
					'kfw-ele-widget-pbar-line-bar-ease-out-anim'    => esc_html__( 'Ease Out', 'onnat-extra' ),
					'kfw-ele-widget-pbar-line-bar-expo-anim'        => esc_html__( 'Expo', 'onnat-extra' ),
					'kfw-ele-widget-pbar-line-bar-linear-anim'      => esc_html__( 'Linear', 'onnat-extra' ),
					'kfw-ele-widget-pbar-line-bar-quad-anim'        => esc_html__( 'Quad', 'onnat-extra' ),
					'kfw-ele-widget-pbar-line-bar-quart-anim'       => esc_html__( 'Quart', 'onnat-extra' ),
					'kfw-ele-widget-pbar-line-bar-quint-anim'       => esc_html__( 'Quint', 'onnat-extra' ),
					'kfw-ele-widget-pbar-line-bar-sine-anim'        => esc_html__( 'Sine', 'onnat-extra' ),
				],
			]);
		$this->end_controls_section();
	}

	public function _register_primary_bar_style_section() {
		$this->start_controls_section( 'section_primary_bar_style',[
			'label' => esc_html__( 'Primary Bar', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_control( 'pbar_height', [
				'label'     => esc_html__( 'Height', 'onnat-extra' ),
				'type'      => KFCM::SLIDER,
				'range'     => [
					'px' => [
						'min'  => 1,
						'max'  => 500,
						'step' => 1,
					]
				],
				'default'   => [
					'size' => 20
				],
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-pbar-line-bar'  => 'height:{{SIZE}}{{UNIT}};',
					'{{WRAPPER}}.kfw-ele-pbar-line-inside-the-bar-style .kfw-ele-widget-pbar-line-info' => 'line-height:{{SIZE}}{{UNIT}};',
					'{{WRAPPER}}.kfw-ele-pbar-line-inside-the-bar-centered-style .kfw-ele-widget-pbar-line-info' => 'line-height:{{SIZE}}{{UNIT}};',
				],
			]);
			$this->add_group_control( KFGCBG::get_type(), [
				'name'     => 'pbar_bg',
				'types'    => ['classic', 'gradient'],
				'exclude'  => [ 'image'],
				'selector' => '{{WRAPPER}} .kfw-ele-widget-pbar-line-bar, {{WRAPPER}}.kfw-ele-pbar-line-show-as-tooltip-style .kfw-ele-widget-pbar-line-info',
			]);
			$this->add_group_control( KFGCBR::get_type(), [
				'name'     => 'pbar_border',
				'selector' => '{{WRAPPER}} .kfw-ele-widget-pbar-line-bar'
			]);
			$this->add_responsive_control( 'pbar_border_radius', [
				'label'      => esc_html__('Border Radius', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-pbar-line-bar, {{WRAPPER}}.kfw-ele-pbar-line-show-as-tooltip-style .kfw-ele-widget-pbar-line-info' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_responsive_control( 'pbar_margin',[
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-pbar-line-bar' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
		$this->end_controls_section();
	}

	public function _register_secondary_bar_style_section() {
		$this->start_controls_section( 'section_secondary_bar_style',[
			'label' => esc_html__( 'Secondary Bar', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_control( 'sbar_height', [
				'label'     => esc_html__( 'Height', 'onnat-extra' ),
				'type'      => KFCM::SLIDER,
				'range'     => [
					'px' => [
						'min'  => 1,
						'max'  => 500,
						'step' => 1,
					]
				],
				'default'   => [
					'size' => 20
				],
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-pbar-line-bar-wrapper'                                        => 'height:{{SIZE}}{{UNIT}};',
					'{{WRAPPER}}.kfw-ele-pbar-line-inside-the-empty-bar-style .kfw-ele-widget-pbar-line-info' => 'line-height:{{SIZE}}{{UNIT}};',
				],
			]);
			$this->add_group_control( KFGCBG::get_type(), [
				'name'     => 'sbar_bg',
				'types'    => ['classic', 'gradient'],
				'exclude'  => [ 'image'],
				'selector' => '{{WRAPPER}} .kfw-ele-widget-pbar-line-bar-wrapper',
			]);
			$this->add_group_control( KFGCBR::get_type(), [
				'name'     => 'sbar_border',
				'selector' => '{{WRAPPER}} .kfw-ele-widget-pbar-line-bar-wrapper'
			]);
			$this->add_responsive_control( 'sbar_border_radius', [
				'label'      => esc_html__('Border Radius', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-pbar-line-bar-wrapper' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_group_control( KFGCBS::get_type(), [
				'name'     => 'sbar_box_shadow',
				'selector' => '{{WRAPPER}} .kfw-ele-widget-pbar-line-bar-wrapper'
			]);
			$this->add_responsive_control( 'sbar_margin',[
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-pbar-line-bar-wrapper' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
		$this->end_controls_section();
	}

	public function _register_text_style_section() {
		$this->start_controls_section( 'section_text_style',[
			'label' => esc_html__( 'Text', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_group_control(KFGCT::get_type(), [
				'name'     => 'text_typo',
				'label'    => esc_html__( 'Typography', 'onnat-extra' ),
				'selector' => '{{WRAPPER}} .kfw-ele-widget-pbar-line-name'
			]);
			$this->add_control( 'text_color', [
				'label'     => esc_html__('Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-pbar-line-name' => 'color:{{VALUE}};'
				]
			]);
			$this->add_group_control( KFGCTS::get_type(), [
				'name'     => 'text_shadow',
				'selector' => '{{WRAPPER}} .kfw-ele-widget-pbar-line-name'
			]);
			$this->add_group_control( KFGCTST::get_type(), [
				'name'     => 'text_stroke',
				'selector' => '{{WRAPPER}} .kfw-ele-widget-pbar-line-name'
			]);
			$this->add_responsive_control( 'text_padding',[
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-pbar-line-name' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_responsive_control( 'text_margin',[
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-pbar-line-name' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
		$this->end_controls_section();
	}

	public function _register_val_style_section() {
		$this->start_controls_section( 'section_val_style',[
			'label' => esc_html__( 'Value', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_group_control(KFGCT::get_type(), [
				'name'     => 'val_typo',
				'label'    => esc_html__( 'Typography', 'onnat-extra' ),
				'selector' => '{{WRAPPER}} .kfw-ele-widget-pbar-line-info'
			]);
			$this->add_control( 'val_color', [
				'label'     => esc_html__('Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-pbar-line-info' => 'color:{{VALUE}};'
				]
			]);
			$this->add_group_control( KFGCTS::get_type(), [
				'name'     => 'val_text_shadow',
				'selector' => '{{WRAPPER}} .kfw-ele-widget-pbar-line-info'
			]);
			$this->add_group_control( KFGCTST::get_type(), [
				'name'     => 'val_text_stroke',
				'selector' => '{{WRAPPER}} .kfw-ele-widget-pbar-line-info'
			]);
			$this->add_control( 'pbar_tooltip_style_1_width', [
				'label'     => esc_html__( 'Tooltip Width', 'onnat-extra' ),
				'type'      => KFCM::SLIDER,
				'range'     => [
					'px' => [
						'min'  => 1,
						'max'  => 500,
						'step' => 1,
					]
				],
				'default'   => [
					'size' => 80
				],
				'condition' => [
					'bar_type' => 'show-as-tooltip-style'
				],
				'selectors' => [
					'{{WRAPPER}}.kfw-ele-pbar-line-show-as-tooltip-style .kfw-ele-widget-pbar-line-info' => 'width:{{SIZE}}{{UNIT}};right:calc({{SIZE}}{{UNIT}}/2*-1);',
				],
			]);
			$this->add_control('symbol_heading',[
				'label'     => esc_html__( '% Symbol Settings', 'onnat-extra' ),
				'type'      => KFCM::HEADING,
				'separator' => 'before',
				'condition' => [
					'show' => 'yes'
				]
			]);
			$this->add_responsive_control( 'symbol_size',[
				'label'      => esc_html__( 'Font Size', 'onnat-extra' ),
				'type'       => KFCM::SLIDER,
				'size_units' => [ 'px', 'em' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-pbar-line-info span' =>'font-size: {{SIZE}}{{UNIT}}'
				],
				'condition'  => [
					'show' => 'yes'
				]
			]);
			$this->add_responsive_control( 'symbol_alignment',[
				'label'                => esc_html__( 'Alignment', 'onnat-extra' ),
				'type'                 => KFCM::CHOOSE,
				'options'              => [
					'top'    => [
						'title' => esc_html__('Top','onnat-extra'),
						'icon'  => 'eicon-v-align-top'
					],
					'center' => [
						'title' => esc_html__('Center','onnat-extra'),
						'icon'  => 'eicon-v-align-middle'
					],
					'bottom' => [
						'title' => esc_html__('Bottom','onnat-extra'),
						'icon'  => 'eicon-v-align-bottom'
					],
				],
				'default'              => 'center',
				'tablet_default'       => 'center',
				'mobile_default'       => 'center',
				'toggle'               => false,
				'selectors_dictionary' => [
					'top'    => 'vertical-align:top;',
					'center' => 'vertical-align:middle;',
					'bottom' => 'vertical-align:bottom;',
				],
				'selectors'            => [
					'{{WRAPPER}} .kfw-ele-widget-pbar-line-info span' => '{{VALUE}};'
				],
				'condition'            => [
					'show' => 'yes'
				],
			]);
		$this->end_controls_section();
	}

	/**
	 * Generate setting json
	 */
	protected function _generate_setting_json( array $settings ) {
		extract( $settings );

		$options = [];

		if( $bar_val_type === 'percent' ) {
			$options[ 'percent' ] = $percent.'%';
			$options[ 'max' ]     = 100;
		} else if( $bar_val_type === 'absolute' ) {
			$percent              = ceil( $abs_fill_val / ( $abs_max_val / 100 ) );
			$options[ 'percent' ] = $percent.'%';
			$options[ 'current' ] = $abs_fill_val;
			$options[ 'max' ]     = $abs_max_val;
		}

		return wp_json_encode( $options );
	}

	public function _bar_title() {
		$settings = $this->get_settings();
		extract( $settings );

		if( !empty( $name ) ) {

			$this->add_render_attribute( 'name-wrapper', [
				'class' => array_filter([
					'kfw-ele-widget-pbar-line-name'
				])
			]);

			echo '<div '.$this->get_render_attribute_string( 'name-wrapper' ).'>' . $name . '</div>';
		}
	}

	public function _bar_info( $ext_class = '' ) {
		$settings = $this->get_settings();
		extract( $settings );

		$this->add_render_attribute( 'info-wrapper', [
			'class' => array_filter([
				'kfw-ele-widget-pbar-line-info',
				$ext_class
			])
		]);

		echo '<div '.$this->get_render_attribute_string( 'info-wrapper' ).'>';
			if( $settings['bar_val_type'] === 'percent' ) {
				if( $settings['show'] === 'yes' ) {
					printf( '%s <span> %s </span>', $settings['percent'], '%' );
				} else {
					printf( '%s / %s', $settings['percent'], '100' );
				}
			} else if( $settings['bar_val_type'] === 'absolute' ) {
				if( $settings['show'] === 'yes' ) {
					printf( '%s <span> %s </span>', $settings['abs_fill_val'], '%' );
				} else {
					printf( '%s / %s', $settings['abs_fill_val'], $settings['abs_max_val'] );
				}
			}
		echo '</div>';
	}

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-progressbar-line");
		include $render;
    }
}