<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use \Elementor\Plugin as KFEl;
use \Elementor\Widget_Base as KFWB;
use \Elementor\Controls_Manager as KFCM;
use \Elementor\Group_Control_Image_Size as KFGCIS;
use \Elementor\Group_Control_Background as KFGCBG;
use \Elementor\Group_Control_Border as KFGCBR;
use \Elementor\Group_Control_Box_Shadow as KFGCBS;

class KFW_Site_Logo extends KFWB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-site-logo';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Site Logo', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_header_footer_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-logo';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'kfw', 'kin', 'logo', 'image' ];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
        return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/site-logo';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

		wp_register_script( 'kfw-site-logo',
			KFE_CONST_URL . 'libraries/elementor/assets/js/site-logo' . KFE_CONST_DEBUG_SUFFIX . '.js',
			[ 'jquery', 'elementor-frontend' ],
			KFE_CONST_VERSION,
			true
		);
    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [];
	}

	/**
	 * Get script dependencies.
	 *
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @return array Element scripts dependencies.
	 */
	public function get_script_depends() {
		$scripts = $this->_is_editor_mode() ? [ 'kfw-site-logo' ] : [];
		return $scripts;
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
		$this->_register_logo_section();
		$this->_register_logo_style_section();
    }

    public function _register_logo_section() {
		$this->start_controls_section( 'section_logo', [
			'label' => esc_html__( 'Site Logo', 'onnat-extra' )
        ]);
			$this->add_control( 'source', [
				'label'   => esc_html__( 'Source', 'onnat-extra' ),
				'type'    => KFCM::CHOOSE,
				'options' => [
					'logo'     => [
						'title' => esc_html__('Logo', 'onnat-extra'),
						'icon'  => 'eicon-logo',
					],
					'logo_alt' => [
						'title' => esc_html__('Logo Alt', 'onnat-extra'),
						'icon'  => 'eicon-site-logo',
					],
					'sticky'   => [
						'title' => esc_html__('Sticky Logo', 'onnat-extra'),
						'icon'  => 'eicon-settings',
					],
					'mobile'   => [
						'title' => esc_html__('Mobile Logo', 'onnat-extra'),
						'icon'  => 'eicon-device-mobile',
					],
					'custom'   => [
						'title' => esc_html__('Custom', 'onnat-extra'),
						'icon'  => 'eicon-image',
					],
				],
				'default' => 'custom',
				'toggle'  => false,
			]);
            $this->add_control('image',[
                'label'   => esc_html__( 'Site Logo', 'onnat-extra' ),
                'type'    => KFCM::MEDIA,
                'default' => [ 'url' => KFE_CONST_URL .'libraries/elementor/assets/images/logo.svg' ],
                'dynamic' => [ 'active' => true ],
				'condition' => [
					'source' => 'custom',
				],
            ]);
            $this->add_group_control( KFGCIS::get_type(), [
                'name'        => 'image_size',
                'default'     => 'full',
                'label_block' => true,
				'condition'   => [
					'source' => 'custom',
				],
            ]);
			$this->add_responsive_control('align', [
                'label'          => esc_html__( 'Alignment', 'onnat-extra' ),
				'type'           => KFCM::CHOOSE,
				'options'        => [
					'left'   => [
						'title' => esc_html__( 'Left', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-center',
					],
					'right'  => [
						'title' => esc_html__( 'Right', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-right',
					],
				],
				'toggle'               => false,
				'default'              => 'center',
				'tablet_default'       => 'center',
				'mobile_default'       => 'center',
				'selectors_dictionary' => [
					'left'   => 'text-align:left;',
					'center' => 'text-align:center;',
					'right'  => 'text-align:right;',
				],
				'selectors'            => [
					'{{WRAPPER}}.kfw-ele-kfw-site-logo' => '{{VALUE}};'
				],
			]);
			$this->add_control('link_type', [
				'label'   => esc_html__( 'Link', 'onnat-extra' ),
				'type'    => KFCM::CHOOSE,
				'options' => [
                    'site-link'   => [
						'title' => esc_html__( 'Site Link', 'onnat-extra' ),
						'icon'  => 'eicon-link',
                    ],
                    'custom-link' => [
						'title' => esc_html__( 'Custom Link', 'onnat-extra' ),
						'icon'  => 'eicon-editor-external-link',
                    ],
                ],
                'toggle'    => false,
                'default'   => 'site-link',
                'condition' => [
                    'image[url]!' => ''
                ]
            ]);
            $this->add_control('custom_link',[
                'label'     => esc_html__( 'Custom Link', 'onnat-extra' ),
                'type'      => KFCM::URL,
                'dynamic'   => [
                    'active' => true
                ],
                'options'   => [
                    'url',
                    'is_external',
                    'nofollow'
                ],
                'default'   => [
                    'url'         => '',
                    'is_external' => true,
                    'nofollow'    => true,
                ],
                'condition' => [
                    'image[url]!' => '',
                    'link_type'   => 'custom-link'
                ]
            ]);
		$this->end_controls_section();
    }

    public function _register_logo_style_section() {
		$this->start_controls_section( 'section_logo_style', [
			'label' => esc_html__( 'Site Logo', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
        ]);
            $this->add_responsive_control( 'max_size', [
                'label'      => esc_html__('Max Size', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'size_units' => ['px'],
                'range'      => [
                    'px' => [
                        'min' => 5,
                        'max' => 500,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-site-logo-main img' => 'max-width:{{SIZE}}px;'
                ],
            ]);
			$this->add_responsive_control( 'margin', [
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-site-logo-main' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_responsive_control( 'padding', [
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-site-logo-main.kfw-ele-widget-site-logo-has-link a' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}; display: inline-block;',
                    '{{WRAPPER}} .kfw-ele-widget-site-logo-main' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .kfw-ele-widget-site-logo-main.kfw-ele-widget-site-logo-has-link' => 'padding: 0;'
				]
			]);
            $this->add_group_control( KFGCBG::get_type(), [
                'name'           => 'bg',
                'types'          => ['classic', 'gradient'],
                'exclude'        => [ 'image'],
                'fields_options' => [
                    'background' => [ 'default' => 'classic' ],
                    'color'      => [ 'default' => '' ],
                ],
                'selector'       => '{{WRAPPER}} .kfw-ele-widget-site-logo-main',
            ]);
            $this->add_group_control( KFGCBR::get_type(), [
                'name'      => 'border',
                'selector'  => '{{WRAPPER}} .kfw-ele-widget-site-logo-main'
            ]);
            $this->add_responsive_control( 'border_radius', [
                'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-site-logo-main' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ]
            ]);
            $this->add_group_control( KFGCBS::get_type(), [
                'name'      => 'box_shadow',
                'selector'  => '{{WRAPPER}} .kfw-ele-widget-site-logo-main'
            ]);
		$this->end_controls_section();
    }

    public function _is_svg( $url ) {

        // Get the file extension from the URL
        $file_extension = pathinfo( $url, PATHINFO_EXTENSION );

        // Convert the extension to lowercase for consistency
        $file_extension = strtolower( $file_extension );

        if( 'svg' === $file_extension ) {
            return true;
        }

        return false;

    }

	/**
	 * Get attachment image src.
	 */
	public static function _get_image_src( $attachment_id, $image_size_key, array $settings ) {
        return KFGCIS::get_attachment_image_src( $attachment_id, $image_size_key, $settings );
	}

    /**
     * Get attachment image html
     */
	public static function _get_image_html( $settings ) {
		return KFGCIS::get_attachment_image_html( $settings );
	}

    /**
     * Get attachment image title
     */
	public function _get_image_title( $attachment ) {
		if ( empty( $attachment['id'] ) ) {
			return '';
		}

		return get_the_title( $attachment['id'] );
	}

    /**
     * Get attachment image alt
     */
	public function _get_image_alt( $instance ) {
		if ( empty( $instance['id'] ) ) {
			// For `Insert From URL` images.
			return isset( $instance['alt'] ) ? trim( strip_tags( $instance['alt'] ) ) : '';
		}

		$attachment_id = $instance['id'];
		if ( ! $attachment_id ) {
			return '';
		}

		$attachment = get_post( $attachment_id );
		if ( ! $attachment ) {
			return '';
		}

		$alt = get_post_meta( $attachment_id, '_wp_attachment_image_alt', true );
		if ( ! $alt ) {
			$alt = $attachment->post_excerpt;
			if ( ! $alt ) {
				$alt = $attachment->post_title;
			}
		}
		return trim( strip_tags( $alt ) );
	}

    public function _is_editor_mode() {
        $preview = KFEl::instance()->preview->is_preview_mode();
        $editor  = KFEl::instance()->editor->is_edit_mode();

        return ( $preview || $editor ) ? true : false;
    }

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-site-logo");
		include $render;
    }
}