<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use \Elementor\Plugin as KFEl;
use \Elementor\Widget_Base as KFWB;
use \Elementor\Controls_Manager as KFCM;

use \Elementor\Group_Control_Typography as KFGCT;


use \Elementor\Group_Control_Text_Shadow as KFGCTS;
use \Elementor\Group_Control_Text_Stroke as KFGCTST;
use \Elementor\Group_Control_Background as KFGCBG;
use \Elementor\Group_Control_Border as KFGCBR;
use \Elementor\Group_Control_Box_Shadow as KFGCBS;

class KFW_Text_Highlighter extends KFWB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-text-highlighter';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Text Highlighter', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_core_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-animated-headline';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'kfw', 'kin', 'text', 'title', 'heading', 'highlight', 'highlighter' ];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
        return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/text-highlighter';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

		wp_register_script( 'kfw-text-highlighter',
			KFE_CONST_URL . 'libraries/elementor/assets/js/text-highlighter' . KFE_CONST_DEBUG_SUFFIX . '.js',
			[ 'jquery', 'elementor-frontend' ],
			KFE_CONST_VERSION,
			true
		);

		wp_register_style( 'kfw-text-highlighter',
			KFE_CONST_URL . 'libraries/elementor/assets/css/text-highlighter' . KFE_CONST_DEBUG_SUFFIX . '.css',
			[],
			KFE_CONST_VERSION
		);
    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'kfw-text-highlighter' ];
	}

	/**
	 * Get script dependencies.
	 *
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @return array Element scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'kfw-text-highlighter' ];
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
		$this->_register_subtitle_section();
		$this->_register_title_section();
		$this->_register_settings_section();

		$this->_register_highlighter_style_section();
        $this->_register_subtitle_style_section();
		$this->_register_title_style_section();
    }

	public function _register_subtitle_section() {
		$this->start_controls_section( 'section_subtitle', [
			'label' => esc_html__( 'Sub Title', 'onnat-extra' )
		]);
            $this->add_control('sub_title',[
                'label'       => esc_html__( 'Sub Title', 'onnat-extra' ),
                'type'        => KFCM::TEXT,
                'dynamic'     => [ 'active' => true, ],
                'placeholder' => esc_html__( 'Enter Sub Title', 'onnat-extra' ),
                'default'     => esc_html__( 'Here is the sub title', 'onnat-extra' ),
                'label_block' => true,
            ]);
			$this->add_control('sub_title_pos',[
                'label'     => esc_html__( 'Position?', 'onnat-extra' ),
				'type'      => KFCM::CHOOSE,
				'options'   => [
					'top'    => [
						'title' => esc_html__( 'Above Title', 'onnat-extra' ),
						'icon'  => 'eicon-v-align-top',
					],
					'bottom' => [
						'title' => esc_html__( 'Below Title', 'onnat-extra' ),
						'icon'  => 'eicon-v-align-bottom',
					],
				],
				'default'   => 'top',
				'toggle'    => false,
				'conditions' => [
                    'relation' => 'or',
                    'terms'    => [
                        [
                            'name'     => 'sub_title',
                            'operator' => '!=',
                            'value'    => ''
                        ],
                        [
                            'name'     => 'primary',
                            'operator' => '!=',
                            'value'    => ''
                        ],
                        [
                            'name'     => 'secondary',
                            'operator' => '!=',
                            'value'    => ''
                        ],
                        [
                            'name'     => 'tertiary',
                            'operator' => '!=',
                            'value'    => ''
                        ],
                    ],
				]
			]);
		$this->end_controls_section();
    }

	public function _register_title_section() {
		$this->start_controls_section( 'section_title', [
			'label' => esc_html__( 'Title', 'onnat-extra' )
		]);
            $this->add_control('primary',[
                'label'       => esc_html__( 'Primary Text', 'onnat-extra' ),
                'type'        => KFCM::TEXT,
                'dynamic'     => [ 'active' => true, ],
                'placeholder' => esc_html__( 'Enter Primary Text', 'onnat-extra' ),
                'default'     => 'Lorem ipsum',
                'label_block' => true,
            ]);
            $this->add_control('secondary',[
                'label'       => esc_html__( 'Secondary Text', 'onnat-extra' ),
                'type'        => KFCM::TEXT,
                'dynamic'     => [ 'active' => true, ],
                'placeholder' => esc_html__( 'Enter Secondary Text', 'onnat-extra' ),
                'default'     => ' dolor sit ',
                'label_block' => true,
            ]);
            $this->add_control('tertiary',[
                'label'       => esc_html__( 'Tertiary Text', 'onnat-extra' ),
                'type'        => KFCM::TEXT,
                'dynamic'     => [ 'active' => true, ],
                'placeholder' => esc_html__( 'Enter Tertiary Text', 'onnat-extra' ),
                'default'     => 'amet',
                'label_block' => true,
            ]);
            $this->add_control('title_link',[
                'label'     => esc_html__( 'Title Link', 'onnat-extra' ),
                'type'      => KFCM::URL,
                'dynamic'   => [
                    'active' => true
                ],
                'options'   => [
                    'url',
                    'is_external',
                    'nofollow'
                ],
                'default'   => [
                    'url'         => '',
                    'is_external' => true,
                    'nofollow'    => true,
                ],
                'conditions' => $this->_title_condition(),
            ]);
            $this->add_control('title_tag', [
                'label'       => esc_html__( 'HTML Tag', 'onnat-extra' ),
                'type'        => KFCM::CHOOSE,
                'label_block' => true,
                'options'     => [
                    'h1'   => [
                        'title' => esc_html__( 'H1 Tag', 'onnat-extra' ),
                        'icon'  => 'eicon-editor-h1',
                    ],
                    'h2'   => [
                        'title' => esc_html__( 'H2 Tag', 'onnat-extra' ),
                        'icon'  => 'eicon-editor-h2',
                    ],
                    'h3'   => [
                        'title' => esc_html__( 'H3 Tag', 'onnat-extra' ),
                        'icon'  => 'eicon-editor-h3',
                    ],
                    'h4'   => [
                        'title' => esc_html__( 'H4 Tag', 'onnat-extra' ),
                        'icon'  => 'eicon-editor-h4',
                    ],
                    'h5'   => [
                        'title' => esc_html__( 'H5 Tag', 'onnat-extra' ),
                        'icon'  => 'eicon-editor-h5',
                    ],
                    'h6'   => [
                        'title' => esc_html__( 'H6 Tag', 'onnat-extra' ),
                        'icon'  => 'eicon-editor-h6',
                    ],
                    'div'  => [
                        'title' => esc_html__( 'div Tag', 'onnat-extra' ),
                        'icon'  => 'kfw-ctrl-icon  kfw-ctrl-div',
                    ],
                    'span' => [
                        'title' => esc_html__( 'span Tag', 'onnat-extra' ),
                        'icon'  => 'kfw-ctrl-icon kfw-ctrl-span',
                    ],
                    'p'    => [
                        'title' => esc_html__( 'p Tag', 'onnat-extra' ),
                        'icon'  => 'eicon-editor-paragraph',
                    ],
                ],
                'default'    => 'h2',
                'toggle'     => false,
                'conditions' => $this->_title_condition()
            ]);

		$this->end_controls_section();
    }

	public function _register_settings_section() {
		$this->start_controls_section( 'section_settings', [
            'label'      => esc_html__( 'Settings', 'onnat-extra' ),
            'conditions' => $this->_title_condition(),
		]);
            $this->add_responsive_control('align',[
                'label'        => esc_html__( 'Alignment', 'onnat-extra' ),
                'type'         => KFCM::CHOOSE,
                'options'      => [
                    'left'    => [
                        'title' => esc_html__( 'Left', 'onnat-extra' ),
                        'icon'  => 'eicon-text-align-left',
                    ],
                    'center'  => [
                        'title' => esc_html__( 'Center', 'onnat-extra' ),
                        'icon'  => 'eicon-text-align-center',
                    ],
                    'right'   => [
                        'title' => esc_html__( 'Right', 'onnat-extra' ),
                        'icon'  => 'eicon-text-align-right',
                    ],
                ],
                'toggle'       => false,
                'default'      => 'center',
                'selectors_dictionary' => [
                    'left'   => 'text-align:left;',
					'center' => 'text-align:center;',
					'right'  => 'text-align:right;',
				],
                'selectors'            => [
                    '{{WRAPPER}} .kfw-ele-widget-text-highlighter-main' => '{{VALUE}};'
                ],
                'conditions'           => $this->_title_condition(),
            ]);
			$this->add_control('highlight_type',[
                'label'              => esc_html__( 'Highlight Type', 'onnat-extra' ),
                'type'               => KFCM::SELECT,
                'default'            => 'underline',
                'options'            => [
					'circle'           => esc_html__( 'Circle', 'Shapes', 'onnat-extra' ),
					'curly'            => esc_html__( 'Curly', 'Shapes', 'onnat-extra' ),
					'underline'        => esc_html__( 'Underline', 'Shapes', 'onnat-extra' ),
					'double'           => esc_html__( 'Double', 'Shapes', 'onnat-extra' ),
					'double_underline' => esc_html__( 'Double Underline', 'Shapes', 'onnat-extra' ),
					'underline_zigzag' => esc_html__( 'Underline Zigzag', 'Shapes', 'onnat-extra' ),
                ],
                'conditions'         => $this->_title_condition(),
                'frontend_available' => true,
            ]);
            $this->add_control('highlight_ele',[
                'label'      => esc_html__( 'Highlighter', 'onnat-extra' ),
                'type'       => KFCM::CHOOSE,
                'options'    => [
                    'primary'   => [
                        'title' => esc_html__( 'Primary Text', 'onnat-extra' ),
                        'icon'  => 'kfw-ctrl-numeric-icon',
                    ],
                    'secondary' => [
                        'title' => esc_html__( 'Secondary Text', 'onnat-extra' ),
                        'icon'  => 'kfw-ctrl-numeric-2-icon',
                    ],
                    'tertiary'  => [
                        'title' => esc_html__( 'Tertiary Text', 'onnat-extra' ),
                        'icon'  => 'kfw-ctrl-numeric-3-icon',
                    ],
                ],
                'toggle'     => false,
                'default'    => 'secondary',
                'conditions' => $this->_title_condition(),
            ]);
			$this->add_control('loop',[
                'label'              => esc_html__( 'Infinite Loop', 'onnat-extra' ),
                'type'               => KFCM::SWITCHER,
                'default'            => 'yes',
                'frontend_available' => true,
                'selectors'          => [
                    '{{WRAPPER}}' => '--iteration-count: infinite',
                ],
                'conditions'         => $this->_title_condition(),
            ]);
			$this->add_control('anim_duration',[
                'label'              => esc_html__( 'Duration (ms)', 'onnat-extra' ),
                'type'               => KFCM::NUMBER,
                'default'            => 1200,
                'frontend_available' => true,
                'selectors'          => [
					'{{WRAPPER}}' => '--animation-duration: {{VALUE}}ms',
                ],
                'conditions'         => $this->_title_condition(),
            ]);
			$this->add_control('anim_delay',[
                'label'              => esc_html__( 'Delay (ms)', 'onnat-extra' ),
                'type'               => KFCM::NUMBER,
                'default'            => 8000,
                'frontend_available' => true,
                'condition'          => [
                    'loop' => 'yes'
                ],
            ]);
		$this->end_controls_section();
    }

    public function _register_highlighter_style_section() {
		$this->start_controls_section( 'section_highlighter_style_settings', [
            'label'      => esc_html__( 'Highlighter Shape', 'onnat-extra' ),
            'tab'        => KFCM::TAB_STYLE,
            'conditions' => $this->_title_condition(),
		]);
            $this->add_control( 'highlighter_color_settings', [
				'label'        => esc_html__('Colors', 'onnat-extra'),
				'type'         => KFCM::POPOVER_TOGGLE,
				'return_value' => 'yes',
				'conditions'   => $this->_title_condition(),
            ]);
            $this->start_popover();
                $this->add_control('highlighter_normal_color',[
                    'label'      => esc_html__( 'Normal Color', 'onnat-extra' ),
                    'type'       => KFCM::COLOR,
                    'conditions' => $this->_title_condition(),
                    'selectors'  => [
                        '{{WRAPPER}} .kfw-elementor-text-highlighter-wrap path' => 'stroke: {{VALUE}}',
                    ],
                ]);
                $this->add_control('highlighter_hover_color',[
                    'label'      => esc_html__( 'Hover Color', 'onnat-extra' ),
                    'type'       => KFCM::COLOR,
                    'conditions' => $this->_title_condition(),
                    'selectors'  => [
                        '{{WRAPPER}}:hover .kfw-elementor-text-highlighter-wrap path' => 'stroke: {{VALUE}}',
                    ],
                ]);
            $this->end_popover();

			$this->add_control('highlighter_width',[
                'label'      => esc_html__( 'Width', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'size_units' => [ 'px', 'em', ],
                'range'      => [
                    'px' => [
						'min' => 1,
						'max' => 20,
					],
					'em' => [
						'max' => 2,
					],
				],
				'conditions' => $this->_title_condition(),
				'selectors'  => [
					'{{WRAPPER}} .kfw-elementor-text-highlighter-wrap path' => 'stroke-width: {{SIZE}}{{UNIT}}',
				],
            ]);
			$this->add_control('above_content',[
                'label'      => esc_html__( 'Bring to Front', 'onnat-extra' ),
                'type'       => KFCM::SWITCHER,
                'selectors'  => [
					'{{WRAPPER}} .kfw-elementor-text-highlighter-wrap svg' => 'z-index: 2',
					'{{WRAPPER}} .kfw-elementor-text-highlighter-text'     => 'z-index: auto',
                ],
                'conditions' => $this->_title_condition(),
            ]);
			$this->add_control('rounded_edges',[
                'label'      => esc_html__( 'Rounded Edges', 'onnat-extra' ),
                'type'       => KFCM::SWITCHER,
                'selectors'  => [
					'{{WRAPPER}} .kfw-elementor-text-highlighter-wrap path' => 'stroke-linecap: round; stroke-linejoin: round',
                ],
                'conditions' => $this->_title_condition(),
            ]);
		$this->end_controls_section();
    }

    public function _register_subtitle_style_section() {
		$this->start_controls_section( 'section_sub_title_style', [
			'label'     => esc_html__( 'Sub Title', 'onnat-extra' ),
			'tab'       => KFCM::TAB_STYLE,
			'condition' => [ 'sub_title!'=> '' ]
		]);
            $this->add_group_control(KFGCT::get_type(), [
                'name'      => 'sub_title_typo',
                'label'     => esc_html__( 'Typography', 'onnat-extra' ),
                'condition' => [ 'sub_title!'=> '' ],
                'selector'  => '{{WRAPPER}} .kfw-ele-widget-text-highlighter-sub-title',
            ]);
            $this->add_responsive_control( 'sub_title_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'condition'  => [ 'sub_title!'=> '' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-text-highlighter-sub-title-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'sub_title_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'condition'  => [ 'sub_title!'=> '' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-text-highlighter-sub-title-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->start_controls_tabs( 'sub_title_customize_tabs' );
                $this->start_controls_tab( 'sub_title_normal_state', [
                    'label'     => esc_html__( 'Normal', 'onnat-extra' ),
                    'condition' => [ 'sub_title!'=> '' ],
                ]);
                    $this->add_control( 'sub_title_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kfw-ele-widget-text-highlighter-sub-title' => 'color: {{VALUE}};' ],
                        'condition' => [ 'sub_title!'=> '' ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'      => 'sub_title_txt_shadow',
                        'condition' => [ 'sub_title!'=> '' ],
                        'selector'  => '{{WRAPPER}} .kfw-ele-widget-text-highlighter-sub-title',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'      => 'sub_title_txt_stroke',
                        'condition' => [ 'sub_title!'=> '' ],
                        'selector'  => '{{WRAPPER}} .kfw-ele-widget-text-highlighter-sub-title',
                    ]);
                    $this->add_group_control( KFGCBG::get_type(), [
                        'name'      => 'sub_title_bg',
                        'label'     => esc_html__('Background', 'onnat-extra' ),
                        'types'     => [ 'classic', 'gradient' ],
                        'exclude'   => [ 'image' ],# To remove background image option in classic mode
                        'selector'  => '{{WRAPPER}} .kfw-ele-widget-text-highlighter-sub-title-wrap',
                        'condition' => [ 'sub_title!'=> '' ],
                    ]);
                    $this->add_group_control( KFGCBR::get_type(), [
                        'name'           => 'sub_title_border',
                        'fields_options' => [
                            'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
                            'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
                        ],
                        'selector'       => '{{WRAPPER}} .kfw-ele-widget-text-highlighter-sub-title-wrap',
                        'condition'      => [ 'sub_title!'=> '' ],
                    ]);
                    $this->add_responsive_control( 'sub_title_border_radius', [
                        'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                        'type'       => KFCM::DIMENSIONS,
                        'size_units' => [ 'px', '%' ],
                        'selectors'  => [ '{{WRAPPER}} .kfw-ele-widget-text-highlighter-sub-title-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ],
                        'condition'  => [ 'sub_title!'=> '' ],
                    ]);
                    $this->add_group_control( KFGCBS::get_type(), [
                        'name'      => 'sub_title_box_shadow',
                        'selector'  => '{{WRAPPER}} .kfw-ele-widget-text-highlighter-sub-title-wrap',
                        'condition' => [ 'sub_title!'=> '' ],
                    ]);
                $this->end_controls_tab();
				$this->start_controls_tab( 'sub_title_hover_state', [
					'label'     => esc_html__( 'Hover', 'onnat-extra' ),
					'condition' => [ 'sub_title!'=> '' ],
				]);
                    $this->add_control( 'sub_title_hover_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}}:hover .kfw-ele-widget-text-highlighter-sub-title' => 'color: {{VALUE}};' ],
                        'condition' => [ 'sub_title!'=> '' ],
                    ]);
                    $this->add_group_control( KFGCTS::get_type(), [
                        'name'      => 'sub_title_hover_txt_shadow',
                        'condition' => [ 'sub_title!'=> '' ],
                        'selector'  => '{{WRAPPER}}:hover .kfw-ele-widget-text-highlighter-sub-title',
                    ]);
                    $this->add_group_control( KFGCTST::get_type(), [
                        'name'      => 'sub_title_hover_txt_stroke',
                        'condition' => [ 'sub_title!'=> '' ],
                        'selector'  => '{{WRAPPER}}:hover .kfw-ele-widget-text-highlighter-sub-title',
                    ]);
                    $this->add_group_control( KFGCBG::get_type(), [
                        'name'      => 'sub_title_hover_bg',
                        'label'     => esc_html__('Background', 'onnat-extra' ),
                        'types'     => [ 'classic', 'gradient' ],
                        'exclude'   => [ 'image' ],# To remove background image option in classic mode
                        'selector'  => '{{WRAPPER}}:hover .kfw-ele-widget-text-highlighter-sub-title-wrap',
                        'condition' => [ 'sub_title!'=> '' ],
                    ]);
                    $this->add_group_control( KFGCBR::get_type(), [
                        'name'           => 'sub_title_hover_border',
                        'fields_options' => [
                            'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
                            'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
                        ],
                        'selector'       => '{{WRAPPER}}:hover .kfw-ele-widget-text-highlighter-sub-title-wrap',
                        'condition'      => [ 'sub_title!'=> '' ],
                    ]);
                    $this->add_responsive_control( 'sub_title_hover_border_radius', [
                        'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                        'type'       => KFCM::DIMENSIONS,
                        'size_units' => [ 'px', '%' ],
                        'selectors'  => [ '{{WRAPPER}}:hover .kfw-ele-widget-text-highlighter-sub-title-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ],
                        'condition'  => [ 'sub_title!'=> '' ],
                    ]);
                    $this->add_group_control( KFGCBS::get_type(), [
                        'name'      => 'sub_title_hover_box_shadow',
                        'selector'  => '{{WRAPPER}}:hover .kfw-ele-widget-text-highlighter-sub-title-wrap',
                        'condition' => [ 'sub_title!'=> '' ],
                    ]);
                $this->end_controls_tab();
			$this->end_controls_tabs();
		$this->end_controls_section();
    }

    public function _register_title_style_section() {
		$this->start_controls_section( 'section_title_style', [
            'label'      => esc_html__( 'Title', 'onnat-extra' ),
            'tab'        => KFCM::TAB_STYLE,
            'conditions' => $this->_title_condition(),
		]);
            $this->add_responsive_control( 'title_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'conditions' => $this->_title_condition(),
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-text-highlighter-title-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'title_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'conditions' => $this->_title_condition(),
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-text-highlighter-title-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);

            $this->add_control('primary_title_style_before_hr', [
                'type'       => KFCM::DIVIDER,
                'conditions' => $this->_title_condition(),
            ]);
            $this->add_control('primary_title_style_heading',[
                'label'      => esc_html__( 'PRIMARY TEXT', 'onnat-extra' ),
                'type'       => KFCM::HEADING,
                'conditions' => $this->_title_condition(),
            ]);
                $this->add_group_control(KFGCT::get_type(), [
                    'name'       => 'title_typo',
                    'label'      => esc_html__( 'Typography', 'onnat-extra' ),
                    'conditions' => $this->_title_condition(),
                    'selector'   => '{{WRAPPER}} .kfw-elementor-text-highlighter-plain-text',
                ]);
                $this->start_controls_tabs( 'title_customize_tabs' );
                    $this->start_controls_tab( 'title_normal_state', [
                        'label'      => esc_html__( 'Normal', 'onnat-extra' ),
                        'conditions' => $this->_title_condition(),
                    ]);
                        $this->add_control( 'title_color', [
                            'label'      => esc_html__('Color', 'onnat-extra' ),
                            'type'       => KFCM::COLOR,
                            'selectors'  => ['{{WRAPPER}} .kfw-elementor-text-highlighter-plain-text' => 'color: {{VALUE}};' ],
                            'conditions' => $this->_title_condition(),
                        ]);
                        $this->add_group_control( KFGCTS::get_type(), [
                            'name'       => 'title_txt_shadow',
                            'conditions' => $this->_title_condition(),
                            'selector'   => '{{WRAPPER}} .kfw-elementor-text-highlighter-plain-text',
                        ]);
                        $this->add_group_control( KFGCTST::get_type(), [
                            'name'       => 'title_txt_stroke',
                            'conditions' => $this->_title_condition(),
                            'selector'   => '{{WRAPPER}} .kfw-elementor-text-highlighter-plain-text',
                        ]);
                        $this->add_group_control( KFGCBG::get_type(), [
                            'name'       => 'title_bg',
                            'label'      => esc_html__('Background', 'onnat-extra' ),
                            'types'      => [ 'classic', 'gradient' ],
                            'exclude'    => [ 'image' ],# To remove background image option in classic mode
                            'selector'   => '{{WRAPPER}} .kfw-ele-widget-text-highlighter-title-wrap',
                            'conditions' => $this->_title_condition(),
                        ]);
                        $this->add_group_control( KFGCBR::get_type(), [
                            'name'           => 'title_border',
                            'fields_options' => [
                                'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
                                'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
                            ],
                            'selector'       => '{{WRAPPER}} .kfw-ele-widget-text-highlighter-title-wrap',
                            'conditions'     => $this->_title_condition(),
                        ]);
                        $this->add_responsive_control( 'title_border_radius', [
                            'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                            'type'       => KFCM::DIMENSIONS,
                            'size_units' => [ 'px', '%' ],
                            'selectors'  => [ '{{WRAPPER}} .kfw-ele-widget-text-highlighter-title-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ],
                            'conditions' => $this->_title_condition(),
                        ]);
                        $this->add_group_control( KFGCBS::get_type(), [
                            'name'       => 'title_box_shadow',
                            'selector'   => '{{WRAPPER}} .kfw-ele-widget-text-highlighter-title-wrap',
                            'conditions' => $this->_title_condition(),
                        ]);
                    $this->end_controls_tab();
                    $this->start_controls_tab( 'title_hover_state', [
                        'label'      => esc_html__( 'Hover', 'onnat-extra' ),
                        'conditions' => $this->_title_condition(),
                    ]);
                        $this->add_control( 'title_hover_color', [
                            'label'      => esc_html__('Color', 'onnat-extra' ),
                            'type'       => KFCM::COLOR,
                            'selectors'  => ['{{WRAPPER}}:hover .kfw-elementor-text-highlighter-plain-text' => 'color: {{VALUE}};' ],
                            'conditions' => $this->_title_condition(),
                        ]);
                        $this->add_group_control( KFGCTS::get_type(), [
                            'name'       => 'title_hover_txt_shadow',
                            'conditions' => $this->_title_condition(),
                            'selector'   => '{{WRAPPER}}:hover .kfw-elementor-text-highlighter-plain-text',
                        ]);
                        $this->add_group_control( KFGCTST::get_type(), [
                            'name'       => 'title_hover_txt_stroke',
                            'conditions' => $this->_title_condition(),
                            'selector'   => '{{WRAPPER}}:hover .kfw-elementor-text-highlighter-plain-text',
                        ]);
                        $this->add_group_control( KFGCBG::get_type(), [
                            'name'       => 'title_hover_bg',
                            'label'      => esc_html__('Background', 'onnat-extra' ),
                            'types'      => [ 'classic', 'gradient' ],
                            'exclude'    => [ 'image' ],# To remove background image option in classic mode
                            'selector'   => '{{WRAPPER}}:hover .kfw-ele-widget-text-highlighter-title-wrap',
                            'conditions' => $this->_title_condition(),
                        ]);
                        $this->add_group_control( KFGCBR::get_type(), [
                            'name'           => 'title_hover_border',
                            'fields_options' => [
                                'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
                                'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
                            ],
                            'selector'       => '{{WRAPPER}}:hover .kfw-ele-widget-text-highlighter-title-wrap',
                            'conditions'     => $this->_title_condition(),
                        ]);
                        $this->add_responsive_control( 'title_hover_border_radius', [
                            'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                            'type'       => KFCM::DIMENSIONS,
                            'size_units' => [ 'px', '%' ],
                            'selectors'  => [ '{{WRAPPER}}:hover .kfw-ele-widget-text-highlighter-title-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ],
                            'conditions' => $this->_title_condition(),
                        ]);
                        $this->add_group_control( KFGCBS::get_type(), [
                            'name'       => 'title_hover_box_shadow',
                            'selector'   => '{{WRAPPER}}:hover .kfw-ele-widget-text-highlighter-title-wrap',
                            'conditions' => $this->_title_condition(),
                        ]);
                    $this->end_controls_tab();
                $this->end_controls_tabs();
            $this->add_control('secondary_title_style_before_hr', [
                'type'       => KFCM::DIVIDER,
                'conditions' => $this->_title_condition(),
            ]);
            $this->add_control('secondary_title_style_heading',[
                'label'      => esc_html__( 'HIGHLIGHTED TEXT', 'onnat-extra' ),
                'type'       => KFCM::HEADING,
                'conditions' => $this->_title_condition(),
            ]);
                $this->add_group_control(KFGCT::get_type(), [
                    'name'       => 'secondary_title_typo',
                    'label'      => esc_html__( 'Typography', 'onnat-extra' ),
                    'conditions' => $this->_title_condition(),
                    'selector'   => '{{WRAPPER}} .kfw-elementor-text-highlighter-text',
                ]);
                $this->start_controls_tabs( 'secondary_title_customize_tabs' );
                    $this->start_controls_tab( 'secondary_title_normal_state', [
                        'label'      => esc_html__( 'Normal', 'onnat-extra' ),
                        'conditions' => $this->_title_condition(),
                    ]);
                        $this->add_control( 'secondary_title_color', [
                            'label'      => esc_html__('Color', 'onnat-extra' ),
                            'type'       => KFCM::COLOR,
                            'selectors'  => ['{{WRAPPER}} .kfw-elementor-text-highlighter-text' => 'color: {{VALUE}};' ],
                            'conditions' => $this->_title_condition(),
                        ]);
                        $this->add_group_control( KFGCTS::get_type(), [
                            'name'       => 'secondary_title_txt_shadow',
                            'conditions' => $this->_title_condition(),
                            'selector'   => '{{WRAPPER}} .kfw-elementor-text-highlighter-text',
                        ]);
                        $this->add_group_control( KFGCTST::get_type(), [
                            'name'       => 'secondary_title_txt_stroke',
                            'conditions' => $this->_title_condition(),
                            'selector'   => '{{WRAPPER}} .kfw-elementor-text-highlighter-text',
                        ]);
                        $this->add_group_control( KFGCBG::get_type(), [
                            'name'       => 'secondary_title_bg',
                            'label'      => esc_html__('Background', 'onnat-extra' ),
                            'types'      => [ 'classic', 'gradient' ],
                            'exclude'    => [ 'image' ],# To remove background image option in classic mode
                            'selector'   => '{{WRAPPER}} .kfw-elementor-text-highlighter-wrap',
                            'conditions' => $this->_title_condition(),
                        ]);
                        $this->add_group_control( KFGCBR::get_type(), [
                            'name'           => 'secondary_title_border',
                            'fields_options' => [
                                'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
                                'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
                            ],
                            'selector'       => '{{WRAPPER}} .kfw-elementor-text-highlighter-wrap',
                            'conditions'     => $this->_title_condition(),
                        ]);
                        $this->add_responsive_control( 'secondary_title_border_radius', [
                            'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                            'type'       => KFCM::DIMENSIONS,
                            'size_units' => [ 'px', '%' ],
                            'selectors'  => [ '{{WRAPPER}} .kfw-elementor-text-highlighter-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ],
                            'conditions' => $this->_title_condition(),
                        ]);
                        $this->add_group_control( KFGCBS::get_type(), [
                            'name'       => 'secondary_title_box_shadow',
                            'selector'   => '{{WRAPPER}} .kfw-elementor-text-highlighter-wrap',
                            'conditions' => $this->_title_condition(),
                        ]);
                    $this->end_controls_tab();
                    $this->start_controls_tab( 'secondary_title_hover_state', [
                        'label'      => esc_html__( 'Hover', 'onnat-extra' ),
                        'conditions' => $this->_title_condition(),
                    ]);
                        $this->add_control( 'secondary_title_hover_color', [
                            'label'      => esc_html__('Color', 'onnat-extra' ),
                            'type'       => KFCM::COLOR,
                            'selectors'  => ['{{WRAPPER}}:hover .kfw-elementor-text-highlighter-text' => 'color: {{VALUE}};' ],
                            'conditions' => $this->_title_condition(),
                        ]);
                        $this->add_group_control( KFGCTS::get_type(), [
                            'name'       => 'secondary_title_hover_txt_shadow',
                            'conditions' => $this->_title_condition(),
                            'selector'   => '{{WRAPPER}}:hover .kfw-elementor-text-highlighter-text',
                        ]);
                        $this->add_group_control( KFGCTST::get_type(), [
                            'name'       => 'secondary_title_hover_txt_stroke',
                            'conditions' => $this->_title_condition(),
                            'selector'   => '{{WRAPPER}}:hover .kfw-elementor-text-highlighter-text',
                        ]);
                        $this->add_group_control( KFGCBG::get_type(), [
                            'name'       => 'secondary_title_hover_bg',
                            'label'      => esc_html__('Background', 'onnat-extra' ),
                            'types'      => [ 'classic', 'gradient' ],
                            'exclude'    => [ 'image' ],# To remove background image option in classic mode
                            'selector'   => '{{WRAPPER}}:hover .kfw-elementor-text-highlighter-wrap',
                            'conditions' => $this->_title_condition(),
                        ]);
                        $this->add_group_control( KFGCBR::get_type(), [
                            'name'           => 'secondary_title_hover_border',
                            'fields_options' => [
                                'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
                                'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
                            ],
                            'selector'       => '{{WRAPPER}}:hover .kfw-elementor-text-highlighter-wrap',
                            'conditions'     => $this->_title_condition(),
                        ]);
                        $this->add_responsive_control( 'secondary_title_hover_border_radius', [
                            'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                            'type'       => KFCM::DIMENSIONS,
                            'size_units' => [ 'px', '%' ],
                            'selectors'  => [ '{{WRAPPER}}:hover .kfw-elementor-text-highlighter-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ],
                            'conditions' => $this->_title_condition(),
                        ]);
                        $this->add_group_control( KFGCBS::get_type(), [
                            'name'       => 'secondary_title_hover_box_shadow',
                            'selector'   => '{{WRAPPER}}:hover .kfw-elementor-text-highlighter-wrap',
                            'conditions' => $this->_title_condition(),
                        ]);
                    $this->end_controls_tab();
                $this->end_controls_tabs();
        $this->end_controls_section();
    }

    public function _title_condition() {
        return [
            'relation' => 'or',
            'terms'    => [
                [
                    'name'     => 'primary',
                    'operator' => '!=',
                    'value'    => ''
                ],
                [
                    'name'     => 'secondary',
                    'operator' => '!=',
                    'value'    => ''
                ],
                [
                    'name'     => 'tertiary',
                    'operator' => '!=',
                    'value'    => ''
                ],
            ],
        ];
    }

    public function _title_tag( $tag ) {
        $tags = [
            'div',
            'span',
            'p',
            'h1',
            'h2',
            'h3',
            'h4',
            'h5',
            'h6',
        ];

        return in_array( strtolower( $tag ), $tags) ? $tag : 'div';
    }

	public function _sub_title( $settings = [] ) {
		if( !empty( $settings['sub_title'] ) ) {
			$this->add_render_attribute( 'sub-title', [
				'class' => array_filter([
					'kfw-ele-widget-text-highlighter-sub-title'
				])
			]);

			echo '<div class="kfw-ele-widget-text-highlighter-sub-title-wrap">
                    <div '.$this->get_render_attribute_string( 'sub-title' ).'>'
                    . esc_html( $settings['sub_title'] ) .
                    '</div>
                </div>';
        }
    }


    public function _is_editor_mode() {
        $preview = KFEl::instance()->preview->is_preview_mode();
        $editor  = KFEl::instance()->editor->is_edit_mode();

        return ( $preview || $editor ) ? true : false;
    }

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-text-highlighter");
		include $render;
    }
}