<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use \Elementor\Widget_Base as WB;
use \Elementor\Controls_Manager as KFCM;
use \Elementor\Icons_Manager as KFIM;

use \Elementor\Group_Control_Background as KFGCBG;
use \Elementor\Group_Control_Typography as KFGCT;
use \Elementor\Group_Control_Border as KFGCBR;
use \Elementor\Group_Control_Box_Shadow as KFGCBS;

class KFW_Theme_Button extends WB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-theme-button';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Onnat Button', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_theme_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-button';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'kfw', 'kin', 'force', 'onnat', 'button', 'btn' ];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
        return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/onnat-button';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

		wp_register_style( 'kfw-theme-button',
			KFE_CONST_URL . 'libraries/elementor/assets/css/theme-button' . KFE_CONST_DEBUG_SUFFIX . '.css',
			[],
			KFE_CONST_VERSION
		);

        if ( ! wp_script_is( 'kfw-gsap', 'enqueued' ) ) {
            wp_register_script( 'kfw-gsap', KFE_CONST_URL . 'assets/js/gsap.min.js', [], KFE_CONST_VERSION, true );
        }

		wp_register_script( 'kfw-theme-button',
			KFE_CONST_URL . 'libraries/elementor/assets/js/theme-button' . KFE_CONST_DEBUG_SUFFIX . '.js',
			[ 'jquery', 'elementor-frontend' ],
			KFE_CONST_VERSION,
			true
		);

    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'kfw-theme-button' ];
	}

	/**
	 * Get script dependencies.
	 *
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @return array Element scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'kfw-gsap', 'kfw-theme-button' ];
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
        $this->_register_button_section();
        $this->_register_button_icon_wrap_section();
        $this->_register_button_icon_section();
        $this->_register_button_txt_section();
    }

    public function _register_button_section() {
		$this->start_controls_section( 'section_button', [
			'label' => esc_html__( 'Button', 'onnat-extra' )
		]);
            $this->add_control( 'btn_text', [
                'label'     => esc_html__( 'Text', 'onnat-extra' ),
                'type'      => KFCM::TEXT,
                'default'   => esc_html__( 'Button', 'onnat-extra' ),
            ]);
            $this->add_control( 'btn_link', [
                'label'       => esc_html__( 'Link', 'onnat-extra' ),
                'type'        => KFCM::URL,
                'placeholder' => 'https://your-link.com',
                'default'     => [ 'url' => '#' ],
                'conditions'  => [
                    'relation' => 'or',
                    'terms'    => [
                        [
                            'name'     => 'btn_text',
                            'operator' => '!=',
                            'value'    => ''
                        ],
                        [
                            'name'     => 'btn_icon[value]',
                            'operator' => '!=',
                            'value'    => ''
                        ],
                    ]
                ]
            ]);
			$this->add_control( 'btn_icon', [
				'label'       => esc_html__( 'Icon', 'onnat-extra' ),
				'type'        => KFCM::ICONS,
				'skin'        => 'inline',
				'label_block' => false,
				'default'     => [
					'value'   => 'fas fa-slash',
					'library' => 'fa-solid'
				]
			]);
            $this->add_control('magnetic_effect', [
				'label'        => esc_html__( 'Magnetic Effect', 'onnat-extra' ),
				'type'         => KFCM::SWITCHER,
				'label_on'     => esc_html__( 'On', 'onnat-extra' ),
				'label_off'    => esc_html__( 'Off', 'onnat-extra' ),
				'return_value' => 'kfw-ele-button-magnetic-effect',
				'prefix_class' => '',
            ]);
        $this->end_controls_section();
    }

    public function _register_button_icon_wrap_section() {
        $condition = [
            'btn_icon[value]!' => ''
        ];

		$this->start_controls_section( 'section_button_icon_wrap_style', [
            'label'     => esc_html__( 'Icon Wrap', 'onnat-extra' ),
            'tab'       => KFCM::TAB_STYLE,
            'condition' => $condition,
		]);
            $this->add_responsive_control( 'btn_icon_wrap_width', [
                'label'      => esc_html__('Width', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'size_units' => [ 'px' ],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 999,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-theme-button .kfw-elementor-icon' => 'width: {{SIZE}}{{UNIT}};',
                ],
                'condition'  => $condition,
            ]);
            $this->add_responsive_control( 'btn_icon_wrap_height', [
                'label'      => esc_html__('Height', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'size_units' => [ 'px' ],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 999,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-theme-button .kfw-elementor-icon' => 'height: {{SIZE}}{{UNIT}};',
                ],
                'condition' => $condition,
            ]);
            $this->add_responsive_control( 'btn_icon_wrap_border_radius', [
                'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', '%' ],
                'selectors'  => [ '{{WRAPPER}} .kfw-ele-widget-theme-button .kfw-elementor-icon:before, {{WRAPPER}} .kfw-ele-widget-theme-button .kfw-elementor-icon:after' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ],
                'condition' => $condition,
            ]);
            $this->start_controls_tabs( 'btn_icon_wrap_customize_tabs', [
                'condition' => $condition,
            ]);
                $this->start_controls_tab( 'btn_icon_wrap_normal_state', [
                    'label' => esc_html__( 'Normal', 'onnat-extra' ),
                ]);
                    $this->add_group_control( KFGCBG::get_type(), [
                        'name'     => 'btn_icon_wrap_bg',
                        'label'    => esc_html__('Background', 'onnat-extra' ),
                        'types'    => [ 'classic', 'gradient' ],
                        'exclude'  => [ 'image' ],
                        'selector' => '{{WRAPPER}} .kfw-ele-widget-theme-button .kfw-elementor-icon:before',
                    ]);
                    $this->add_group_control( KFGCBR::get_type(), [
                        'name'           => 'btn_icon_wrap_border',
                        'fields_options' => [
                            'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
                            'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
                        ],
                        'selector'       => '{{WRAPPER}} .kfw-ele-widget-theme-button .kfw-elementor-icon:before',
                    ]);
                    $this->add_group_control( KFGCBS::get_type(), [
                        'name'     => 'btn_icon_wrap_box_shadow',
                        'selector' => '{{WRAPPER}} .kfw-ele-widget-theme-button .kfw-elementor-icon:before',
                    ]);
                $this->end_controls_tab();
                $this->start_controls_tab( 'btn_icon_wrap_hover_state', [
                    'label' => esc_html__( 'Hover', 'onnat-extra' ),
                ]);
                    $this->add_control( 'btn_icon_wrap_h_box_shadow_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => [
                            '{{WRAPPER}}'   => '--kfw-icon-wrap-bg-h-color:{{VALUE}};',
                        ],
                    ]);
                    $this->add_control( 'btn_icon_wrap_h_box_shadow_spread', [
                        'label'      => esc_html__('Spread', 'onnat-extra' ),
                        'type'       => KFCM::SLIDER,
                        'size_units' => [ 'px' ],
                        'range'      => [
                            'px' => [
                                'min' => 0,
                                'max' => 100,
                            ],
                        ],
                        'selectors'  => [
                            '{{WRAPPER}}'   => '--kfw-icon-wrap-bg-h-spread:{{SIZE}}{{UNIT}};',
                        ],
                    ]);
                $this->end_controls_tab();
            $this->end_controls_tabs();
        $this->end_controls_section();
    }

    public function _register_button_icon_section() {
        $condition = [
            'btn_icon[value]!' => ''
        ];

		$this->start_controls_section( 'section_button_icon_style', [
            'label'     => esc_html__( 'Icon', 'onnat-extra' ),
            'tab'       => KFCM::TAB_STYLE,
            'condition' => $condition,
		]);
            $this->add_responsive_control( 'btn_icon_size', [
                'label'      => esc_html__('Size', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'size_units' => [ 'px' ],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 999,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-theme-button .kfw-elementor-icon-wrap' => 'font-size: {{SIZE}}{{UNIT}};'
                ],
                'condition'  => $condition,
            ]);
            $this->add_control('icon_rotate',[
                'label'     => esc_html__( 'Rotate', 'onnat-extra' ),
                'type'      => KFCM::SLIDER,
                'default'   => [
                    'size' => 0,
                    'unit' => 'deg',
                ],
                'range'     => [
                    'deg' => [
                        'max' => 360,
                        'min' => -360,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}}' => '--kfw-icon-rotate:{{SIZE}}{{UNIT}};',
                ],
                'condition' => $condition,
            ]);

            $this->add_responsive_control( 'btn_icon_position', [
				'label'              => esc_html__( 'Position', 'onnat-extra' ),
				'type'               => KFCM::DIMENSIONS,
				'size_units'         => [ 'px', 'em', '%' ],
				'allowed_dimensions' => [ 'top', 'left', ],
				'placeholder'        => [
                    'top'    => '',
                    'right'  => 'auto',
                    'bottom' => 'auto',
                    'left'   => '',
                ],
                'selectors'          => [
                    '{{WRAPPER}} .kfw-ele-widget-theme-button .kfw-elementor-icon-wrap' => 'top: {{TOP}}{{UNIT}}; left: {{LEFT}}{{UNIT}};'
                ],
                'condition'          => $condition,
            ]);
            $this->start_controls_tabs( 'btn_icon_customize_tabs' );
                $this->start_controls_tab( 'btn_icon_normal_state', [
                    'label'     => esc_html__( 'Normal', 'onnat-extra' ),
                    'condition' => $condition,
                ]);
                    $this->add_control( 'btn_icon_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => [
                            '{{WRAPPER}} .kfw-ele-widget-theme-button .kfw-elementor-icon-wrap i'   => 'color: {{VALUE}};',
                            '{{WRAPPER}} .kfw-ele-widget-theme-button .kfw-elementor-icon-wrap svg' => 'fill: {{VALUE}};',
                        ],
                        'condition' => $condition,
                    ]);
                $this->end_controls_tab();
                $this->start_controls_tab( 'btn_icon_hover_state', [
                    'label'     => esc_html__( 'Hover', 'onnat-extra' ),
                    'condition' => $condition,
                ]);
                    $this->add_control( 'btn_icon_hover_color', [
                        'label'     => esc_html__('Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => [
                            '{{WRAPPER}}:hover .kfw-ele-widget-theme-button .kfw-elementor-icon-wrap i' => 'color: {{VALUE}};',
                            '{{WRAPPER}}:hover .kfw-ele-widget-theme-button .kfw-elementor-icon-wrap svg' => 'fill: {{VALUE}};',
                        ],
                        'condition' => $condition,
                    ]);
                $this->end_controls_tab();
            $this->end_controls_tabs();
        $this->end_controls_section();
    }

    public function _register_button_txt_section() {
        $condition = [
            'btn_text!' => ''
        ];

		$this->start_controls_section( 'section_button_txt_style', [
            'label'     => esc_html__( 'Text', 'onnat-extra' ),
            'tab'       => KFCM::TAB_STYLE,
            'condition' => $condition,
		]);
            $this->add_control('text_position', [
                'label'          => esc_html__( 'Position', 'onnat-extra' ),
                'type'           => KFCM::CHOOSE,
                'options'        => [
                    'left'   => [
                        'title' => esc_html__( 'Left', 'onnat-extra' ),
                        'icon'  => 'eicon-h-align-left',
                    ],
                    'top'    => [
                        'title' => esc_html__( 'Top', 'onnat-extra' ),
                        'icon'  => 'eicon-v-align-top',
                    ],
                    'right'  => [
                        'title' => esc_html__( 'Right', 'onnat-extra' ),
                        'icon'  => 'eicon-h-align-right',
                    ],
                    'bottom' => [
                        'title' => esc_html__( 'Bottom', 'onnat-extra' ),
                        'icon'  => 'eicon-v-align-bottom',
                    ],
                ],
                'default'        => 'right',
                'toggle'         => false,
                'prefix_class'   => 'kfw-ele-button-text-position-',
                'condition'   => $condition,
            ]);
            $this->add_group_control( KFGCT::get_type(), [
                'name'        => 'btn_txt_typo',
                'description' => esc_html__( 'Typography', 'onnat-extra' ),
                'selector'    => '{{WRAPPER}} .kfw-ele-widget-theme-button .kfw-elementor-button-text',
                'condition'   => $condition,
            ]);
            $this->add_responsive_control( 'btn_txt_margin', [
                'label'      => esc_html__( 'Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-theme-button .kfw-elementor-button-text' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ],
                'condition'  => $condition,
            ]);
            $this->start_controls_tabs( 'btn_txt_customize_tabs' );
                $this->start_controls_tab( 'btn_txt_normal_state', [
                    'label'     => esc_html__( 'Normal', 'onnat-extra' ),
                    'condition' => $condition,
                ]);
                    $this->add_control( 'btn_txt_color', [
                        'label'     => esc_html__('Text Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kfw-ele-widget-theme-button .kfw-elementor-button-text' => 'color: {{VALUE}};' ],
                        'condition' => $condition,
                    ]);
                $this->end_controls_tab();
                $this->start_controls_tab( 'btn_txt_hover_state', [
                    'label'     => esc_html__( 'Hover', 'onnat-extra' ),
                    'condition' => $condition,
                ]);
                    $this->add_control( 'btn_txt_hover_color', [
                        'label'     => esc_html__('Text Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kfw-ele-widget-theme-button-main:hover .kfw-ele-widget-theme-button .kfw-elementor-button-text' => 'color: {{VALUE}};' ],
                        'condition' => $condition,
                    ]);
                    $this->add_control( 'btn_txt_stroke_color', [
                        'label'     => esc_html__('Stoke Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kfw-ele-widget-theme-button .kfw-elementor-button-text:before' => 'background: {{VALUE}};' ],
                        'condition' => $condition,
                    ]);
                $this->end_controls_tab();
            $this->end_controls_tabs();
        $this->end_controls_section();
    }

	public function _render_icon( $icon , $echo = true ) {
		$icon = array_filter( $icon );

		if( !empty( $icon ) ) {
            echo '<span class="kfw-elementor-icon">';
                echo '<span class="kfw-elementor-icon-wrap">';
                    KFIM::render_icon( $icon, [ 'aria-hidden' => 'true' ] );
                echo '</span>';
            echo '</span>';
		}
	}

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-theme-button");
		include $render;
    }
}