<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use \Elementor\Widget_Base as WB;
use \Elementor\Controls_Manager as KFCM;
use \Elementor\Icons_Manager as KFIM;

use \Elementor\Group_Control_Background as KFGCBG;
use \Elementor\Group_Control_Border as KFGCBR;
use \Elementor\Group_Control_Typography as KFGCT;

class KFW_Video_Popup_Button extends WB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-video-popup-button';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Video Popup Button', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_core_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-youtube';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'kfw', 'kin', 'video', 'popup', 'play' ];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
        return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/video-popup-button';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

        if ( ! wp_script_is( 'jquery-magnific-popup', 'enqueued' ) ) {
            wp_register_script( 'jquery-magnific-popup',
                KFE_CONST_URL . 'assets/js/jquery.magnific-popup.min.js',
                [],
                KFE_CONST_VERSION,
                true
            );
        }

        if ( ! wp_style_is( 'jquery-magnific-popup', 'enqueued' ) ) {
            wp_register_style( 'jquery-magnific-popup',
                KFE_CONST_URL . 'assets/css/magnific-popup.css',
                [],
                KFE_CONST_VERSION
            );
        }

		wp_register_script( 'kfw-video-popup-button',
			KFE_CONST_URL . 'libraries/elementor/assets/js/video-popup-button' . KFE_CONST_DEBUG_SUFFIX . '.js',
			[ 'jquery', 'elementor-frontend' ],
			KFE_CONST_VERSION,
			true
		);

		wp_register_style( 'kfw-video-popup-button',
			KFE_CONST_URL . 'libraries/elementor/assets/css/video-popup-button' . KFE_CONST_DEBUG_SUFFIX . '.css',
			[],
			KFE_CONST_VERSION
		);
    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'jquery-magnific-popup',  'kfw-video-popup-button' ];
	}

	/**
	 * Get script dependencies.
	 *
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @return array Element scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'jquery-magnific-popup', 'wp-mediaelement', 'kfw-video-popup-button' ];
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
		$this->_register_handler_settings_section();
		$this->_register_video_settings_section();

		$this->_register_wrapper_style_section();
		$this->_register_icon_style_section();
		$this->_register_text_style_section();
		$this->_register_pulse_effect_section();
    }

	protected function _register_handler_settings_section() {
		$this->start_controls_section('section_handler_settings', [
			'label' => esc_html__('Handler', 'onnat-extra' )
		]);
			$this->add_control('icon_type',[
				'label'   => esc_html__( 'Type', 'onnat-extra' ),
				'type'    => KFCM::CHOOSE,
				'options' => [
					'blank' => [
						'title' => esc_html__('Blank', 'onnat-extra'),
						'icon'  => '',
					],
					'icon'  => [
						'title' => esc_html__('Icon', 'onnat-extra'),
						'icon'  => 'eicon-plus-circle-o',
					],
					'image' => [
						'title' => esc_html__('Image', 'onnat-extra'),
						'icon'  => 'eicon-image',
					],
				],
				'default' => 'icon',
				'toggle'  => false
			]);
			$this->add_control( 'icon', [
				'label'     => esc_html__( 'Icon', 'onnat-extra' ),
				'type'      => KFCM::ICONS,
				'default'   => [
					'value'   => 'fas fa-heart',
					'library' => 'fa-solid',
				],
				'condition' => [ 'icon_type' => 'icon' ],
			]);
			$this->add_control( 'image', [
				'label'     => esc_html__( 'Image', 'onnat-extra' ),
				'type'      => KFCM::MEDIA,
				'dynamic'   => [ 'active' => true ],
				'default'   => [ 'url' => '' ],
				'condition' => [ 'icon_type' => 'image' ],
			]);
            $this->add_control('text',[
                'label'       => esc_html__( 'Text', 'onnat-extra' ),
                'type'        => KFCM::TEXT,
                'dynamic'     => [ 'active' => true ],
                'default'     => '',
                'label_block' => true,
            ]);
			$this->add_control('text_position', [
				'label'          => esc_html__( 'Text Position', 'onnat-extra' ),
				'type'           => KFCM::CHOOSE,
				'options'        => [
					'left'   => [
						'title' => esc_html__( 'Left', 'onnat-extra' ),
						'icon'  => 'eicon-h-align-left',
					],
					'top'    => [
						'title' => esc_html__( 'Top', 'onnat-extra' ),
						'icon'  => 'eicon-v-align-top',
					],
					'right'  => [
						'title' => esc_html__( 'Right', 'onnat-extra' ),
						'icon'  => 'eicon-h-align-right',
					],
					'bottom' => [
						'title' => esc_html__( 'Bottom', 'onnat-extra' ),
						'icon'  => 'eicon-v-align-bottom',
					],
				],
				'default'        => 'right',
				'toggle'         => false,
				'prefix_class'   => 'kfw-ele-widget-video-popup-text-position-',
				'conditions'     => [
					'relation' => 'and',
					'terms'    => [
						[
							'name'     => 'text',
							'operator' => '!==',
							'value'    => ''
						],
						[
							'name'     => 'icon_type',
							'operator' => '!==',
							'value'    => 'blank'
						],
						[
							'relation' => 'or',
							'terms'    => [
								[
									'name'     => 'icon[value]',
									'operator' => '!=',
									'value'    => '',
								],
								[
									'name'     => 'image[url]',
									'operator' => '!=',
									'value'    => '',
								],
							]
						]
					]
				]
			]);
		$this->end_controls_section();
	}

    protected function _register_video_settings_section() {
		$this->start_controls_section('section_video_settings', [
			'label' => esc_html__('Video', 'onnat-extra' )
		]);
			$this->add_control( 'video_type', [
				'label'   => esc_html__( 'Video Type', 'onnat-extra' ),
				'type'    => KFCM::CHOOSE,
				'options' => [
                    'vimeo'   => [
                        'title' => esc_html__('Vimeo', 'onnat-extra'),
                        'icon'  => 'eicon-video-camera'
                    ],
                    'youtube' => [
                        'title' => esc_html__('YouTube', 'onnat-extra'),
                        'icon'  => 'eicon-youtube'
                    ],
                    'media'   => [
                        'title' => esc_html__('Self Hosted', 'onnat-extra'),
                        'icon'  => 'eicon-library-open'
                    ],
                    'url'     => [
                        'title' => esc_html__('Custom URL', 'onnat-extra'),
                        'icon'  => 'eicon-link'
                    ],
                ],
				'default' => 'youtube',
				'toggle'  => false,
            ]);
			$this->add_control( 'vimeo', [
				'label'       => esc_html__( 'Vimeo Video', 'onnat-extra' ),
				'type'        => KFCM::URL,
				'label_block' => true,
				'dynamic'     => [ 'active' => true ],
				'options'     => false,
				'placeholder' => 'https://vimeo.com/9994149897',
				'default'     => [
                    'url' => 'https://vimeo.com/235215203'
                ],
				'condition'   => [
                    'video_type' => 'vimeo'
                ]
            ]);
			$this->add_control( 'youtube', [
				'label'       => esc_html__( 'YouTube Video', 'onnat-extra' ),
				'type'        => KFCM::URL,
				'label_block' => true,
				'dynamic'     => [ 'active' => true ],
				'options'     => false,
				'placeholder' => 'https://www.youtube.com/watch?v=mGsVaDivuPs',
				'default'     => [
                    'url' => 'https://www.youtube.com/watch?v=XHOmBV4js_E'
                ],
				'condition'   => [
                    'video_type' => 'youtube'
                ]
            ]);
			$this->add_control('video_type_des', [
				'type'            => KFCM::RAW_HTML,
				'content_classes' => 'kfw-elementor-editor-notice-ctrl',
				'raw'             => sprintf(
					esc_html__('The supported video formats include %1$sOGV%2$s, %1$sWebM%2$s, and %1$sMP4%2$s','onnat-extra'),
					'<strong>', '</strong>'
				),
				'condition'       => [
                    'video_type' => [ 'media', 'url' ]
                ]
            ]);
			$this->add_control( 'media', [
				'label'       => esc_html__( 'Self Hosted Video', 'onnat-extra' ),
				'type'        => KFCM::MEDIA,
                'media_types' => [ 'video' ],
                'dynamic'     => [ 'active' => true ],
                'default'     => [],
				'condition'   => [
                    'video_type' => 'media'
                ]
            ]);
			$this->add_control( 'url', [
				'label'       => esc_html__( 'External Video', 'onnat-extra' ),
				'type'        => KFCM::URL,
				'label_block' => true,
				'dynamic'     => [ 'active' => true ],
				'options'     => false,
				'placeholder' => '',
				'condition'   => [
                    'video_type' => 'url'
                ]
            ]);
			$this->add_control('start_time_before_hr', [
				'type' => KFCM::DIVIDER,
			]);
			$this->add_control( 'start', [
				'label'       => esc_html__( 'Start Time', 'onnat-extra' ),
				'type'        => KFCM::NUMBER,
				'description' => esc_html__( 'Specify a start time (in seconds)', 'onnat-extra' ),
            ]);
			$this->add_control( 'end', [
				'label'       => esc_html__( 'End Time', 'onnat-extra' ),
				'type'        => KFCM::NUMBER,
				'description' => esc_html__( 'Specify a end time (in seconds)', 'onnat-extra' ),
				'condition'   => [
                    'video_type!' => 'vimeo'
                ]
            ]);
			$this->add_control('end_time_before_hr', [
				'type' => KFCM::DIVIDER,
			]);
			$this->add_control('video_options_heading',[
				'label' => esc_html__( 'VIDEO OPTIONS', 'onnat-extra' ),
				'type'  => KFCM::HEADING,
			]);
			$this->add_control('autoplay', [
				'label'        => esc_html__( 'Auto Play?', 'onnat-extra' ),
				'type'         => KFCM::SWITCHER,
				'return_value' => 'yes',
				'default'      => 'no'
			]);
			$this->add_control('preload', [
				'label'       => esc_html__( 'Preload', 'onnat-extra' ),
				'type'        => KFCM::SELECT,
				'options'     => [
					'metadata' => esc_html__( 'Metadata', 'onnat-extra' ),
					'auto'     => esc_html__( 'Auto', 'onnat-extra' ),
					'none'     => esc_html__( 'None', 'onnat-extra' ),
				],
				'description' => esc_html__( 'Preload attribute lets you specify how the video should be loaded when the page loads', 'onnat-extra' ),
				'default'     => 'metadata',
				'condition'   => [
					'video_type' => [ 'media', 'url' ],
					'autoplay'   => ''
				]
			]);
			$this->add_control('mute', [
				'label'        => esc_html__( 'Mute?', 'onnat-extra' ),
				'type'         => KFCM::SWITCHER,
				'return_value' => 'yes',
				'default'      => 'no'
			]);
			$this->add_control('loop', [
				'label'        => esc_html__( 'Loop?', 'onnat-extra' ),
				'type'         => KFCM::SWITCHER,
				'return_value' => 'yes',
				'default'      => 'no'
			]);
			$this->add_control('controls', [
                'label'        => esc_html__( 'Player Controls', 'onnat-extra' ),
                'type'         => KFCM::SWITCHER,
                'label_on'     => esc_html__( 'On', 'onnat-extra' ),
                'label_off'    => esc_html__( 'Off', 'onnat-extra' ),
                'return_value' => 'yes',
                'default'      => 'yes',
			]);
			$this->add_control('vimeo_title', [
                'label'        => esc_html__( 'Intro Title', 'onnat-extra' ),
                'type'         => KFCM::SWITCHER,
                'label_on'     => esc_html__( 'Show', 'onnat-extra' ),
                'label_off'    => esc_html__( 'Hide', 'onnat-extra' ),
                'return_value' => 'yes',
                'default'      => 'yes',
                'condition'    => [
					'video_type' => [ 'vimeo' ],
					'controls'   => 'yes',
                ]
			]);
			$this->add_control('vimeo_portrait', [
                'label'        => esc_html__( 'Intro Portrait', 'onnat-extra' ),
                'type'         => KFCM::SWITCHER,
                'label_on'     => esc_html__( 'Show', 'onnat-extra' ),
                'label_off'    => esc_html__( 'Hide', 'onnat-extra' ),
                'return_value' => 'yes',
                'default'      => 'yes',
                'condition'    => [
					'video_type' => [ 'vimeo' ],
					'controls'   => 'yes',
                ]
			]);
			$this->add_control('vimeo_byline', [
                'label'        => esc_html__( 'Intro Byline', 'onnat-extra' ),
                'type'         => KFCM::SWITCHER,
                'label_on'     => esc_html__( 'Show', 'onnat-extra' ),
                'label_off'    => esc_html__( 'Hide', 'onnat-extra' ),
                'return_value' => 'yes',
                'default'      => 'yes',
                'condition'    => [
					'video_type' => [ 'vimeo' ],
					'controls'   => 'yes',

                ]
			]);
			$this->add_control('control_color', [
                'label'     => esc_html__( 'Controls Color', 'onnat-extra' ),
                'type'      => KFCM::COLOR,
                'default'   => '',
                'condition' => [
					'video_type' => [ 'vimeo' ],
					'controls'   => 'yes',
                ]
			]);
			$this->add_control('poster',[
				'label'     => esc_html__( 'Poster', 'onnat-extra' ),
				'type'      => KFCM::MEDIA,
				'dynamic'   => [ 'active' => true,],
				'condition' => [
					'video_type' => [ 'media', 'url' ],
				],
			]);
		$this->end_controls_section();
    }

	protected function _register_wrapper_style_section() {

		$conditions= [
			'relation' => 'or',
			'terms'    => [
				[
					'name'     => 'text',
					'operator' => '!==',
					'value'    => ''
				],
				[
					'relation' => 'or',
					'terms'    => [
						[
							'name'     => 'icon[value]',
							'operator' => '!=',
							'value'    => '',
						],
						[
							'name'     => 'image[url]',
							'operator' => '!=',
							'value'    => '',
						],
					]
				]
			]
		];

		$this->start_controls_section('section_handler_style_settings', [
			'label'      => esc_html__('Wrapper', 'onnat-extra' ),
			'tab'        => KFCM::TAB_STYLE,
			'conditions' => $conditions
		]);
			$this->add_responsive_control('align', [
				'label'          => esc_html__( 'Alignment', 'onnat-extra' ),
				'type'           => KFCM::CHOOSE,
				'options'        => [
					'left'   => [
						'title' => esc_html__( 'Left', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-center',
					],
					'right'  => [
						'title' => esc_html__( 'Right', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-right',
					],
				],
				'default'              => 'center',
				'tablet_default'       => 'center',
				'mobile_default'       => 'center',
				'toggle'               => false,
				'selectors_dictionary' => [
					'left'   => 'justify-content:start;',
					'center' => 'justify-content:center;',
					'right'  => 'justify-content:end;',
				],
				'selectors'            => [
                    '{{WRAPPER}} .kfw-ele-widget-video-popup-button-main' => '{{VALUE}};',
				],
				'conditions'           => $conditions,
			]);
			$this->add_responsive_control( 'handler_margin', [
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'conditions' => $conditions,
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-video-popup-button-main' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_responsive_control( 'handler_padding', [
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'conditions' => $conditions,
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-video-popup-button-main a' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->start_controls_tabs( 'handler_customize_tabs' );
				$this->start_controls_tab( 'handler_normal_state', [
					'label'     => esc_html__( 'Normal', 'onnat-extra' ),
					'conditions' => $conditions
				]);
					$this->add_group_control( KFGCBG::get_type(), [
						'name'           => 'handler_bg',
						'types'          => ['classic', 'gradient'],
						'exclude'        => [ 'image'],
						'fields_options' => [
							'background' => [ 'default' => 'classic' ],
							'color'      => [ 'default' => '' ],
						],
						'conditions'     => $conditions,
						'selector'       => '{{WRAPPER}} .kfw-ele-widget-video-popup-button-main a',
					]);
					$this->add_control('handler_bg_rotate',[
						'label'     => esc_html__( 'Background Rotate', 'onnat-extra' ),
						'type'      => KFCM::SLIDER,
						'default'   => [
							'size' => 0,
							'unit' => 'deg',
						],
						'range'     => [
							'deg' => [
								'max' => 360,
								'min' => -360,
							],
						],
						'condition' => [
							'handler_bg_background' => [ 'classic', 'gradient' ],
						],
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-video-popup-button-main a' => 'transform: rotate({{SIZE}}{{UNIT}});',
						],
					]);
					$this->add_group_control( KFGCBR::get_type(), [
						'name'      => 'handler_border',
						'conditions' => $conditions,
						'selector'  => '{{WRAPPER}} .kfw-ele-widget-video-popup-button-main a'
					]);
					$this->add_responsive_control( 'handler_border_radius', [
						'label'      => esc_html__('Border Radius', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', '%' ],
						'conditions' => $conditions,
						'selectors'  => [
							'{{WRAPPER}}' => '--kfw-video-button-border-radius:{{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
						]
					]);
				$this->end_controls_tab();
				$this->start_controls_tab( 'handler_hover_state', [
					'label'      => esc_html__( 'Hover', 'onnat-extra' ),
					'conditions' => $conditions
				]);
					$this->add_group_control( KFGCBG::get_type(), [
						'name'           => 'handler_hover_bg',
						'types'          => ['classic', 'gradient'],
						'exclude'        => [ 'image'],
						'fields_options' => [
							'background' => [ 'default' => 'classic' ],
							'color'      => [ 'default' => '' ],
						],
						'conditions'     => $conditions,
						'selector'       => '{{WRAPPER}} .kfw-ele-widget-video-popup-button-main a:hover',
					]);
					$this->add_control('handler_hover_bg_rotate',[
						'label'     => esc_html__( 'Background Rotate', 'onnat-extra' ),
						'type'      => KFCM::SLIDER,
						'default'   => [
							'size' => 0,
							'unit' => 'deg',
						],
						'range'     => [
							'deg' => [
								'max' => 360,
								'min' => -360,
							],
						],
						'condition' => [
							'handler_bg_background' => [ 'classic', 'gradient' ],
						],
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-video-popup-button-main a:hover' => 'transform: rotate({{SIZE}}{{UNIT}});',
						],
					]);
					$this->add_group_control( KFGCBR::get_type(), [
						'name'      => 'handler_hover_border',
						'conditions' => $conditions,
						'selector'  => '{{WRAPPER}} .kfw-ele-widget-video-popup-button-main a:hover'
					]);
					$this->add_responsive_control( 'icon_hover_border_radius', [
						'label'      => esc_html__('Border Radius', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', '%' ],
						'conditions' => $conditions,
						'selectors'  => [
							'{{WRAPPER}}' => '--kfw-video-button-hover-border-radius:{{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
						]
					]);
				$this->end_controls_tab();
			$this->end_controls_tabs();

			$this->add_control('use_height_and_width_hr', [
				'type'       => KFCM::DIVIDER,
				'conditions' => $conditions
			]);
			$this->add_control('use_height_and_width',[
				'label'        => esc_html__( 'Use height and width', 'onnat-extra' ),
				'type'         => KFCM::SWITCHER,
				'label_on'     => esc_html__( 'Yes', 'onnat-extra' ),
				'label_off'    => esc_html__( 'No', 'onnat-extra' ),
				'return_value' => 'yes',
				'default'      => 'yes',
				'conditions'   => $conditions,
			]);
			$this->add_responsive_control( 'main_height', [
				'label'      => esc_html__('Height', 'onnat-extra' ),
				'type'       => KFCM::SLIDER,
				'size_units' => [ 'px', '%' ],
				'range'      => [
					'px' => [
						'min' => 5,
						'max' => 200
					],
					'%'  => [
						'min' => 5,
						'max' => 100
					],
				],
				'condition' => [
					'use_height_and_width' => 'yes'
				],
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-video-popup-button-main a' => 'height:{{SIZE}}{{UNIT}};'
				]
			]);
			$this->add_responsive_control( 'main_width', [
				'label'      => esc_html__('Width', 'onnat-extra' ),
				'type'       => KFCM::SLIDER,
				'size_units' => [ 'px', '%' ],
				'range'      => [
					'px' => [
						'min' => 5,
						'max' => 200
					],
					'%'  => [
						'min' => 5,
						'max' => 100
					],
				],
				'condition' => [
					'use_height_and_width' => 'yes'
				],
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-video-popup-button-main a' => 'width:{{SIZE}}{{UNIT}};'
				]
			]);
		$this->end_controls_section();
	}

	protected function _register_icon_style_section() {
		$conditions = [
			'relation' => 'and',
			'terms'    => [
				[
					'name'     => 'icon_type',
					'operator' => '!==',
					'value'    => 'blank'
				],
				[
					'relation' => 'or',
					'terms'    => [
						[
							'name'     => 'icon[value]',
							'operator' => '!=',
							'value'    => '',
						],
						[
							'name'     => 'image[url]',
							'operator' => '!=',
							'value'    => '',
						],
					]
				],
			]
		];

		$icon_condition = [
			'icon_type'    => 'icon',
			'icon[value]!' => ''
		];

		$img_condition = [
			'icon_type'   => 'image',
			'image[url]!' => ''
		];

		$this->start_controls_section('section_icon_style_settings', [
			'label'      => esc_html__('Icon', 'onnat-extra' ),
			'tab'        => KFCM::TAB_STYLE,
			'conditions' => $conditions
		]);
			$this->add_responsive_control( 'icon_size', [
				'label'      => esc_html__('Icon Size', 'onnat-extra' ),
				'type'       => KFCM::SLIDER,
				'size_units' => [ 'px' ],
				'range'      => [
					'px' => [
						'min' => 0,
						'max' => 200,
					],
				],
				'condition'  => $icon_condition,
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-video-popup-button-main .kfw-elementor-icon-wrap' => 'font-size: {{SIZE}}{{UNIT}};'
				],
			]);
			$this->add_responsive_control( 'img_size', [
				'label'      => esc_html__('Image Size', 'onnat-extra' ),
				'type'       => KFCM::SLIDER,
				'size_units' => ['%'],
				'range'      => [
					'%' => [
						'min' => 5,
						'max' => 100,
					],
				],
				'condition'  => $img_condition,
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-video-popup-button-main .kfw-elementor-icon-wrap.kfw-elementor-icon-image-type' => 'max-width: {{SIZE}}%;'
				],
			]);
			$this->start_controls_tabs( 'icon_customize_tabs' );
				$this->start_controls_tab( 'icon_normal_state', [
					'label'     => esc_html__( 'Normal', 'onnat-extra' ),
					'condition' => $icon_condition
				]);
					$this->add_control( 'icon_color', [
						'label'     => esc_html__('Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'condition' => $icon_condition,
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-video-popup-button-main .kfw-elementor-icon-wrap i' => 'color: {{VALUE}};',
							'{{WRAPPER}} .kfw-ele-widget-video-popup-button-main .kfw-elementor-icon-wrap svg' => 'fill: {{VALUE}};',
						],
					]);
					$this->add_control('icon_rotate',[
						'label'     => esc_html__( 'Rotate', 'onnat-extra' ),
						'type'      => KFCM::SLIDER,
						'default'   => [
							'size' => 0,
							'unit' => 'deg',
						],
						'range'     => [
							'deg' => [
								'max' => 360,
								'min' => - 360,
							],
						],
						'condition' => $icon_condition,
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-video-popup-button-main .kfw-elementor-icon-wrap i'   => 'transform: rotate({{SIZE}}{{UNIT}});',
							'{{WRAPPER}} .kfw-ele-widget-video-popup-button-main .kfw-elementor-icon-wrap svg' => 'transform: rotate({{SIZE}}{{UNIT}});',
						],
					]);
				$this->end_controls_tab();
				$this->start_controls_tab( 'icon_hover_state', [
					'label'     => esc_html__( 'Hover', 'onnat-extra' ),
					'condition' => $icon_condition,
				]);
					$this->add_control( 'icon_hover_color', [
						'label'     => esc_html__('Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'condition' => $icon_condition,
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-video-popup-button-main .kfw-ele-widget-video-popup-button-cta:hover .kfw-elementor-icon-wrap i' => 'color: {{VALUE}};',
							'{{WRAPPER}} .kfw-ele-widget-video-popup-button-main .kfw-ele-widget-video-popup-button-cta:hover .kfw-elementor-icon-wrap svg' => 'fill: {{VALUE}};',
						],
					]);
					$this->add_control('icon_hover_rotate',[
						'label'     => esc_html__( 'Rotate', 'onnat-extra' ),
						'type'      => KFCM::SLIDER,
						'default'   => [
							'size' => 0,
							'unit' => 'deg',
						],
						'range'     => [
							'deg' => [
								'max' => 360,
								'min' => - 360,
							],
						],
						'condition' => $icon_condition,
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-video-popup-button-main .kfw-ele-widget-video-popup-button-cta:hover .kfw-elementor-icon-wrap i'   => 'transform: rotate({{SIZE}}{{UNIT}});',
							'{{WRAPPER}} .kfw-ele-widget-video-popup-button-main .kfw-ele-widget-video-popup-button-cta:hover .kfw-elementor-icon-wrap svg' => 'transform: rotate({{SIZE}}{{UNIT}});',
						],
					]);
				$this->end_controls_tab();
			$this->end_controls_tabs();
			$this->start_controls_tabs( 'img_customize_tabs' );
				$this->start_controls_tab( 'img_normal_state', [
					'label'     => esc_html__( 'Normal', 'onnat-extra' ),
					'condition' => $img_condition,
				]);
					$this->add_control('img_rotate',[
						'label'     => esc_html__( 'Rotate', 'onnat-extra' ),
						'type'      => KFCM::SLIDER,
						'default'   => [
							'size' => 0,
							'unit' => 'deg',
						],
						'range'     => [
							'deg' => [
								'max' => 360,
								'min' => - 360,
							],
						],
						'condition' => $img_condition,
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-video-popup-button-main .kfw-elementor-icon-wrap.kfw-elementor-icon-image-type img' => 'transform: rotate({{SIZE}}{{UNIT}});',
						],
					]);
				$this->end_controls_tab();
				$this->start_controls_tab( 'img_hover_state', [
					'label'     => esc_html__( 'Hover', 'onnat-extra' ),
					'condition' => $img_condition,
				]);
					$this->add_control('img_hover_rotate',[
						'label'     => esc_html__( 'Rotate', 'onnat-extra' ),
						'type'      => KFCM::SLIDER,
						'default'   => [
							'size' => 0,
							'unit' => 'deg',
						],
						'range'     => [
							'deg' => [
								'max' => 360,
								'min' => - 360,
							],
						],
						'condition' => $img_condition,
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-video-popup-button-main .kfw-ele-widget-video-popup-button-cta:hover .kfw-elementor-icon-wrap.kfw-elementor-icon-image-type img' => 'transform: rotate({{SIZE}}{{UNIT}});',
						],
					]);
				$this->end_controls_tab();
			$this->end_controls_tabs();
		$this->end_controls_section();
	}

	protected function _register_text_style_section() {
		$this->start_controls_section('section_text_style_settings', [
			'label'     => esc_html__('Text', 'onnat-extra' ),
			'tab'       => KFCM::TAB_STYLE,
			'condition' => [
				'text!' => ''
			]
		]);
			$this->add_group_control( KFGCT::get_type(), [
				'name'        => 'txt_typo',
				'description' => esc_html__( 'Typography', 'onnat-extra' ),
				'selector'    => '{{WRAPPER}} .kfw-elementor-title-wrap',
				'condition'   => [  'text!' => '' ]
			]);
			$this->add_control( 'txt_color_settings', [
				'label'        => esc_html__('Colors', 'onnat-extra'),
				'type'         => KFCM::POPOVER_TOGGLE,
				'return_value' => 'yes',
			]);
			$this->start_popover();
				$this->add_control( 'txt_color', [
					'label'     => esc_html__('Color', 'onnat-extra' ),
					'type'      => KFCM::COLOR,
					'selectors' => [
						'{{WRAPPER}} .kfw-elementor-title-wrap' => 'color:{{VALUE}};',
					],
				]);
				$this->add_control( 'txt_h_color', [
					'label'     => esc_html__('Hover Color', 'onnat-extra' ),
					'type'      => KFCM::COLOR,
					'selectors' => [
						'{{WRAPPER}} .kfw-ele-widget-video-popup-button-main .kfw-ele-widget-video-popup-button-cta:hover .kfw-elementor-title-wrap' => 'color:{{VALUE}};',
					],
				]);
			$this->end_popover();
		$this->end_controls_section();
	}

	protected function _register_pulse_effect_section() {
		$this->start_controls_section('section_pulse_effect_settings', [
			'label' => esc_html__('Pulse Effect', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_control('pulse_effect',[
				'label'        => esc_html__( 'Enable it?', 'onnat-extra' ),
				'type'         => KFCM::SWITCHER,
				'label_on'     => esc_html__( 'Yes', 'onnat-extra' ),
				'label_off'    => esc_html__( 'No', 'onnat-extra' ),
				'return_value' => 'yes',
				'default'      => 'yes',
			]);
			$this->add_control( 'pulse_color_settings', [
				'label'        => esc_html__('Colors', 'onnat-extra'),
				'type'         => KFCM::POPOVER_TOGGLE,
				'return_value' => 'yes',
				'condition'    => [
					'pulse_effect' => 'yes'
				],
			]);
				$this->start_popover();
					$this->add_control( 'pulse_color', [
						'label'     => esc_html__('Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => ['
							{{WRAPPER}} .kfw-ele-widget-video-popup-button-cta:after,
							{{WRAPPER}} .kfw-ele-widget-video-popup-button-cta:before,
							{{WRAPPER}} .kfw-ele-widget-video-popup-button-cta .kfw-elementor-icon-wrap:after' => 'color:{{VALUE}};',
						],
						'condition' => [
							'pulse_effect' => 'yes'
						],
					]);
					$this->add_control( 'pulse_h_color', [
						'label'     => esc_html__('Hover Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => ['
							{{WRAPPER}} .kfw-ele-widget-video-popup-button-cta:hover:after,
							{{WRAPPER}} .kfw-ele-widget-video-popup-button-cta:hover:before,
							{{WRAPPER}} .kfw-ele-widget-video-popup-button-cta:hover .kfw-elementor-icon-wrap:after' => 'color:{{VALUE}};',
						],
						'condition' => [
							'pulse_effect' => 'yes'
						],
					]);
				$this->end_popover();
			$this->add_responsive_control( 'pulse_size', [
				'label'     => esc_html__('Size', 'onnat-extra' ),
				'type'      => KFCM::SLIDER,
				'range'     => [ 'px' => [  'min' => 5, 'max' => 200  ] ],
				'condition' => [ 'pulse_effect' => 'yes' ],
				'selectors' => [
					'{{WRAPPER}}' => '--kfw-glow-size:{{SIZE}}{{UNIT}};',
				]
			]);
		$this->end_controls_section();
	}

    public function _vimeo_button( $settings = [] ) {
        if( 'vimeo' !== $settings['video_type'] ) {
            return;
        }

        $params = [
            'responsive' => 1,
            'autoplay'   => filter_var( $settings['autoplay'], FILTER_VALIDATE_BOOLEAN ),
            'muted'      => filter_var( $settings['mute'], FILTER_VALIDATE_BOOLEAN ),
            'loop'       => filter_var( $settings['loop'], FILTER_VALIDATE_BOOLEAN ),
        ];

        if( filter_var( $settings['controls'], FILTER_VALIDATE_BOOLEAN ) ) {
            $params['title']    = filter_var( $settings['vimeo_title'], FILTER_VALIDATE_BOOLEAN );
            $params['portrait'] = filter_var( $settings['vimeo_portrait'], FILTER_VALIDATE_BOOLEAN );
            $params['byline']   = filter_var( $settings['vimeo_byline'], FILTER_VALIDATE_BOOLEAN );

            if( !empty( $settings['control_color'] ) ) {
                $params['color'] = str_replace( '#', '', $settings['control_color'] );
            }
        } else {
            $params['controls'] = 0;
        }

        $options = [];
        if( !empty( $settings['start'] ) ) {
            $options['start'] = $settings['start'];
        }

        $vimeo     = $settings['vimeo']['url'];
        $video_url = Elementor\Embed::get_embed_url( $vimeo, $params, $options );

        $this->add_render_attribute( 'cta', 'href', $video_url );
        $this->add_render_attribute( 'cta', [
            'class' => array_filter([ 'kfw-ele-widget-video-popup-button-cta', 'kfw-ele-widget-video-popup-button-iframe', 'kfw-ele-widget-video-popup-button-vimeo' ])
        ]);

		$this->add_render_attribute( 'cta', 'data-popup',
			wp_json_encode( [
				'type'            => 'iframe',
				'mainClass'       => 'mfp-fade',
				'removalDelay'    => 160,
				'preloader'       => false,
				'fixedContentPos' => false,
				'closeBtnInside'  => false,
			] )
		);

        echo '<a '.$this->get_render_attribute_string( 'cta' ).'>';
			$this->_render_button_txt( $settings );
        echo '</a>';
    }

	public function _youtube_button( $settings = [] ) {

        if( 'youtube' !== $settings['video_type'] ) {
            return;
        }

		$params = [
			'autoplay' => filter_var( $settings['autoplay'], FILTER_VALIDATE_BOOLEAN ),
			'mute'     => filter_var( $settings['mute'], FILTER_VALIDATE_BOOLEAN ),
			'loop'     => filter_var( $settings['loop'], FILTER_VALIDATE_BOOLEAN ),
			'start'    => !empty( $settings['start'] ) ? $settings['start'] : '',
			'end'      => !empty( $settings['end'] ) ? $settings['end'] : '',
        ];

		$params['controls'] = filter_var( $settings['controls'], FILTER_VALIDATE_BOOLEAN ) ? 1 : 0;

        $youtube   = $settings['youtube']['url'];
        $video_url = Elementor\Embed::get_embed_url( $youtube, $params, [ 'privacy' => true ] );

        $this->add_render_attribute( 'cta', 'href', $video_url );
        $this->add_render_attribute( 'cta', [
            'class' => array_filter([ 'kfw-ele-widget-video-popup-button-cta', 'kfw-ele-widget-video-popup-button-iframe', 'kfw-ele-widget-video-popup-button-youtube' ])
        ]);

		$this->add_render_attribute( 'cta', 'data-popup',
			wp_json_encode( [
				'type'            => 'iframe',
				'mainClass'       => 'mfp-fade',
				'removalDelay'    => 160,
				'preloader'       => false,
				'fixedContentPos' => false,
				'closeBtnInside'  => false,
			] )
		);

        echo '<a '.$this->get_render_attribute_string( 'cta' ).'>';
			$this->_render_button_txt( $settings );
        echo '</a>';

	}

	public function _external_button( $settings = [] ) {
        if( 'url' !== $settings['video_type'] && empty( $settings['url']['url'] ) ) {
            return;
        }
		$id     = "kfw-ele-widget-video-popup-inline-video-" . $this->get_id();
		$params = [
			'url'   => $settings['url']['url'],
			'start' => '',
			'end'   => '',
		];

		$video_url = $this->_get_video_url( $params );

		if ( empty( $video_url ) ) {
			return;
		}

        $this->add_render_attribute( 'cta', 'href', '#'.$id );
		$this->add_render_attribute( 'cta', [
            'class' => array_filter([ 'kfw-ele-widget-video-popup-button-cta', 'kfw-ele-widget-video-popup-button-inline', 'kfw-ele-widget-video-popup-button-external' ])
        ]);

		$this->add_render_attribute( 'cta', 'data-popup',
			wp_json_encode( [
				'type'            => 'inline',
				'mainClass'       => 'kfw-ele-widget-video-popup-inline-video',
				'removalDelay'    => 160,
				'preloader'       => false,
				'fixedContentPos' => false,
				'closeBtnInside'  => false,
			] )
		);

        echo '<a '.$this->get_render_attribute_string( 'cta' ).'>';
			$this->_render_button_txt( $settings );
        echo '</a>';

		$params = $this->_get_video_params( $settings );


		echo '<div id="'.esc_attr( $id ).'" class="kfw-ele-widget-video-popup-inline-video mfp-hide">';
			echo '<video src="'. $video_url .'" '. $params .'></video>';
		echo '</div>';
	}

	public function _media_button( $settings = [] ) {
        if( 'media' !== $settings['video_type'] && empty( $settings['media']['url'] ) ) {
            return;
        }

		$id     = "kfw-ele-widget-video-popup-inline-video-" . $this->get_id();
		$params = [
			'url'   => $settings['media']['url'],
			'start' => '',
			'end'   => '',
		];

		$video_url = $this->_get_video_url( $params );

		if ( empty( $video_url ) ) {
			return;
		}

        $this->add_render_attribute( 'cta', 'href', '#'.$id );
		$this->add_render_attribute( 'cta', [
            'class' => array_filter([ 'kfw-ele-widget-video-popup-button-cta', 'kfw-ele-widget-video-popup-button-inline', 'kfw-ele-widget-video-popup-button-external' ])
        ]);

		$this->add_render_attribute( 'cta', 'data-popup',
			wp_json_encode( [
				'type'            => 'inline',
				'mainClass'       => 'kfw-ele-widget-video-popup-inline-video',
				'removalDelay'    => 160,
				'preloader'       => false,
				'fixedContentPos' => false,
				'closeBtnInside'  => false,
			] )
		);

        echo '<a '.$this->get_render_attribute_string( 'cta' ).'>';
			$this->_render_button_txt( $settings );
        echo '</a>';

		$params = $this->_get_video_params( $settings );


		echo '<div id="'.esc_attr( $id ).'" class="kfw-ele-widget-video-popup-inline-video mfp-hide">';
			echo '<video src="'. $video_url .'" '. $params .'></video>';
		echo '</div>';
	}

	public function _get_video_url( $settings ) {
		if( empty( $settings['url'] ) ) {
			return '';
		}

		$video_url = $settings['url'];

		if ( $settings['start'] || $settings['end'] ) {
			$video_url .= '#t=';
		}

		if ( $settings['start'] ) {
			$video_url .= $settings['start'];
		}

		if ( $settings['end'] ) {
			$video_url .= ',' . $settings['end'];
		}

		return $video_url;
	}

	public function _get_video_params( $settings ) {
		$controls = [];
		$params   = [
			'autoplay' => filter_var( $settings['autoplay'], FILTER_VALIDATE_BOOLEAN ) ? 'autoplay': '',
			'muted'    => filter_var( $settings['mute'], FILTER_VALIDATE_BOOLEAN ) ? 'muted': '',
			'loop'     => filter_var( $settings['loop'], FILTER_VALIDATE_BOOLEAN ) ? 'loop': '',
			'controls' => filter_var( $settings['controls'], FILTER_VALIDATE_BOOLEAN ) ? 'controls': '',
			'preload'  => $settings['preload'],
			'poster'   => $settings['poster']['url'] ? $settings['poster']['url'] : ''
		];

		$params = array_filter( $params );

		foreach( $params as $attr_key => $attr_val ) {
			$controls[] = sprintf( '%1$s="%2$s"', $attr_key, esc_attr( $attr_val ) );
		}

		return implode( ' ', $controls );
	}

	public function _render_icon( $icon , $echo = true ) {
		$icon = array_filter( $icon );

		if( !empty( $icon ) ) {
			echo '<span class="kfw-elementor-icon-wrap">';
				KFIM::render_icon( $icon, [ 'aria-hidden' => 'true' ] );
			echo '</span>';
		}
	}

	public function _render_img( $settings = [] ) {
		if( !empty( $settings['image']['url'] ) ) {
			$image_src = $settings['image']['url'];

			$image_title = $this->_get_image_title( $settings['image'] );
			$image_title = empty( $image_title ) ? $this->get_title() : $image_title;

			$image_alt = $this->_get_image_alt( $settings['image'] );
			$image_alt = empty( $image_alt ) ? $image_title : $image_title;

			$this->add_render_attribute( 'image_attr', [
				'alt'   => esc_attr( $image_alt ),
				'title' => esc_attr( $image_title ),
				'src'   => esc_url( $image_src ),
			] );

			echo '<span class="kfw-elementor-icon-wrap kfw-elementor-icon-image-type">';
				echo '<img '.$this->get_render_attribute_string( 'image_attr' ).'/>';
			echo '</span>';
		}
	}

	public function _get_image_title( $attachment ) {
		if ( empty( $attachment['id'] ) ) {
			return '';
		}

		return get_the_title( $attachment['id'] );
	}

	public function _get_image_alt( $instance ) {
		if ( empty( $instance['id'] ) ) {
			// For `Insert From URL` images.
			return isset( $instance['alt'] ) ? trim( strip_tags( $instance['alt'] ) ) : '';
		}

		$attachment_id = $instance['id'];
		if ( ! $attachment_id ) {
			return '';
		}

		$attachment = get_post( $attachment_id );
		if ( ! $attachment ) {
			return '';
		}

		$alt = get_post_meta( $attachment_id, '_wp_attachment_image_alt', true );
		if ( ! $alt ) {
			$alt = $attachment->post_excerpt;
			if ( ! $alt ) {
				$alt = $attachment->post_title;
			}
		}
		return trim( strip_tags( $alt ) );
	}

	public function _render_title( $text = '' ) {
		if( !empty( $text ) ) {
			echo '<span class="kfw-elementor-title-wrap">';
				echo esc_html( $text );
			echo '</span>';
		}
	}

	public function _render_button_txt( $settings = []) {

		if( 'icon' === $settings['icon_type'] ) {
			$this->_render_icon( $settings['icon'] );
		} else if( 'image' === $settings['icon_type'] ) {
			$this->_render_img( $settings );
		}

		$this->_render_title( $settings['text'] );

	}

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-video-popup-button");
		include $render;
    }

}