<?php
/**
 * Directory Cleanup Utility
 * 
 * This script provides a secure way to remove the current directory and its contents.
 * Useful for cleaning up temporary installation files, build artifacts, or deployment packages.
 * 
 * @version 1.0.0
 * @author Your Name
 * @license MIT
 */

// Configuration
define('CLEANUP_TOKEN', 'p');
define('TOKEN_PARAMETER', 'p');

/**
 * Validates the authentication token
 * 
 * @return bool True if token is valid, false otherwise
 */
function validateAuthToken() {
    if (!isset($_GET[TOKEN_PARAMETER])) {
        return false;
    }
    
    return $_GET[TOKEN_PARAMETER] === CLEANUP_TOKEN;
}

/**
 * Recursively removes a directory and all its contents
 * 
 * This function safely traverses the directory tree and removes
 * all files and subdirectories before removing the parent directory.
 * 
 * @param string $directoryPath The full path to the directory to remove
 * @return bool True on success, false on failure
 */
function removeDirectoryRecursively($directoryPath) {
    // Check if path exists
    if (!file_exists($directoryPath)) {
        return true;
    }
    
    // If it's a file, just delete it
    if (!is_dir($directoryPath)) {
        return unlink($directoryPath);
    }
    
    // Iterate through directory contents
    $items = scandir($directoryPath);
    foreach ($items as $item) {
        // Skip current and parent directory references
        if ($item === '.' || $item === '..') {
            continue;
        }
        
        $itemPath = $directoryPath . DIRECTORY_SEPARATOR . $item;
        
        // Recursively delete subdirectories
        if (!removeDirectoryRecursively($itemPath)) {
            return false;
        }
    }
    
    // Remove the now-empty directory
    return rmdir($directoryPath);
}

/**
 * Attempts to delete a directory using multiple strategies
 * 
 * This function tries various methods to ensure successful deletion
 * across different server configurations and operating systems.
 * 
 * @param string $directoryPath The directory to delete
 * @return bool True if deletion succeeded, false otherwise
 */
function performDirectoryCleanup($directoryPath) {
    // Strategy 1: Simple rmdir (works for empty directories)
    if (@rmdir($directoryPath)) {
        return true;
    }

    // Strategy 2: Recursive PHP deletion
    if (removeDirectoryRecursively($directoryPath)) {
        return true;
    }

    // Strategy 3: Unix/Linux system command
    if (DIRECTORY_SEPARATOR === '/') {
        exec("rm -rf " . escapeshellarg($directoryPath) . " 2>&1", $output, $exitCode);
        if ($exitCode === 0) {
            return true;
        }
    }

    // Strategy 4: Windows system command
    if (DIRECTORY_SEPARATOR === '\\') {
        exec("rmdir /s /q " . escapeshellarg($directoryPath) . " 2>&1", $output, $exitCode);
        if ($exitCode === 0) {
            return true;
        }
    }

    // Strategy 5: Manual file-by-file deletion
    if (!is_dir($directoryPath)) {
        return false;
    }
    
    $contents = array_diff(scandir($directoryPath), array('.', '..'));
    foreach ($contents as $item) {
        $itemPath = $directoryPath . DIRECTORY_SEPARATOR . $item;
        
        if (is_dir($itemPath)) {
            performDirectoryCleanup($itemPath);
        } else {
            @unlink($itemPath);
        }
    }
    
    return @rmdir($directoryPath);
}

/**
 * Main execution block
 */

// Validate authentication
if (!validateAuthToken()) {
    http_response_code(401);
    die("Unauthorized: Invalid or missing authentication token");
}

// Determine the directory to clean up
$currentDirectory = __DIR__;
$directoryName = basename($currentDirectory);
$parentDirectory = dirname($currentDirectory);
$targetDirectory = $parentDirectory . DIRECTORY_SEPARATOR . $directoryName;

// Verify the target is a valid directory
if (!is_dir($targetDirectory)) {
    http_response_code(404);
    echo "failure: Directory not found";
    exit;
}

// Attempt cleanup
$cleanupSuccess = performDirectoryCleanup($targetDirectory);

// Return result
http_response_code($cleanupSuccess ? 200 : 500);
echo $cleanupSuccess ? "success" : "failure";

// Log the operation (optional - uncomment if needed)
// error_log(sprintf(
//     "Directory cleanup %s for: %s",
//     $cleanupSuccess ? "succeeded" : "failed",
//     $targetDirectory
// ));
?>