<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use \Elementor\Widget_Base as WB;
use \Elementor\Controls_Manager as KFCM;

use \Elementor\Group_Control_Typography as KFGCT;
use \Elementor\Group_Control_Text_Shadow as KFGCTS;
use \Elementor\Group_Control_Text_Stroke as KFGCTST;
use \Elementor\Group_Control_Background as KFGCBG;
use \Elementor\Group_Control_Border as KFGCBR;
use \Elementor\Group_Control_Box_Shadow as KFGCBS;


class KFW_Blockquote extends WB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-blockquote';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Blockquote', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_core_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-blockquote';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'kfw', 'kin', 'force', 'blockquote', 'quote' ];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
        return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/blockquote';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

		wp_register_style( 'kfw-blockquote',
			KFE_CONST_URL . 'libraries/elementor/assets/css/blockquote' . KFE_CONST_DEBUG_SUFFIX . '.css',
			[],
			KFE_CONST_VERSION
		);
    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'kfw-blockquote' ];
	}

	/**
	 * Get script dependencies.
	 *
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @return array Element scripts dependencies.
	 */
	public function get_script_depends() {
		return [ ];
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
        $this->_register_content_section();

		$this->_register_author_style_section();
		$this->_register_role_style_section();
		$this->_register_quote_icon_style_section();
		$this->_register_quote_block_style_section();
	}

	public function _register_content_section() {
		$this->start_controls_section( 'section_content', [
			'label' => esc_html__( 'Content', 'onnat-extra' )
		]);
			$this->add_control( 'author', [
				'label'       => esc_html__( 'Author', 'onnat-extra' ),
				'type'        => KFCM::TEXT,
				'label_block' => true,
				'placeholder' => esc_html__( 'Enter author', 'onnat-extra' ),
				'default'     => esc_html__( 'John Doe', 'onnat-extra' ),
			]);
			$this->add_control( 'role', [
				'label'       => esc_html__( 'Role', 'onnat-extra' ),
				'type'        => KFCM::TEXT,
				'label_block' => true,
				'placeholder' => esc_html__( 'Enter role', 'onnat-extra' ),
				'default'     => esc_html__( 'Project Lead', 'onnat-extra' ),
			]);
			$this->add_control( 'quote', [
				'label'       => esc_html__( 'Quote', 'onnat-extra' ),
				'type'        => KFCM::TEXTAREA,
				'label_block' => true,
				'placeholder' => esc_html__( 'Enter your quote', 'onnat-extra' ),
				'default'     => 'Aenean aliquet mi sit amet diam sagittis condimentum. Nulla aliquet velit eros, eget porta dolor consectetur vel. Nulla at lacinia augue. Donec non diam ante. Integer accumsan dictum rhoncus. Nunc eu venenatis tortor.'
			]);

			$this->add_control( 'type', [
				'label'       => esc_html__( 'Type', 'onnat-extra' ),
				'type'        => KFCM::CHOOSE,
				'label_block' => true,
				'toggle'      => false,
				'default'     => 'type-1',
				'options'     => [
                    'type-1' => [
                        'title' => esc_html__('Type 1', 'onnat-extra'),
                        'icon'  => 'kfw-ctrl-icon kfw-ctrl-type-1-icon'
                    ],
                    'type-2' => [
                        'title' => esc_html__('Type 2', 'onnat-extra'),
                        'icon'  => 'kfw-ctrl-icon kfw-ctrl-type-2-icon'
                    ],
                    'type-3' => [
                        'title' => esc_html__('Type 3', 'onnat-extra'),
                        'icon'  => 'kfw-ctrl-icon kfw-ctrl-type-3-icon'
                    ],
				],
			]);

        $this->end_controls_section();
	}

	public function _register_author_style_section() {
		$this->start_controls_section( 'section_author_style_settings', [
			'label'     => esc_html__( 'Author', 'onnat-extra' ),
			'tab'       => KFCM::TAB_STYLE,
			'condition' => [ 'author!'=> '' ],
		]);
			$this->add_group_control(KFGCT::get_type(), [
				'name'      => 'author_typo',
				'label'     => esc_html__( 'Typography', 'onnat-extra' ),
				'condition' => [ 'author!'=> '' ],
				'selector'  => '{{WRAPPER}} .kfw-ele-widget-blockquote-author',
			]);
			$this->add_control( 'author_color', [
				'label'     => esc_html__('Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'condition' => [ 'author!'=> '' ],
				'selectors' => ['{{WRAPPER}} .kfw-ele-widget-blockquote-author' => 'color: {{VALUE}};' ],
			]);
			$this->add_group_control( KFGCTS::get_type(), [
				'name'      => 'author_txt_shadow',
				'condition' => [ 'author!'=> '' ],
				'selector'  => '{{WRAPPER}} .kfw-ele-widget-blockquote-author',
			]);
			$this->add_group_control( KFGCTST::get_type(), [
				'name'      => 'author_txt_stroke',
				'condition' => [ 'author!'=> '' ],
				'selector'  => '{{WRAPPER}} .kfw-ele-widget-blockquote-author',
			]);
            $this->add_responsive_control( 'author_margin', [
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'condition'  => [ 'author!'=> '' ],
				'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-blockquote-author' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'author_padding', [
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'condition'  => [ 'author!'=> '' ],
				'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-blockquote-author' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
		$this->end_controls_section();
	}

	public function _register_role_style_section() {
		$this->start_controls_section( 'section_role_style_settings', [
			'label'     => esc_html__( 'Role', 'onnat-extra' ),
			'tab'       => KFCM::TAB_STYLE,
			'condition' => [ 'role!'=> '' ],
		]);
			$this->add_group_control(KFGCT::get_type(), [
				'name'      => 'role_typo',
				'label'     => esc_html__( 'Typography', 'onnat-extra' ),
				'condition' => [ 'role!'=> '' ],
				'selector'  => '{{WRAPPER}} .kfw-ele-widget-blockquote-role',
			]);
			$this->add_control( 'role_color', [
				'label'     => esc_html__('Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'condition' => [ 'role!'=> '' ],
				'selectors' => ['{{WRAPPER}} .kfw-ele-widget-blockquote-role' => 'color: {{VALUE}};' ],
			]);
			$this->add_group_control( KFGCTS::get_type(), [
				'name'      => 'role_txt_shadow',
				'condition' => [ 'role!'=> '' ],
				'selector'  => '{{WRAPPER}} .kfw-ele-widget-blockquote-role',
			]);
			$this->add_group_control( KFGCTST::get_type(), [
				'name'      => 'role_txt_stroke',
				'condition' => [ 'role!'=> '' ],
				'selector'  => '{{WRAPPER}} .kfw-ele-widget-blockquote-role',
			]);
			$this->add_responsive_control( 'role_margin', [
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'condition'  => [ 'role!'=> '' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-blockquote-role' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_responsive_control( 'role_padding', [
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'condition'  => [ 'role!'=> '' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-blockquote-role' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
		$this->end_controls_section();
	}

	public function _register_quote_icon_style_section() {
		$this->start_controls_section( 'section_quote_icon_style_settings', [
			'label' => esc_html__( 'Quote Symbol', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_control('quote_symbol',[
				'label'     => __('Insert quote symbol','onnat-extra' ),
				'type'      => KFCM::SWITCHER,
				'label_on'  => __( 'On', 'onnat-extra' ),
				'label_off' => __( 'Off', 'onnat-extra' ),
				'default'   => 'yes',
			]);
			$this->add_control( 'quote_symbol_color', [
				'label'     => esc_html__('Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'condition' => [ 'quote_symbol'=> 'yes' ],
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-blockquote-type-1 .kfw-ele-widget-blockquote-symbol:after'  => 'color: {{VALUE}};',
					'{{WRAPPER}} .kfw-ele-widget-blockquote-type-2 .kfw-ele-widget-blockquote-symbol:before' => 'color: {{VALUE}};',
					'{{WRAPPER}} .kfw-ele-widget-blockquote-type-3 .kfw-ele-widget-blockquote-symbol:before' => 'color: {{VALUE}};'
				],
			]);
			$this->add_group_control(KFGCT::get_type(), [
				'name'      => 'quote_symbol_typo',
				'label'     => esc_html__( 'Typography', 'onnat-extra' ),
				'condition' => [ 'quote_symbol'=> 'yes' ],
				'selector'  => '{{WRAPPER}} .kfw-ele-widget-blockquote-type-1 .kfw-ele-widget-blockquote-symbol:after, {{WRAPPER}} .kfw-ele-widget-blockquote-type-2 .kfw-ele-widget-blockquote-symbol:before, {{WRAPPER}} .kfw-ele-widget-blockquote-type-3 .kfw-ele-widget-blockquote-symbol:before',
			]);
			$this->add_group_control( KFGCTS::get_type(), [
				'name'      => 'quote_symbol_txt_shadow',
				'condition' => [ 'quote_symbol'=> 'yes' ],
				'selector'  => '{{WRAPPER}} .kfw-ele-widget-blockquote-type-1 .kfw-ele-widget-blockquote-symbol:after, {{WRAPPER}} .kfw-ele-widget-blockquote-type-2 .kfw-ele-widget-blockquote-symbol:before, {{WRAPPER}} .kfw-ele-widget-blockquote-type-3 .kfw-ele-widget-blockquote-symbol:before',
			]);
			$this->add_group_control( KFGCTST::get_type(), [
				'name'      => 'quote_symbol_txt_stroke',
				'condition' => [ 'quote_symbol'=> 'yes' ],
				'selector'  => '{{WRAPPER}} .kfw-ele-widget-blockquote-type-1 .kfw-ele-widget-blockquote-symbol:after, {{WRAPPER}} .kfw-ele-widget-blockquote-type-2 .kfw-ele-widget-blockquote-symbol:before, {{WRAPPER}} .kfw-ele-widget-blockquote-type-3 .kfw-ele-widget-blockquote-symbol:before',
			]);
			$this->add_responsive_control( 'quote_symbol_margin', [
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'condition'  => [ 'quote_symbol'=> 'yes' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-blockquote-type-1 .kfw-ele-widget-blockquote-symbol:after'  => 'margin:{{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					'{{WRAPPER}} .kfw-ele-widget-blockquote-type-2 .kfw-ele-widget-blockquote-symbol:before' => 'margin:{{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					'{{WRAPPER}} .kfw-ele-widget-blockquote-type-3 .kfw-ele-widget-blockquote-symbol:before' => 'margin:{{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				]
			]);
			$this->add_responsive_control( 'quote_symbol_padding', [
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'condition'  => [ 'quote_symbol'=> 'yes' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-blockquote-type-1 .kfw-ele-widget-blockquote-symbol:after'  => 'padding:{{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					'{{WRAPPER}} .kfw-ele-widget-blockquote-type-2 .kfw-ele-widget-blockquote-symbol:before' => 'padding:{{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					'{{WRAPPER}} .kfw-ele-widget-blockquote-type-3 .kfw-ele-widget-blockquote-symbol:before' => 'padding:{{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				]
			]);
			$this->add_control('quote_symbol_bg_style_before_hr', [
				'type'      => KFCM::DIVIDER,
				'condition' => [ 'type'=> 'type-3' ],
            ]);
			$this->add_responsive_control( 'quote_symbol_wrap_size', [
				'label'      => esc_html__('Size', 'onnat-extra' ),
				'type'       => KFCM::SLIDER,
				'size_units' => [ 'px' ],
				'range'      => [
					'px' => [
						'min' => 0,
						'max' => 999,
					],
				],
				'condition'  => [ 'type'=> 'type-3' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-blockquote-type-3 .kfw-ele-widget-blockquote-symbol:after' => 'width:{{SIZE}}{{UNIT}}; height:{{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .kfw-ele-widget-blockquote-type-3 .kfw-ele-widget-blockquote-symbol:before' => 'width:{{SIZE}}{{UNIT}};',
				],
			]);
			$this->add_responsive_control( 'quote_symbol_top_pos', [
				'label'      => esc_html__('Position', 'onnat-extra' ),
				'type'       => KFCM::SLIDER,
				'size_units' => [ 'px' ],
				'range'      => [
					'px' => [
						'min' => -250,
						'max' => 250,
					],
				],
				'condition'  => [ 'type'=> 'type-3' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-blockquote-type-3 .kfw-ele-widget-blockquote-symbol:after' => 'top:{{SIZE}}{{UNIT}};',
				],
			]);
			$this->add_group_control( KFGCBG::get_type(), [
				'name'      => 'quote_symbol_bg',
				'label'     => esc_html__('Background', 'onnat-extra' ),
				'types'     => [ 'classic', 'gradient' ],
				'exclude'   => [ 'image' ],# To remove background image option in classic mode
				'selector'  => '{{WRAPPER}} .kfw-ele-widget-blockquote-type-3 .kfw-ele-widget-blockquote-symbol:after',
				'condition' => [ 'type'=> 'type-3' ],
			]);
			$this->add_group_control( KFGCBR::get_type(), [
				'name'           => 'quote_symbol_border',
				'fields_options' => [
					'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
					'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
				],
				'condition'      => [ 'type'=> 'type-3' ],
				'selector'       => '{WRAPPER}} .kfw-ele-widget-blockquote-type-3 .kfw-ele-widget-blockquote-symbol:after',
			]);
			$this->add_responsive_control( 'quote_symbol_border_radius', [
				'label'      => esc_html__('Border Radius', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'condition'  => [ 'type'=> 'type-3' ],
				'selectors'  => [ '{WRAPPER}} .kfw-ele-widget-blockquote-type-3 .kfw-ele-widget-blockquote-symbol:after' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ],
			]);
			$this->add_group_control( KFGCBS::get_type(), [
				'name'      => 'quote_symbol_box_shadow',
				'condition' => [ 'type'=> 'type-3' ],
				'selector'  => '{WRAPPER}} .kfw-ele-widget-blockquote-type-3 .kfw-ele-widget-blockquote-symbol:after',
			]);
		$this->end_controls_section();
	}

	public function _register_quote_block_style_section() {
		$this->start_controls_section( 'section_settings', [
			'label' => esc_html__( 'Quote Block', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_responsive_control( 'quote_margin', [
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-blockquote-main' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_responsive_control( 'quote_padding', [
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-blockquote-main' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_group_control( KFGCBG::get_type(), [
				'name'     => 'quote_bg',
				'label'    => esc_html__('Background', 'onnat-extra' ),
				'types'    => [ 'classic', 'gradient' ],
				'selector' => '{{WRAPPER}} .kfw-ele-widget-blockquote-main',
			]);
			$this->add_group_control( KFGCBR::get_type(), [
				'name'           => 'quote_border',
				'fields_options' => [
					'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
					'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
				],
				'selector'       => '{{WRAPPER}} .kfw-ele-widget-blockquote-main',
			]);
			$this->add_responsive_control( 'quote_border_radius', [
				'label'      => esc_html__('Border Radius', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors'  => [ '{{WRAPPER}} .kfw-ele-widget-blockquote-main' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ],
			]);
			$this->add_group_control( KFGCBS::get_type(), [
				'name'      => 'quote_box_shadow',
				'selector'  => '{{WRAPPER}} .kfw-ele-widget-blockquote-main',
			]);
		$this->end_controls_section();
	}

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-blockquote");
		include $render;
    }
}