<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use \Elementor\Plugin as KFEl;
use \Elementor\Widget_Base as WB;
use \Elementor\Repeater as KFRTR;
use \Elementor\Controls_Manager as KFCM;
use \Elementor\Icons_Manager as KFIM;
use \Elementor\Group_Control_Border as KFGCBR;

class KFW_Blog_Single_Post_Social_Share extends WB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-blog-single-post-social-share';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Post Social Share', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_blog_post_look_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-social-icons';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'kfw', 'kin', 'blog', 'post', 'single', 'social', 'share' ];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
        return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/blog-single-post-social-share';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-social-icons elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

        wp_register_style( 'kfw-blog-single',
            KFE_CONST_URL . 'libraries/elementor/assets/css/blog-single' . KFE_CONST_DEBUG_SUFFIX . '.css',
            [],
            KFE_CONST_VERSION
        );

        wp_register_script( 'lordicon',
            KFE_CONST_URL . 'libraries/elementor/assets/js/lordicon.js',
            [],
            KFE_CONST_VERSION,
            true
        );

		if ( ! wp_script_is( 'jquery-goodshare', 'enqueued' ) ) {
			wp_register_script( 'jquery-goodshare',
				KFE_CONST_URL . 'libraries/elementor/assets/js/goodshare.min.js',
				[],
				KFE_CONST_VERSION,
				true
			);
		}
    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'widget-social-icons', 'kfw-blog-single' ];
	}

	/**
	 * Get script dependencies.
	 *
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @return array Element scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'jquery-goodshare' ];
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
        $this->_register_settings_section();
        $this->_register_style_section();
        $this->_register_hover_style_section();
	}

	protected function _register_settings_section() {
		$repeater = new KFRTR();

		$this->start_controls_section( 'section_social_settings', [
			'label' => esc_html__( 'Social Icons', 'onnat-extra' ),
		]);
			$repeater->add_control( 'medium', [
				'default'   => 'wordpress',
				'label'     => esc_html__('Medium', 'onnat-extra'),
				'options'   => $this->medium(),
				'type'      => KFCM::SELECT,
			]);
			$repeater->add_control( 'icon', [
				'label'            => esc_html__( 'Icon', 'onnat-extra' ),
				'type'             => KFCM::ICONS,
				'fa4compatibility' => 'social',
				'default'          => [
					'value'   => 'fab fa-wordpress',
					'library' => 'fa-brands',
				],
				'recommended'      => [
					'fa-brands' => [
						'delicious',
						'digg',
						'facebook',
						'linkedin',
						'mix',
						'odnoklassniki',
						'pinterest',
						'reddit',
						'skype',
						'telegram',
						'tumblr',
						'twitter',
						'viber',
						'vk',
						'weibo',
						'whatsapp',
						'x-twitter',
						'xing',
					],
				],
			]);
			$repeater->add_control( 'icon_color', [
				'label'   => esc_html__( 'Color', 'onnat-extra' ),
				'type'    => KFCM::SELECT,
				'default' => 'default',
				'options' => [
					'default' => esc_html__( 'Official Color', 'onnat-extra' ),
					'custom'  => esc_html__( 'Custom', 'onnat-extra' ),
				],
			]);
			$repeater->add_control('icon_primary_color',[
				'label'     => esc_html__( 'Primary Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'condition' => [ 'icon_color' => 'custom', ],
				'selectors' => [
					'{{WRAPPER}} {{CURRENT_ITEM}}.elementor-social-icon' => 'background-color: {{VALUE}};',
				],
			]);
			$repeater->add_control('icon_secondary_color',[
				'label'     => esc_html__( 'Secondary Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'condition' => [ 'icon_color' => 'custom', ],
				'selectors' => [
					'{{WRAPPER}} {{CURRENT_ITEM}}.elementor-social-icon i' => 'color: {{VALUE}};',
					'{{WRAPPER}} {{CURRENT_ITEM}}.elementor-social-icon svg' => 'fill: {{VALUE}};',
				],
			]);
			$this->add_control('items', [
				'type'        => KFCM::REPEATER,
				'fields'      => $repeater->get_controls(),
				'title_field' => '{{{ medium }}}',
				'default'     => [
					[
						'medium' => 'facebook',
						'icon'   => [
							'value'   => 'fab fa-facebook',
							'library' => 'fa-brands',
						],
					],
					[
						'medium' => 'twitter',
						'icon'   => [
							'value'   => 'fab fa-twitter',
							'library' => 'fa-brands',
						],
					],
					[
						'medium' => 'wordpress',
						'icon'   => [
							'value'   => 'fab fa-wordpress',
							'library' => 'fa-brands',
						],
					],
					[
						'medium' => 'linkedin',
						'icon'   => [
							'value'   => 'fab fa-linkedin',
							'library' => 'fa-brands',
						],
					],
				],
            ]);
			$this->add_control('hr0', [
				'type' => KFCM::DIVIDER,
			]);
			$this->add_responsive_control('columns',[
				'label'        => esc_html__( 'Columns', 'onnat-extra' ),
				'type'         => KFCM::SELECT,
				'default'      => '0',
				'prefix_class' => 'elementor-grid%s-',
				'options'      => [
					'0' => esc_html__( 'Auto', 'onnat-extra' ),
					'1' => '1',
					'2' => '2',
					'3' => '3',
					'4' => '4',
					'5' => '5',
					'6' => '6',
				],
				'selectors'    => [
					'{{WRAPPER}}' => '--grid-template-columns: repeat({{VALUE}}, auto);',
				],
			]);
			$this->add_responsive_control( 'align', [
				'label'     => esc_html__( 'Alignment', 'onnat-extra' ),
				'type'      => KFCM::CHOOSE,
				'options'   => [
					'left'   => [
						'title' => esc_html__('Left', 'onnat-extra'),
						'icon'  => 'eicon-text-align-left',
					],
					'center' => [
						'title' => esc_html__('Center', 'onnat-extra'),
						'icon'  => 'eicon-text-align-center',
					],
					'right'  => [
						'title' => esc_html__('Right', 'onnat-extra'),
						'icon'  => 'eicon-text-align-right',
					],
				],
				'default'   => 'left',
				'toggle'    => 'false',
				'selectors' => [
					'{{WRAPPER}}' => 'text-align: {{VALUE}};',
                ],
			]);
			$this->add_control('shape',[
				'label'        => esc_html__( 'Shape', 'onnat-extra' ),
				'type'         => KFCM::SELECT,
				'default'      => 'rounded',
				'prefix_class' => 'elementor-shape-',
				'options'      => [
					'rounded' => esc_html__( 'Rounded', 'onnat-extra' ),
					'square'  => esc_html__( 'Square', 'onnat-extra' ),
					'circle'  => esc_html__( 'Circle', 'onnat-extra' ),
				],
			]);
		$this->end_controls_section();
	}

	protected function _register_style_section() {
		$this->start_controls_section( 'section_social_normal_style', [
			'label' => esc_html__( 'Icon', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_control( 'icon_color', [
				'label'   => esc_html__( 'Color', 'onnat-extra' ),
				'type'    => KFCM::SELECT,
				'default' => 'default',
				'options' => [
					'default' => esc_html__( 'Official Color', 'onnat-extra' ),
					'custom'  => esc_html__( 'Custom', 'onnat-extra' ),
				],
			]);
			$this->add_control('icon_primary_color',[
				'label'     => esc_html__( 'Primary Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'condition' => [ 'icon_color' => 'custom', ],
				'selectors' => [
					'{{WRAPPER}} .elementor-social-icon' => 'background-color: {{VALUE}};',
				],
			]);
			$this->add_control('icon_secondary_color',[
				'label'     => esc_html__( 'Secondary Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'condition' => [ 'icon_color' => 'custom', ],
				'selectors' => [
					'{{WRAPPER}} .elementor-social-icon i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .elementor-social-icon svg' => 'fill: {{VALUE}};',
				],
			]);
			$this->add_responsive_control( 'icon_size', [
				'label'      => esc_html__('Icon Size', 'onnat-extra' ),
				'type'       => KFCM::SLIDER,
				'size_units' => [ 'px' ],
				'range'      => [
					'px' => [
						'min' => 0,
						'max' => 999,
					],
				],
				'selectors'  => [
					'{{WRAPPER}}' => '--icon-size: {{SIZE}}{{UNIT}}',
				],
			]);
			$this->add_responsive_control( 'icon_padding', [
				'label'      => esc_html__('Icon Padding', 'onnat-extra' ),
				'type'       => KFCM::SLIDER,
				'size_units' => [ 'px' ],
				'range'      => [
					'px' => [
						'min' => 0,
						'max' => 999,
					],
				],
				'selectors'  => [
					'{{WRAPPER}} .elementor-social-icon' => '--icon-padding: {{SIZE}}{{UNIT}}',
				],
			]);
			$this->add_responsive_control( 'icon_spacing', [
				'label'      => esc_html__('Icon Spacing', 'onnat-extra' ),
				'type'       => KFCM::SLIDER,
				'size_units' => [ 'px' ],
				'range'      => [
					'px' => [
						'min' => 0,
						'max' => 999,
					],
				],
				'selectors'  => [
					'{{WRAPPER}}' => '--grid-row-gap: {{SIZE}}{{UNIT}}; --grid-column-gap: {{SIZE}}{{UNIT}};',
				],
			]);
			$this->add_group_control( KFGCBR::get_type(), [
				'name'           => 'border',
				'fields_options' => [
					'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
					'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
				],
				'selector'       => '{{WRAPPER}} .elementor-social-icon',
			]);
			$this->add_responsive_control( 'border_radius', [
				'label'      => esc_html__('Border Radius', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .elementor-icon' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
			]);
		$this->end_controls_section();
	}

	protected function _register_hover_style_section() {
		$this->start_controls_section( 'section_social_hover_style', [
			'label' => esc_html__( 'Icon Hover', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_control('icon_hover_primary_color',[
				'label'     => esc_html__( 'Primary Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'condition' => [ 'icon_color' => 'custom', ],
				'selectors' => [
					'{{WRAPPER}} .elementor-social-icon:hover' => 'background-color: {{VALUE}};',
				],
			]);
			$this->add_control('icon_hover_secondary_color',[
				'label'     => esc_html__( 'Secondary Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'condition' => [ 'icon_color' => 'custom', ],
				'selectors' => [
					'{{WRAPPER}} .elementor-social-icon:hover i'   => 'color: {{VALUE}};',
					'{{WRAPPER}} .elementor-social-icon:hover svg' => 'fill: {{VALUE}};',
				],
			]);
			$this->add_control('icon_hover_border_color',[
				'label'     => esc_html__( 'Border Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'condition' => [ 'border_border!' => '', ],
				'selectors' => [
					'{{WRAPPER}} .elementor-social-icon:hover' => 'border-color: {{VALUE}};',
				],
			]);
			$this->add_control('hover_animation',[
				'label' => esc_html__( 'Hover Animation', 'elementor' ),
				'type' => KFCM::HOVER_ANIMATION,
			]);
		$this->end_controls_section();
	}

	public function medium() {
		return [
			'vkontakte'     => "vk.com",
			'facebook'      => "facebook.com",
			'odnoklassniki' => "ok.ru",
			'moimir'        => "my.mail.ru",
			'linkedin'      => "linkedin.com",
			'twitter'       => "twitter.com",
			'livejournal'   => "livejournal.com",
			'tumblr'        => "tumblr.com",
			'blogger'       => "blogger.com",
			'pinterest'     => "pinterest.com",
			'digg'          => "digg.com",
			'evernote'      => "evernote.com",
			'reddit'        => "reddit.com",
			'delicious'     => "delicious.com",
			'flipboard'     => "flipboard.com",
			'mix'           => "mix.com",
			'meneame'       => "meneame.net",
			'pocket'        => "getpocket.com",
			'surfingbird'   => "surfingbird.ru",
			'liveinternet'  => "liveinternet.ru",
			'buffer'        => "buffer.com",
			'instapaper'    => "instapaper.com",
			'xing'          => "xing.com",
			'wordpress'     => "wordpress.com",
			'baidu'         => "baidu.com",
			'renren'        => "renren.com",
			'weibo'         => "weibo.com",
			'skype'         => "skype.com",
			'telegram'      => "telegram.org",
			'viber'         => "viber.com",
			'whatsapp'      => "whatsapp.com",
		];
	}

    public function _is_editor_mode() {
        $preview = KFEl::instance()->preview->is_preview_mode();
        $editor  = KFEl::instance()->editor->is_edit_mode();

        return ( $preview || $editor ) ? true : false;
    }

	public function _render_icon( $icon , $echo = true ) {
		$icon = array_filter( $icon );

		if( !empty( $icon ) ) {
            echo '<div class="kfw-social-icon-wrap">';
                echo '<span class="kfw-elementor-icon-wrap">';
                    KFIM::render_icon( $icon, [ 'aria-hidden' => 'true' ] );
                echo '</span>';
            echo '</div>';
		}
    }	

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-blog-single-post-social-share");
		include $render;
    }
}