<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use \Elementor\Widget_Base as WB;
use \Elementor\Controls_Manager as KFCM;
use \Elementor\Icons_Manager as KFIM;

use \Elementor\Group_Control_Background as KFGCBG;
use \Elementor\Group_Control_Typography as KFGCT;
use \Elementor\Group_Control_Border as KFGCBR;
use \Elementor\Group_Control_Box_Shadow as KFGCBS;

class KFW_Button extends WB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-button';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Button', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_core_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-button';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'kfw', 'kin', 'force', 'button', 'btn' ];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
        return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/button';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

		wp_register_style( 'kfw-button',
			KFE_CONST_URL . 'libraries/elementor/assets/css/button' . KFE_CONST_DEBUG_SUFFIX . '.css',
			[],
			KFE_CONST_VERSION
		);
    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'kfw-button' ];
	}

	/**
	 * Get script dependencies.
	 *
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @return array Element scripts dependencies.
	 */
	public function get_script_depends() {
		return [ ];
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
        $this->_register_button_section();
        $this->_register_button_wrap_section();
        $this->_register_button_txt_section();
        $this->_register_button_icon_section();
    }

    public function _register_button_section() {
		$this->start_controls_section( 'section_button', [
			'label' => esc_html__( 'Button', 'onnat-extra' )
		]);
            $this->add_control( 'btn_text', [
                'label'     => esc_html__( 'Text', 'onnat-extra' ),
                'type'      => KFCM::TEXT,
                'default'   => esc_html__( 'Button', 'onnat-extra' ),
            ]);
            $this->add_control( 'btn_link', [
                'label'       => esc_html__( 'Link', 'onnat-extra' ),
                'type'        => KFCM::URL,
                'placeholder' => 'https://your-link.com',
                'default'     => [ 'url' => '#' ],
                'conditions'  => [
                    'relation' => 'or',
                    'terms'    => [
                        [
                            'name'     => 'btn_text',
                            'operator' => '!=',
                            'value'    => ''
                        ],
                        [
                            'name'     => 'btn_icon[value]',
                            'operator' => '!=',
                            'value'    => ''
                        ],
                    ]
                ]
            ]);
			$this->add_control( 'btn_icon', [
				'label'       => esc_html__( 'Icon', 'onnat-extra' ),
				'type'        => KFCM::ICONS,
				'skin'        => 'inline',
				'label_block' => false,
				'default'     => [
					'value'   => 'fa fa-check',
					'library' => 'fa-solid'
				]
			]);
        $this->end_controls_section();
    }

    public function _register_button_wrap_section() {

        $btn_condition = [
            'relation' => 'or',
            'terms'    => [
                [
                    'name'     => 'btn_text',
                    'operator' => '!=',
                    'value'    => ''
                ],
                [
                    'name'     => 'btn_icon[value]',
                    'operator' => '!=',
                    'value'    => ''
                ],
            ]
        ];

		$this->start_controls_section( 'section_button_wrap_style', [
            'label'      => esc_html__( 'Button Wrap', 'onnat-extra' ),
            'tab'        => KFCM::TAB_STYLE,
            'conditions' => $btn_condition,
		]);
            $this->add_responsive_control('menu_trigger_align', [
                'label'          => esc_html__( 'Alignment', 'onnat-extra' ),
                'type'           => KFCM::CHOOSE,
                'options'        => [
                    'left'   => [
                        'title' => esc_html__( 'Left', 'onnat-extra' ),
                        'icon'  => 'eicon-text-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__( 'Center', 'onnat-extra' ),
                        'icon'  => 'eicon-text-align-center',
                    ],
                    'right'  => [
                        'title' => esc_html__( 'Right', 'onnat-extra' ),
                        'icon'  => 'eicon-text-align-right',
                    ],
                ],
				'default'              => 'center',
				'tablet_default'       => 'center',
				'mobile_default'       => 'center',
				'toggle'               => false,
				'selectors_dictionary' => [
					'left'   => 'text-align:left;',
					'center' => 'text-align:center;',
					'right'  => 'text-align:right;',
				],
                'selectors'            => [
                    '{{WRAPPER}} .kfw-ele-widget-button-main' => '{{VALUE}};'
                ],
            ]);
            $this->add_responsive_control( 'btn_padding', [
                'label'      => esc_html__( 'Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ],
                'conditions' => $btn_condition
            ]);
			$this->add_responsive_control( 'btn_margin', [
				'label'      => esc_html__( 'Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'separator'  => 'after',
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-button-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
				'conditions' => $btn_condition
			]);
			$this->add_control( 'btn_animation_hover', [
				'label'      => esc_html__( 'Hover Animation', 'onnat-extra' ),
				'type'       => KFCM::HOVER_ANIMATION,
				'conditions' => $btn_condition,
			]);
        $this->end_controls_section();
    }

    public function _register_button_txt_section() {
		$this->start_controls_section( 'section_button_txt_style', [
            'label'     => esc_html__( 'Button Text', 'onnat-extra' ),
            'tab'       => KFCM::TAB_STYLE,
            'condition' => [
                'btn_text!' => ''
            ]
		]);
            $this->add_group_control( KFGCT::get_type(), [
                'name'        => 'btn_typo',
                'description' => esc_html__( 'Typography', 'onnat-extra' ),
                'selector'    => '{{WRAPPER}} .kfw-ele-widget-button',
                'condition'   => [
                    'btn_text!' => ''
                ]
            ]);

            $this->start_controls_tabs( 'btn_customize_tabs' );
                $this->start_controls_tab( 'btn_normal_state', [
                    'label'     => esc_html__( 'Normal', 'onnat-extra' ),
                    'condition' => [ 'btn_text!' => '' ],
                ]);
                    $this->add_control( 'btn_color', [
                        'label'     => esc_html__('Text Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kfw-ele-widget-button' => 'color: {{VALUE}};' ],
                        'condition' => [ 'btn_text!' => '' ],
                    ]);
                    $this->add_group_control( KFGCBG::get_type(), [
                        'name'      => 'btn_bg',
                        'label'     => esc_html__('Background', 'onnat-extra' ),
                        'types'     => [ 'classic', 'gradient' ],
                        'exclude'   => [ 'image' ],
                        'selector'  => '{{WRAPPER}} .kfw-ele-widget-button',
                        'condition' => [ 'btn_text!' => '' ],
                    ]);
                    $this->add_group_control( KFGCBR::get_type(), [
                        'name'           => 'btn_border',
                        'fields_options' => [
                            'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
                            'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
                        ],
                        'selector'       => '{{WRAPPER}} .kfw-ele-widget-button',
                        'condition' => [ 'btn_text!' => '' ],
                    ]);
                    $this->add_responsive_control( 'btn_border_radius', [
                        'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                        'type'       => KFCM::DIMENSIONS,
                        'size_units' => [ 'px', '%' ],
                        'selectors'  => [ '{{WRAPPER}} .kfw-ele-widget-button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ],
                        'condition'  => [ 'btn_text!' => '' ],
                    ]);
                    $this->add_group_control( KFGCBS::get_type(), [
                        'name'      => 'btn_box_shadow',
                        'selector'  => '{{WRAPPER}} .kfw-ele-widget-button',
                        'condition' => [ 'btn_text!' => '' ],
                    ]);
                $this->end_controls_tab();
                $this->start_controls_tab( 'btn_hover_state', [
                    'label'     => esc_html__( 'Hover', 'onnat-extra' ),
                    'condition' => [ 'btn_text!' => '' ],
                ]);
                    $this->add_control( 'btn_hover_color', [
                        'label'     => esc_html__('Text Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'selectors' => ['{{WRAPPER}} .kfw-ele-widget-button:hover' => 'color: {{VALUE}};' ],
                        'condition' => [ 'btn_text!' => '' ],
                    ]);
                    $this->add_group_control( KFGCBG::get_type(), [
                        'name'      => 'btn_hover_bg',
                        'label'     => esc_html__('Background', 'onnat-extra' ),
                        'types'     => [ 'classic', 'gradient' ],
                        'exclude'   => [ 'image' ],                                 # To remove background image option in classic mode
                        'selector'  => '{{WRAPPER}} .kfw-ele-widget-button:hover',
                        'condition' => [ 'btn_text!' => '' ],
                    ]);
                    $this->add_group_control( KFGCBR::get_type(), [
                        'name'           => 'btn_hover_border',
                        'fields_options' => [
                            'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
                            'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
                        ],
                        'selector'       => '{{WRAPPER}} .kfw-ele-widget-button:hover',
                        'condition'      => [ 'btn_text!' => '' ],
                    ]);
                    $this->add_responsive_control( 'btn_hover_border_radius', [
                        'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                        'type'       => KFCM::DIMENSIONS,
                        'size_units' => [ 'px', '%' ],
                        'selectors'  => [ '{{WRAPPER}} .kfw-ele-widget-button:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ],
                        'condition'  => [ 'btn_text!' => '' ],
                    ]);
                    $this->add_group_control( KFGCBS::get_type(), [
                        'name'      => 'btn_hover_box_shadow',
                        'selector'  => '{{WRAPPER}} .kfw-ele-widget-button:hover',
                        'condition' => [ 'btn_text!' => '' ],
                    ]);
                $this->end_controls_tab();
            $this->end_controls_tabs();

        $this->end_controls_section();
    }

    public function _register_button_icon_section() {
        $btn_icon_animation_condition = [
            'relation' => 'and',
            'terms'    => [
                [
                    'name'     => 'btn_text',
                    'operator' => '!=',
                    'value'    => ''
                ],
                [
                    'name'     => 'btn_icon[value]',
                    'operator' => '!=',
                    'value'    => ''
                ],
            ]
        ];

		$this->start_controls_section( 'section_button_icon_style', [
            'label' => esc_html__( 'Button Icon', 'onnat-extra' ),
            'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_responsive_control( 'btn_icon_padding', [
                'label'      => esc_html__( 'Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-button .kfw-elementor-icon-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ],
                'condition'  => [ 'btn_icon[value]!' => '' ],
            ]);
            $this->add_responsive_control( 'btn_icon_margin', [
                'label'      => esc_html__( 'Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-button .kfw-elementor-icon-wrap' => 'top: {{TOP}}{{UNIT}}; right: {{RIGHT}}{{UNIT}}; bottom: {{BOTTOM}}{{UNIT}}; left: {{LEFT}}{{UNIT}};'
                ],
                'condition'  => [ 'btn_icon[value]!' => '' ],
            ]);
            $this->add_control( 'btn_icon_pos', [
                'label'      => esc_html__( 'Position', 'onnat-extra' ),
                'type'       => KFCM::CHOOSE,
                'options'    => [
                    'left'           => [
                        'title' => esc_html__('Left', 'onnat-extra'),
                        'icon'  => 'eicon-text-align-left',
                    ],
                    'right'          => [
                        'title' => esc_html__('Right', 'onnat-extra'),
                        'icon'  => 'eicon-text-align-right',
                    ],
                    'animated-left'  => [
                        'title' => esc_html__('Animated Left', 'onnat-extra'),
                        'icon'  => 'eicon-animation',
                    ],
                    'animated-right' => [
                        'title' => esc_html__('Animated Right', 'onnat-extra'),
                        'icon'  => 'eicon-animation kfw-ctrl-rotate-180',
                    ],
                    'toggle'         => [
                        'title' => esc_html__('Toogle Button Text', 'onnat-extra'),
                        'icon'  => 'eicon-share-arrow',
                    ]
                ],
                'default'    => 'left',
                'toggle'     => false,
                'conditions' => $btn_icon_animation_condition
            ]);
            $this->add_control( 'alr_btn_icon_animation', [
                'label'      => esc_html__( 'Animation Direction', 'onnat-extra' ),
                'type'       => KFCM::CHOOSE,
                'options'    => [
                    'in'  => [
                        'title' => esc_html__('Animate In', 'onnat-extra'),
                        'icon'  => 'eicon-chevron-left'
                    ],
                    'out' => [
                        'title' => esc_html__('Animate Out', 'onnat-extra'),
                        'icon'  => 'eicon-chevron-right'
                    ],
                ],
                'default'    => 'in',
                'toggle'     => false,
                'conditions' => [
                    'relation' => 'and',
                    'terms'    => [
                        [
                            'name'     => 'btn_text',
                            'operator' => '!=',
                            'value'    => ''
                        ],
                        [
                            'name'     => 'btn_icon[value]',
                            'operator' => '!=',
                            'value'    => ''
                        ],
                        [
                            'name'     => 'btn_icon_pos',
                            'operator' => 'in',
                            'value'    => ['animated-left', 'animated-right' ]
                        ]
                    ]
                ]
            ]);
            $this->add_control( 'toggle_btn_icon_animation', [
                'label'      => esc_html__( 'Animation Direction', 'onnat-extra' ),
                'type'       => KFCM::CHOOSE,
                'options'    => [
                    'top'    => [
                        'title' => esc_html__('Top', 'onnat-extra'),
                        'icon'  => 'eicon-arrow-up',
                    ],
                    'right'  => [
                        'title' => esc_html__('Right', 'onnat-extra'),
                        'icon'  => 'eicon-arrow-right',
                    ],
                    'bottom' => [
                        'title' => esc_html__('Bottom', 'onnat-extra'),
                        'icon'  => 'eicon-arrow-down',
                    ],
                    'left'   => [
                        'title' => esc_html__('Left', 'onnat-extra'),
                        'icon'  => 'eicon-arrow-left',
                    ],
                ],
                'default'    => 'left',
                'toggle'     => false,
                'conditions' => [
                    'relation' => 'and',
                    'terms'    => [
                        [
                            'name'     => 'btn_text',
                            'operator' => '!=',
                            'value'    => ''
                        ],
                        [
                            'name'     => 'btn_icon[value]',
                            'operator' => '!=',
                            'value'    => ''
                        ],
                        [
                            'name'     => 'btn_icon_pos',
                            'operator' => '===',
                            'value'    => 'toggle'
                        ]
                    ]
                ]
            ]);
            $this->add_responsive_control( 'btn_icon_size', [
                'label'      => esc_html__('Size', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'size_units' => [ 'px' ],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 999,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-button .kfw-elementor-icon-wrap i' => 'font-size: {{SIZE}}{{UNIT}};'
                ],
                'condition'  => [ 'btn_icon[value]!' => '' ],
            ]);

            $this->start_controls_tabs( 'btn_icon_customize_tabs' );
                $this->start_controls_tab( 'btn_icon_normal_state', [
                    'label'      => esc_html__( 'Normal', 'onnat-extra' ),
                    'condition'  => [ 'btn_icon[value]!' => '' ],
                ]);
                    $this->add_control( 'btn_icon_color', [
                        'label'      => esc_html__('Icon Color', 'onnat-extra' ),
                        'type'       => KFCM::COLOR,
                        'selectors'  => [
                            '{{WRAPPER}} .kfw-ele-widget-button .kfw-elementor-icon-wrap i'   => 'color: {{VALUE}};',
                            '{{WRAPPER}} .kfw-ele-widget-button .kfw-elementor-icon-wrap svg' => 'fill: {{VALUE}};'
                        ],
                        'condition'  => [ 'btn_icon[value]!' => '' ],
                    ]);
                    $this->add_group_control( KFGCBG::get_type(), [
                        'name'       => 'btn_icon_bg',
                        'label'      => esc_html__('Background', 'onnat-extra' ),
                        'types'      => [ 'classic', 'gradient' ],
                        'exclude'    => [ 'image' ], # To remove background image option in classic mode
                        'selector'   => '{{WRAPPER}} .kfw-ele-widget-button .kfw-elementor-icon-wrap',
                        'condition'  => [ 'btn_icon[value]!' => '' ],
                    ]);
                    $this->add_group_control( KFGCBR::get_type(), [
                        'name'           => 'btn_icon_border',
                        'fields_options' => [
                            'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
                            'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
                        ],
                        'selector'       => '{{WRAPPER}} .kfw-ele-widget-button .kfw-elementor-icon-wrap',
                        'condition'      => [ 'btn_icon[value]!' => '' ],
                    ]);
                    $this->add_responsive_control( 'btn_icon_border_radius', [
                        'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                        'type'       => KFCM::DIMENSIONS,
                        'size_units' => [ 'px', '%' ],
                        'selectors'  => [ '{{WRAPPER}} .kfw-ele-widget-button .kfw-elementor-icon-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ],
                        'condition'  => [ 'btn_icon[value]!' => '' ],
                    ]);
                    $this->add_group_control( KFGCBS::get_type(), [
                        'name'       => 'btn_icon_box_shadow',
                        'selector'   => '{{WRAPPER}} .kfw-ele-widget-button .kfw-elementor-icon-wrap',
                        'condition'  => [ 'btn_icon[value]!' => '' ],
                    ]);
                $this->end_controls_tab();
                $this->start_controls_tab( 'btn_icon_hover_state', [
                    'label'      => esc_html__( 'Hover', 'onnat-extra' ),
                    'condition'  => [ 'btn_icon[value]!' => '' ],
                ]);
                    $this->add_control( 'btn_icon_hover_color', [
                        'label'      => esc_html__('Icon Color', 'onnat-extra' ),
                        'type'       => KFCM::COLOR,
                        'selectors'  => [
                            '{{WRAPPER}} .kfw-ele-widget-button:hover .kfw-elementor-icon-wrap i'   => 'color: {{VALUE}};',
                            '{{WRAPPER}} .kfw-ele-widget-button:hover .kfw-elementor-icon-wrap svg' => 'fill: {{VALUE}};'
                        ],
                        'condition'  => [ 'btn_icon[value]!' => '' ],
                    ]);
                    $this->add_group_control( KFGCBG::get_type(), [
                        'name'       => 'btn_icon_hover_bg',
                        'label'      => esc_html__('Background', 'onnat-extra' ),
                        'types'      => [ 'classic', 'gradient' ],
                        'exclude'    => [ 'image' ], # To remove background image option in classic mode
                        'selector'   => '{{WRAPPER}} .kfw-ele-widget-button:hover .kfw-elementor-icon-wrap',
                        'condition'  => [ 'btn_icon[value]!' => '' ],
                    ]);
                    $this->add_group_control( KFGCBR::get_type(), [
                        'name'           => 'btn_icon_hover_border',
                        'fields_options' => [
                            'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
                            'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
                        ],
                        'selector'       => '{{WRAPPER}} .kfw-ele-widget-button:hover .kfw-elementor-icon-wrap',
                        'condition'      => [ 'btn_icon[value]!' => '' ],
                    ]);
                    $this->add_responsive_control( 'btn_icon_hover_border_radius', [
                        'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                        'type'       => KFCM::DIMENSIONS,
                        'size_units' => [ 'px', '%' ],
                        'selectors'  => [ '{{WRAPPER}} .kfw-ele-widget-button:hover .kfw-elementor-icon-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ],
                        'condition'  => [ 'btn_icon[value]!' => '' ],
                    ]);
                    $this->add_group_control( KFGCBS::get_type(), [
                        'name'       => 'btn_icon_hover_box_shadow',
                        'selector'   => '{{WRAPPER}} .kfw-ele-widget-button:hover .kfw-elementor-icon-wrap',
                        'condition'  => [ 'btn_icon[value]!' => '' ],
                    ]);
                $this->end_controls_tab();
            $this->end_controls_tabs();

        $this->end_controls_section();
    }

	public function _render_icon( $icon , $echo = true ) {
		$icon = array_filter( $icon );

		if( !empty( $icon ) ) {
			echo '<span class="kfw-elementor-icon-wrap">';
				KFIM::render_icon( $icon, [ 'aria-hidden' => 'true' ] );
			echo '</span>';
		}
	}

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-button");
		include $render;
    }
}