<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use \Elementor\Plugin as KFEl;
use \Elementor\Widget_Base as WB;
use \Elementor\Controls_Manager as KFCM;
use \Elementor\Group_Control_Background as KFGCBG;
use \Elementor\Group_Control_Border as KFGCBR;
use \Elementor\Group_Control_Box_Shadow as KFGCBS;
use \Elementor\Icons_Manager as KFIM;

use \Elementor\Utils as KFUtils;

class KFW_Content_Carousel extends WB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-content-carousel';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Content Carousel', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_core_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-nested-carousel kfw-content-carousel';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'kfw', 'kin', 'content', 'carousel' ];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
		return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/content-carousel';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

		wp_register_script( 'kfw-content-carousel',
			KFE_CONST_URL . 'libraries/elementor/assets/js/content-carousel' . KFE_CONST_DEBUG_SUFFIX . '.js',
			[ 'jquery', 'elementor-frontend' ],
			KFE_CONST_VERSION,
			true
		);

		wp_register_style( 'kfw-content-carousel',
			KFE_CONST_URL . 'libraries/elementor/assets/css/content-carousel' . KFE_CONST_DEBUG_SUFFIX . '.css',
			[],
			KFE_CONST_VERSION
		);
    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'swiper', 'kfw-content-carousel' ];
	}

	/**
	 * Get script dependencies.
	 *
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @return array Element scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'kfw-content-carousel' ];
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
		$this->_register_items_section();
        $this->_register_settings_section();
        $this->_register_item_style_section();
		$this->_register_nav_style_section();
		$this->_register_dot_style_section();
    }

    public function _register_items_section() {
		$this->start_controls_section( 'section_items', [
			'label' => esc_html__( 'Templates', 'onnat-extra' ),
			'tab'   => KFCM::TAB_CONTENT,
		]);
            $this->add_control( 'items', [
                'type'    => KFCM::REPEATER,
                'fields'  => [
                    [
                        'name'    => 'template',
                        'label'   => esc_html__('Template', 'onnat-extra'),
                        'type'    => KFCM::SELECT,
                        'dynamic' => [ 'active' => true ],
                        'default' => 0,
                        'options' => kf_extra_plugin_elementor()->get_elementor_library(),
                    ],
                ],
                'default' => [],
            ]);
			$this->add_control( 'hr_0', [
				'type' => KFCM::DIVIDER,
			]);
			$this->add_responsive_control( 'item_min_height', [
				'label'      => esc_html__('Min Height', 'onnat-extra' ),
				'type'       => KFCM::SLIDER,
				'size_units' => ['px'],
				'range'      => [
					'px' => [
						'min' => 5,
						'max' => 1000,
					],
				],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-content-carousel-list, {{WRAPPER}} .kfw-ele-widget-content-carousel-item' => 'min-height:{{SIZE}}px;'
				],
			]);
        $this->end_controls_section();
    }

	public function _register_settings_section() {
		$this->start_controls_section( 'section_settings', [
			'label' => esc_html__( 'Settings', 'onnat-extra' ),
			'tab'   => KFCM::TAB_CONTENT,
		]);
			$this->add_responsive_control('cols', [
				'label'              => esc_html__( 'Columns', 'onnat-extra' ),
				'type'               => KFCM::CHOOSE,
				'toggle'             => false,
				'frontend_available' => true,
				'desktop_default'    => 4,
				'tablet_default'     => 3,
				'mobile_default'     => 1,
				'options'            => [
					'1' => [ 'title' => esc_html__('1 Column', 'onnat-extra'), 'icon' => 'kfw-ctrl-icon kfw-ctrl-numeric-icon' ],
					'2' => [ 'title' => esc_html__('2 Columns', 'onnat-extra'), 'icon' => 'kfw-ctrl-icon kfw-ctrl-numeric-2-icon' ],
					'3' => [ 'title' => esc_html__('3 Columns', 'onnat-extra'), 'icon' => 'kfw-ctrl-icon kfw-ctrl-numeric-3-icon' ],
					'4' => [ 'title' => esc_html__('4 Columns', 'onnat-extra'), 'icon' => 'kfw-ctrl-icon kfw-ctrl-numeric-4-icon' ],
					'5' => [ 'title' => esc_html__('5 Columns', 'onnat-extra'), 'icon' => 'kfw-ctrl-icon kfw-ctrl-numeric-5-icon' ],
					'6' => [ 'title' => esc_html__('6 Columns', 'onnat-extra'), 'icon' => 'kfw-ctrl-icon kfw-ctrl-numeric-6-icon' ],
				],
			]);
			$this->add_responsive_control('col_gap',[
				'label'              => esc_html__( 'Item Gap', 'onnat-extra' ),
				'type'               => KFCM::SLIDER,
				'frontend_available' => true,
				'default'            => [
					'size' => 20,
				],
				'range'              => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
			]);
			$this->add_responsive_control('slide_to_scroll',[
				'label'              => esc_html__( 'Slides To Scroll', 'onnat-extra' ),
				'type'               => KFCM::NUMBER,
				'frontend_available' => true,
				'min'                => 1,
				'max'                => 20,
				'step'               => 1,
				'default'            => 1,
			]);
			$this->add_control( 'loop', [
				'label'              => esc_html__( 'Enable Loop?', 'onnat-extra' ),
				'type'               => KFCM::SWITCHER,
				'return_value'       => 'yes',
				'default'            => 'yes',
				'frontend_available' => true,
            ]);
			$this->add_control( 'hr_1', [
				'type' => KFCM::DIVIDER,
			]);
				$this->add_control('autoplay_settings_heading',[
					'label' => esc_html__( 'Autoplay Settings', 'onnat-extra' ),
					'type'  => KFCM::HEADING,
				]);
				$this->add_control('autoplay', [
					'label'              => esc_html__( 'Enable Autoplay?', 'onnat-extra' ),
					'type'               => KFCM::SWITCHER,
					'return_value'       => 'yes',
					'frontend_available' => true,
				]);
				$this->add_control('autoplay_speed',[
					'label'              => esc_html__('Autoplay Speed (ms)', 'onnat-extra' ),
					'type'               => KFCM::SLIDER,
					'frontend_available' => true,
					'size_units'         => [],
					'range'              => [
						'' => [
							'min'  => 500,
							'max'  => 15000,
							'step' => 100
						]
					],
					'default'    => [
						'unit' => '',
						'size' => 1000
					],
					'condition'  => [
						'autoplay' => 'yes'
					],
				]);
				$this->add_control( 'pause', [
					'label'              => esc_html__( 'Pause on Hover?', 'onnat-extra' ),
					'type'               => KFCM::SWITCHER,
					'return_value'       => 'yes',
					'default'            => 'yes',
					'frontend_available' => true,
					'condition'          => [
						'autoplay' => 'yes'
					],
				]);
				$this->add_control( 'pause_on_interaction', [
					'label'              => esc_html__( 'Pause on Interaction?', 'onnat-extra' ),
					'type'               => KFCM::SWITCHER,
					'return_value'       => 'yes',
					'default'            => 'yes',
					'frontend_available' => true,
					'condition'          => [
						'autoplay' => 'yes'
					],
				]);
			$this->add_control( 'hr_2', [
				'type' => KFCM::DIVIDER,
			]);
				$this->add_control( 'nav_settings_heading',[
					'label' => esc_html__( 'Navigation Settings', 'onnat-extra' ),
					'type'  => KFCM::HEADING,
				]);
				$this->add_control( 'arrows', [
					'label'              => esc_html__( 'Enable Arrows?', 'onnat-extra' ),
					'type'               => KFCM::SWITCHER,
					'return_value'       => 'yes',
					'frontend_available' => true,
				]);
				$this->add_control( 'mobile_arrows', [
					'label'              => esc_html__( 'Remove Arrows on Mobile?', 'onnat-extra' ),
					'type'               => KFCM::SWITCHER,
					'frontend_available' => true,
					'return_value'       => 'yes',
					'condition'          => [
						'arrows' => 'yes'
					],
				]);
				$this->add_control( 'arrow_position', [
					'label'     => esc_html__( 'Position', 'onnat-extra' ),
					'type'      => KFCM::CHOOSE,
					'options'   => [
						'default'       => [
							'title' => esc_html__( 'Default', 'onnat-extra' ),
							'icon'  => 'eicon-chevron-left'
						],
						'top-left'      => [
							'title' => esc_html__( 'Top Left', 'onnat-extra' ),
							'icon'  => 'eicon-chevron-left kfw-ctrl-rotate-45'
						],
						'top-center'    => [
							'title' => esc_html__( 'Top Center', 'onnat-extra' ),
							'icon'  => 'eicon-chevron-left kfw-ctrl-rotate-90'
						],
						'top-right'     => [
							'title' => esc_html__( 'Top Right', 'onnat-extra' ),
							'icon'  => 'eicon-chevron-left kfw-ctrl-rotate-135'
						],
						'bottom-left'   => [
							'title' => esc_html__( 'Bottom Left', 'onnat-extra' ),
							'icon'  => 'eicon-chevron-left kfw-ctrl-rotate-315'
						],
						'bottom-center' => [
							'title' => esc_html__( 'Bottom Center', 'onnat-extra' ),
							'icon'  => 'eicon-chevron-right kfw-ctrl-rotate-90'
						],
						'bottom-right'  => [
							'title' => esc_html__( 'Bottom Right', 'onnat-extra' ),
							'icon'  => 'eicon-chevron-right kfw-ctrl-rotate-45'
						],
						'custom'        => [
							'title' => esc_html__( 'Custom', 'onnat-extra' ),
							'icon'  => 'eicon-map-pin'
						],
					],
					'default'   => 'bottom-center',
					'condition' => [
						'arrows' => 'yes'
					],
					'toggle'             => false,
					'frontend_available' => true,
				]);
				$this->add_control( 'custom_nav_d_color', [
					'label'     => esc_html__( 'Nav Disabled Color', 'onnat-extra' ),
					'type'      => KFCM::COLOR,
					'condition' => [
						'arrows'         => 'yes',
						'arrow_position' => 'custom'
					],
					'selectors' => [
						'.kfw-js-swiper-{{ID}}-nav-next, .kfw-js-swiper-{{ID}}-nav-prev'                                                   => 'cursor:pointer !important;',
						'.kfw-js-swiper-{{ID}}-nav-next.swiper-button-disabled, .kfw-js-swiper-{{ID}}-nav-prev.swiper-button-disabled'     => 'cursor: text !important;',
						'.kfw-js-swiper-{{ID}}-nav-next.swiper-button-disabled *, .kfw-js-swiper-{{ID}}-nav-prev.swiper-button-disabled *' => 'color:{{VALUE}}; opacity:0.5;',
					]
				]);
				$this->add_control( 'left_arrow', [
					'label'     => esc_html__( 'Next Arrow', 'onnat-extra' ),
					'type'      => KFCM::ICONS,
					'default'   => [
						'value'   => 'fas fa-arrow-right',
						'library' => 'fa-solid',
					],
					'condition' => [
						'arrows'          => 'yes',
						'arrow_position!' => 'custom'
					],
				]);
				$this->add_control( 'right_arrow', [
					'label'     => esc_html__( 'Prev Arrow', 'onnat-extra' ),
					'type'      => KFCM::ICONS,
					'default'   => [
						'value'   => 'fas fa-arrow-left',
						'library' => 'fa-solid',
					],
					'condition' => [
						'arrows'          => 'yes',
						'arrow_position!' => 'custom'
					],
				]);
			$this->add_control( 'hr_3', [
				'type' => KFCM::DIVIDER,
			]);
			$this->add_control( 'dots', [
				'label'              => esc_html__( 'Show Dots?', 'onnat-extra' ),
				'type'               => KFCM::SWITCHER,
				'return_value'       => 'yes',
				'default'            => 'yes',
				'frontend_available' => true,
			]);
		$this->end_controls_section();
	}

	public function _register_item_style_section() {
		$this->start_controls_section( 'section_item_style',[
			'label' => esc_html__( 'Items', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->start_controls_tabs( 'tabs_item_style' );
				$this->start_controls_tab( 'tabs_item_normal', [
					'label' => esc_html__( 'Normal', 'onnat-extra' ),
				]);
					$this->add_group_control( KFGCBG::get_type(), [
						'name'     => 'items_bg',
						'selector' => '{{WRAPPER}} .kfw-content-carousel-template-wrap',
					]);
					$this->add_control( 'hr_item_style_1', [
						'type' => KFCM::DIVIDER,
					]);
					$this->add_responsive_control( 'items_padding',[
						'label'      => esc_html__('Padding', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', 'em', '%' ],
						'default'    => [
							'top'      => '',
							'right'    => '',
							'bottom'   => '',
							'left'     => '',
							'unit'     => 'px',
							'isLinked' => false
						],
						'selectors'  => [
							'{{WRAPPER}} .kfw-content-carousel-template-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
						],
					]);
					$this->add_responsive_control( 'items_margin',[
						'label'      => esc_html__('Margin', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', 'em', '%' ],
						'selectors'  => [
							'{{WRAPPER}} .kfw-content-carousel-template-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
						],
					]);
					$this->add_control( 'hr_item_style_2', [
						'type' => KFCM::DIVIDER,
					]);
					$this->add_group_control( KFGCBR::get_type(), [
						'name'     => 'items_border',
						'selector' => '{{WRAPPER}} .kfw-content-carousel-template-wrap'
					]);
					$this->add_control( 'hr_item_style_3', [
						'type' => KFCM::DIVIDER,
					]);
					$this->add_responsive_control( 'items_border_radius', [
						'label'      => esc_html__('Border Radius', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', '%' ],
						'default'    => [
							'top'      => '',
							'right'    => '',
							'bottom'   => '',
							'left'     => '',
							'unit'     => 'px',
							'isLinked' => false
						],
						'selectors'  => [
							'{{WRAPPER}} .kfw-content-carousel-template-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
						]
					]);
					$this->add_control( 'hr_item_style_4', [
						'type' => KFCM::DIVIDER,
					]);
					$this->add_group_control( KFGCBS::get_type(), [
						'name'     => 'items_box_shadow',
						'selector' => '{{WRAPPER}} .kfw-content-carousel-template-wrap'
					]);
				$this->end_controls_tab();
				$this->start_controls_tab( 'tabs_item_active', [
					'label' => esc_html__( 'Active', 'onnat-extra' ),
				]);
					$this->add_group_control( KFGCBG::get_type(), [
						'name'     => 'items_active_bg',
						'selector' => '{{WRAPPER}} .kfw-ele-widget-content-carousel-item.swiper-slide-next > .kfw-content-carousel-template-wrap',
					]);
					$this->add_control('items_active_border_color',[
						'label'     => esc_html__( 'Border Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'condition' => [
							'items_border_border!' => '',
						],
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-widget-content-carousel-item.swiper-slide-next > .kfw-content-carousel-template-wrap' => 'border-color: {{VALUE}};',
						],
					]);
					$this->add_responsive_control( 'items_active_border_radius', [
						'label'      => esc_html__('Border Radius', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', '%' ],
						'default'    => [
							'top'      => '',
							'right'    => '',
							'bottom'   => '',
							'left'     => '',
							'unit'     => 'px',
							'isLinked' => false
						],
						'selectors'  => [
							'{{WRAPPER}} .kfw-ele-widget-content-carousel-item.swiper-slide-next > .kfw-content-carousel-template-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
						]
					]);
					$this->add_group_control(KFGCBS::get_type(),[
						'name'     => 'item_active_box_shadow',
						'selector' => '{{WRAPPER}} .kfw-ele-widget-content-carousel-item.swiper-slide-next > .kfw-content-carousel-template-wrap',
					]);
					$this->add_responsive_control( 'item_active_y_space', [
						'label'      => esc_html__('Y Space', 'onnat-extra' ),
						'type'       => KFCM::SLIDER,
						'size_units' => ['px'],
						'range'      => [
							'px' => [
								'min' => 5,
								'max' => 300,
							],
						],
						'selectors'  => [
							'{{WRAPPER}}' => '--kfw-swiper-active-y-space:{{SIZE}}px;',
						],
					]);					
				$this->end_controls_tab();
				$this->start_controls_tab( 'tabs_item_hover', [
					'label' => esc_html__( 'Hover', 'onnat-extra' ),
				]);
					$this->add_group_control( KFGCBG::get_type(), [
						'name'     => 'items_h_bg',
						'selector' => '{{WRAPPER}} .kfw-content-carousel-template-wrap:hover',
					]);
					$this->add_control('items_h_border_color',[
						'label'     => esc_html__( 'Border Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'condition' => [
							'items_border_border!' => '',
						],
						'selectors' => [
							'{{WRAPPER}} .kfw-content-carousel-template-wrap:hover' => 'border-color: {{VALUE}};',
						],
					]);
					$this->add_responsive_control( 'items_h_border_radius', [
						'label'      => esc_html__('Border Radius', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', '%' ],
						'default'    => [
							'top'      => '',
							'right'    => '',
							'bottom'   => '',
							'left'     => '',
							'unit'     => 'px',
							'isLinked' => false
						],
						'selectors'  => [
							'{{WRAPPER}} .kfw-content-carousel-template-wrap:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
						]
					]);
					$this->add_group_control(KFGCBS::get_type(),[
						'name'     => 'item_hover_box_shadow',
						'selector' => '{{WRAPPER}} .kfw-content-carousel-template-wrap:hover',
					]);
				$this->end_controls_tab();
			$this->end_controls_tabs();
		$this->end_controls_section();
	}

	public function _register_nav_style_section() {
		$this->start_controls_section( 'section_nav_style',[
			'label'     => esc_html__( 'Navigation', 'onnat-extra' ),
			'tab'       => KFCM::TAB_STYLE,
			'condition' => [
				'arrows'          => 'yes',
				'arrow_position!' => 'custom'
			],
		]);
			$this->start_controls_tabs( 'tabs_nav_style', [
				'condition' => [
					'arrows'          => 'yes',
					'arrow_position!' => 'custom'
				],
			]);
				$this->start_controls_tab( 'tabs_nav_normal', [
					'label' => esc_html__( 'Normal', 'onnat-extra' ),
				]);
					$this->add_control( 'nav_color', [
						'label'     => esc_html__('Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => [
							'{{WRAPPER}}' => '--kfw-swiper-nav-color:{{VALUE}};',
						],
					]);
					$this->add_control( 'nav_bg_color', [
						'label'     => esc_html__('Background Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => [
							'{{WRAPPER}}' => '--kfw-swiper-nav-bg-color:{{VALUE}};',
						],
					]);
					$this->add_control( 'hr_nav_style_1', [
						'type' => KFCM::DIVIDER,
					]);
					$this->add_group_control( KFGCBR::get_type(), [
						'name'     => 'nav_border',
						'selector' => '{{WRAPPER}} .kfw-brands-carousel-nav-wrap div'
					]);
					$this->add_control( 'hr_nav_style_3', [
						'type' => KFCM::DIVIDER,
					]);
					$this->add_responsive_control( 'nav_border_radius', [
						'label'      => esc_html__('Border Radius', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', '%' ],
						'default'    => [
							'top'      => '',
							'right'    => '',
							'bottom'   => '',
							'left'     => '',
							'unit'     => 'px',
							'isLinked' => false
						],
						'selectors'  => [
							'{{WRAPPER}} .kfw-brands-carousel-nav-wrap div' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
						]
					]);
					$this->add_control( 'hr_nav_style_4', [
						'type' => KFCM::DIVIDER,
					]);
					$this->add_responsive_control( 'nav_arrow_size',[
						'label'      => esc_html__('Size', 'onnat-extra' ),
						'type'       => KFCM::SLIDER,
						'size_units' => [ 'px' ],
						'range'      => [
							'px' => [
								'min' => 10,
								'max' => 200,
							],
						],
						'selectors'  => [
							'{{WRAPPER}} .kfw-brands-carousel-nav-wrap div'                                                                                                                                                                                                              => 'width:{{SIZE}}{{UNIT}}; height:{{SIZE}}{{UNIT}}; line-height:{{SIZE}}{{UNIT}};',
							'{{WRAPPER}} .kfw-brands-carousel-nav-wrap.kfw-brands-carousel-nav-pos-top-left, {{WRAPPER}} .kfw-brands-carousel-nav-wrap.kfw-brands-carousel-nav-pos-top-center, {{WRAPPER}} .kfw-brands-carousel-nav-wrap.kfw-brands-carousel-nav-pos-top-right'          => 'top:calc(({{SIZE}}{{UNIT}} + {{SIZE}}{{UNIT}}/2)*-1);',
							'{{WRAPPER}} .kfw-brands-carousel-nav-wrap.kfw-brands-carousel-nav-pos-bottom-left, {{WRAPPER}} .kfw-brands-carousel-nav-wrap.kfw-brands-carousel-nav-pos-bottom-center, {{WRAPPER}} .kfw-brands-carousel-nav-wrap.kfw-brands-carousel-nav-pos-bottom-right' => 'bottom:calc(({{SIZE}}{{UNIT}} + {{SIZE}}{{UNIT}}/2)*-1);',
							'{{WRAPPER}} .kfw-brands-carousel-nav-wrap.kfw-brands-carousel-nav-pos-default div.kfw-swiper-nav-prev'                                                                                                                                                     => 'margin-left:calc(({{SIZE}}{{UNIT}}/2)*-1);',
							'{{WRAPPER}} .kfw-brands-carousel-nav-wrap.kfw-brands-carousel-nav-pos-default div.kfw-swiper-nav-next'                                                                                                                                                     => 'margin-right:calc(({{SIZE}}{{UNIT}}/2)*-1);',
						],
					]);
					$this->add_responsive_control( 'nav_arrow_icon_size', [
						'label'      => esc_html__('Icon Size', 'onnat-extra' ),
						'type'       => KFCM::SLIDER,
						'size_units' => [ 'px' ],
						'range'      => [
							'px' => [
								'min' => 0,
								'max' => 999,
							],
						],
						'selectors'  => [
							'{{WRAPPER}} .kfw-brands-carousel-nav-wrap .kfw-elementor-icon-wrap' => 'font-size: {{SIZE}}{{UNIT}};'
						],
					]);
					$this->add_control( 'hr_nav_style_5', [
						'type' => KFCM::DIVIDER,
					]);
					$this->add_group_control( KFGCBS::get_type(), [
						'name'     => 'nav_box_shadow',
						'selector' => '{{WRAPPER}} .kfw-brands-carousel-nav-wrap div'
					]);
				$this->end_controls_tab();
				$this->start_controls_tab( 'tabs_nav_hover', [
					'label' => esc_html__( 'Hover', 'onnat-extra' ),
				]);
					$this->add_control( 'nav_h_color', [
						'label'     => esc_html__('Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => [
							'{{WRAPPER}}' => '--kfw-swiper-nav-hover-color:{{VALUE}};',
						],
					]);
					$this->add_control( 'nav_h_bg_color', [
						'label'     => esc_html__('Background Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => [
							'{{WRAPPER}}' => '--kfw-swiper-nav-bg-hover-color:{{VALUE}};',
						],
					]);
					$this->add_control( 'nav_h_border_color', [
						'label'     => esc_html__('Border Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => [
							'{{WRAPPER}} .kfw-brands-carousel-nav-wrap div:hover' => 'border-color:{{VALUE}};',
						],
						'condition' => [
							'nav_border_border!' => '',
						],
					]);
					$this->add_group_control( KFGCBS::get_type(), [
						'name'     => 'nav_h_box_shadow',
						'selector' => '{{WRAPPER}} .kfw-brands-carousel-nav-wrap div:hover'
					]);
				$this->end_controls_tab();
				$this->start_controls_tab( 'tabs_nav_disabled', [
					'label' => esc_html__( 'Disabled', 'onnat-extra' ),
				]);
					$this->add_control( 'nav_d_color', [
						'label'     => esc_html__('Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => [
							'{{WRAPPER}}' => '--kfw-swiper-nav-disabled-color:{{VALUE}};',
						],
					]);
					$this->add_control( 'nav_d_bg_color', [
						'label'     => esc_html__('Background Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => [
							'{{WRAPPER}}' => '--kfw-swiper-nav-disabled-bg-color:{{VALUE}};',
						],
					]);
					$this->add_control( 'nav_d_border_color', [
						'label'     => esc_html__('Border Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => [
							'{{WRAPPER}} .kfw-brands-carousel-nav-wrap .swiper-button-disabled' => 'border-color:{{VALUE}};',
						],
						'condition' => [
							'nav_border_border!' => '',
						],
					]);
					$this->add_group_control( KFGCBS::get_type(), [
						'name'     => 'nav_d_box_shadow',
						'selector' => '{{WRAPPER}} .kfw-brands-carousel-nav-wrap .swiper-button-disabled'
					]);
				$this->end_controls_tab();
			$this->end_controls_tabs();
		$this->end_controls_section();
	}

	public function _register_dot_style_section() {
		$this->start_controls_section( 'section_dot_style',[
			'label'     => esc_html__( 'Dot', 'onnat-extra' ),
			'tab'       => KFCM::TAB_STYLE,
			'condition' => [
				'dots' => 'yes'
			],
		]);
			$this->start_controls_tabs( 'tabs_dot_style', [
				'condition' => [
					'dots' => 'yes'
				],
			]);
				$this->start_controls_tab( 'tabs_dot_normal', [
					'label' => esc_html__( 'Normal', 'onnat-extra' ),
				]);
					$this->add_control( 'dot_color_settings', [
						'label'        => esc_html__('Colors', 'onnat-extra'),
						'type'         => KFCM::POPOVER_TOGGLE,
						'return_value' => 'yes',
					]);
					$this->start_popover();
						$this->add_control( 'dot_color', [
							'label'     => esc_html__('Color', 'onnat-extra' ),
							'type'      => KFCM::COLOR,
							'selectors' => [
								'{{WRAPPER}} .kfw-swiper-pagination .swiper-pagination-bullet' => 'background:{{VALUE}};',
							],
						]);
						$this->add_control( 'active_dot_color', [
							'label'     => esc_html__('Active Color', 'onnat-extra' ),
							'type'      => KFCM::COLOR,
							'selectors' => [
								'{{WRAPPER}} .kfw-swiper-pagination .swiper-pagination-bullet.swiper-pagination-bullet-active, {{WRAPPER}} .kfw-swiper-pagination .swiper-pagination-bullet.swiper-pagination-bullet-active:hover' => 'background:{{VALUE}};',
							],
						]);
					$this->end_popover();
					$this->add_control( 'hr_dot_style_1', [
						'type' => KFCM::DIVIDER,
					]);
					$this->add_responsive_control( 'dot_padding',[
						'label'      => esc_html__('Padding', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', 'em', '%' ],
						'default'    => [
							'top'      => '',
							'right'    => '',
							'bottom'   => '',
							'left'     => '',
							'unit'     => 'px',
							'isLinked' => false
						],
						'selectors'  => [
							'{{WRAPPER}} .kfw-swiper-pagination .swiper-pagination-bullet' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
						],
					]);
					$this->add_responsive_control( 'dot_margin',[
						'label'      => esc_html__('Margin', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', 'em', '%' ],
						'selectors'  => [
							'{{WRAPPER}} .kfw-swiper-pagination' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
						],
					]);
					$this->add_control( 'hr_dot_style_2', [
						'type' => KFCM::DIVIDER,
					]);
					$this->add_control( 'dot_size_settings_heading',[
						'label' => esc_html__( 'Size Settings', 'onnat-extra' ),
						'type'  => KFCM::HEADING,
					]);
						$this->add_responsive_control( 'dot_width',[
							'label'      => esc_html__('Width', 'onnat-extra' ),
							'type'       => KFCM::SLIDER,
							'size_units' => [ 'px' ],
							'range'      => [
								'px' => [
									'min' => 10,
									'max' => 100,
								],
							],
							'selectors'  => [
								'{{WRAPPER}} .kfw-swiper-pagination .swiper-pagination-bullet' => 'width:{{SIZE}}{{UNIT}};',
								'{{WRAPPER}} .kfw-swiper-pagination .swiper-pagination-bullet.swiper-pagination-bullet-active' => 'width:calc({{SIZE}}{{UNIT}} + ({{SIZE}}{{UNIT}}/2));'
							],
						]);
						$this->add_responsive_control( 'dot_height',[
							'label'      => esc_html__('Height', 'onnat-extra' ),
							'type'       => KFCM::SLIDER,
							'size_units' => [ 'px' ],
							'range'      => [
								'px' => [
									'min' => 10,
									'max' => 100,
								],
							],
							'selectors'  => [
								'{{WRAPPER}} .kfw-swiper-pagination .swiper-pagination-bullet' => 'height:{{SIZE}}{{UNIT}};',
								'{{WRAPPER}} .kfw-swiper-pagination .swiper-pagination-bullet.swiper-pagination-bullet-active' => 'height:calc({{SIZE}}{{UNIT}} + ({{SIZE}}{{UNIT}}/2));'
							],
						]);
						$this->add_responsive_control( 'dot_border_radius', [
							'label'      => esc_html__('Border Radius', 'onnat-extra' ),
							'type'       => KFCM::DIMENSIONS,
							'size_units' => [ 'px', '%' ],
							'default'    => [
								'top'      => '',
								'right'    => '',
								'bottom'   => '',
								'left'     => '',
								'unit'     => 'px',
								'isLinked' => false
							],
							'selectors'  => [
								'{{WRAPPER}} .kfw-swiper-pagination .swiper-pagination-bullet' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
							]
						]);
						$this->add_control( 'hr_dot_style_3', [
							'type' => KFCM::DIVIDER,
						]);
						$this->add_responsive_control( 'dot_space_between',[
							'label'      => esc_html__('Space Between', 'onnat-extra' ),
							'type'       => KFCM::SLIDER,
							'selectors'  => [
								'{{WRAPPER}}' => '--kfw-swiper-dots-space-between: {{SIZE}}{{UNIT}};',
							],
						]);
					$this->add_control( 'hr_dot_style_4', [
						'type' => KFCM::DIVIDER,
					]);
					$this->add_group_control( KFGCBS::get_type(), [
						'name'     => 'dot_box_shadow',
						'selector' => '{{WRAPPER}} .kfw-swiper-pagination .swiper-pagination-bullet'
					]);
				$this->end_controls_tab();
				$this->start_controls_tab( 'tabs_dot_hover', [
					'label' => esc_html__( 'Hover', 'onnat-extra' ),
				]);
					$this->add_control( 'h_dot_color', [
						'label'     => esc_html__('Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => [
							'{{WRAPPER}} .kfw-swiper-pagination .swiper-pagination-bullet:hover' => 'background:{{VALUE}};',
						],
					]);
					$this->add_group_control( KFGCBS::get_type(), [
						'name'     => 'dot_h_box_shadow',
						'selector' => ''
					]);
				$this->end_controls_tab();
			$this->end_controls_tabs();
		$this->end_controls_section();
	}

	protected function _generate_setting_json( array $settings ) {

		$cols = [
			'widescreen'   => 5,
			'desktop'      => 4,
			'laptop'       => 3,
			'tablet'       => 2,
			'tablet_extra' => 2,
			'mobile'       => 1,
			'mobile_extra' => 1,
		];

		$options = [
			'autoHeight'     => true,
			'loop'           => $settings['loop'] === 'yes' ? true : false,
			'slidesPerView'  => 1,
			'slidesPerGroup' => 1,
			'spaceBetween'   => 20,
		];

		/**
		 * Break Points
		 */
		$breakpoints            = \Elementor\Plugin::$instance->breakpoints->get_breakpoints();
		$options['breakpoints'] = [];

		$options['breakpoints'][1440] = [
			"device"         => 'desktop',
			'slidesPerView'  => ( isset($settings["cols"]) && !empty( $settings["cols"]) ) ? (int)$settings["cols"] : $cols['desktop'],
			'slidesPerGroup' => ( isset($settings["slide_to_scroll"]) && !empty( $settings["slide_to_scroll"] ) ) ? (int)$settings["slide_to_scroll"] : 1,
			'spaceBetween'   => ( isset($settings["col_gap"]["size"]) && !empty( $settings["col_gap"]["size"] ) ) ? (int)$settings["col_gap"]["size"] : 20,
		];

		foreach( $breakpoints as $breakpoint ) {
			$device = $breakpoint->get_label();
			$name   = $breakpoint->get_name();
			$value  = $breakpoint->get_value();

			$col_key    = "cols_".$name;
			$scroll_key = "slide_to_scroll_".$name;
			$gap_key    = "col_gap_".$name;

			if( isset( $settings[ $col_key ] ) ) {
				$options['breakpoints'][ $value ] = [
					"device"         => $device,
					"slidesPerView"  => ( isset( $settings[ $col_key ] ) && !empty( $settings[ $col_key ] ) ) ? (int) $settings[ $col_key ] : $cols[$name],
					"slidesPerGroup" => ( isset( $settings[ $scroll_key ] ) && !empty( $settings[ $scroll_key ] ) ) ? (int) $settings[ $scroll_key ] : 1,
					"spaceBetween"   => ( isset( $settings[ $gap_key ]['size'] ) && !empty( $settings[ $gap_key ]['size'] ) ) ? (int) $settings[ $gap_key ]['size'] : 20,
				];
			} else {
				$options['breakpoints'][ $value ] = [
					"device"         => $device,
					"slidesPerView"  => $cols[$name],
					"slidesPerGroup" => 1,
					"spaceBetween"   => 20,
				];
			}
		}

		if( $settings['autoplay'] === 'yes' ) {
			$options['pauseOnHover']  = $settings['pause'] === 'yes' ? true : false;
			$options['autoplay']['delay'] = isset( $settings['autoplay_speed']['size'] ) ? $settings['autoplay_speed']['size'] : 1000;
		} else {
			$options['pauseOnHover']  = false;
		}

		if( $settings['arrows'] === 'yes' ) {
			$options['navigation']['nextEl'] = '.kfw-js-swiper-'.esc_attr( $this->get_id() ).'-nav-next';
			$options['navigation']['prevEl'] = '.kfw-js-swiper-'.esc_attr( $this->get_id() ).'-nav-prev';
			$options['arrowPosition']        = $settings['arrow_position'];
			$options['hideArrowsOnMobile']   = $settings['mobile_arrows'] === 'yes' ? true : false;
		}

		if( $settings['dots'] === 'yes' ) {
			$options['pagination']['type']      = 'bullets';
			$options['pagination']['el']        = '.kfw-js-swiper-'.esc_attr( $this->get_id() ).'-pagination';
			$options['pagination']['clickable'] = true;
		}

		return wp_json_encode( $options );
	}

	public function _message( $message, $echo = true ) {
		$return = sprintf( '
			<div class="elementor-alert elementor-alert-info" role="alert">
				<span class="elementor-alert-description">%s</span>
			</div>',
			$message
		);

		if( $echo ) {
			echo $return;
			return;
		}

		return $return;
	}

	public function _render_icon( $icon , $echo = true ) {
		$icon = array_filter( $icon );

		if( !empty( $icon ) ) {
			echo '<span class="kfw-elementor-icon-wrap">';
				KFIM::render_icon( $icon, [ 'aria-hidden' => 'true' ] );
			echo '</span>';
		}
	}

    public function _is_editor_mode() {
        $preview = KFEl::instance()->preview->is_preview_mode();
        $editor  = KFEl::instance()->editor->is_edit_mode();

        return ( $preview || $editor ) ? true : false;
    }

	/**
	 * Get template content
	 */
	public function _generate_template_content( $template_id ) {
		return KFEl::instance()->frontend->get_builder_content_for_display( $template_id );
	}

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-content-carousel");
		include $render;
    }

}