<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use \Elementor\Plugin as KFEl;
use \Elementor\Widget_Base as WB;
use \Elementor\Controls_Manager as KFCM;
use \Elementor\Group_Control_Image_Size as KFGCIS;

use \Elementor\Group_Control_Typography as KFGCT;
use \Elementor\Group_Control_Text_Shadow as KFGCTS;
use \Elementor\Group_Control_Text_Stroke as KFGCTST;

class KFW_CPT_Project_Post_Title extends WB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-cpt-project-post-title';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Project Title', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_single_project_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-post-title';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'kfw', 'kin', 'project', 'post', 'single', 'title' ];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
        return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/cpt-project-post-title';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

        wp_register_style( 'kfw-cpt-project-single',
            KFE_CONST_URL . 'libraries/elementor/assets/css/cpt-project-single' . KFE_CONST_DEBUG_SUFFIX . '.css',
            [],
            KFE_CONST_VERSION
        );

    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'kfw-cpt-project-single' ];
	}

	/**
	 * Get script dependencies.
	 *
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @return array Element scripts dependencies.
	 */
	public function get_script_depends() {
		return [];
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
		$this->_register_title_section();
		$this->_register_title_style_section();
	}

	public function _register_title_section() {
		$this->start_controls_section( 'section_title', [
			'label' => esc_html__( 'Title', 'onnat-extra' )
		]);
			$this->add_control('title_tag', [
				'label'     => esc_html__( 'HTML Tag', 'onnat-extra' ),
				'type'      => KFCM::CHOOSE,
				'label_block' => true,
				'options'   => [
					'h1'   => [
						'title' => esc_html__( 'H1 Tag', 'onnat-extra' ),
						'icon'  => 'eicon-editor-h1',
					],
					'h2'   => [
						'title' => esc_html__( 'H2 Tag', 'onnat-extra' ),
						'icon'  => 'eicon-editor-h2',
					],
					'h3'   => [
						'title' => esc_html__( 'H3 Tag', 'onnat-extra' ),
						'icon'  => 'eicon-editor-h3',
					],
					'h4'   => [
						'title' => esc_html__( 'H4 Tag', 'onnat-extra' ),
						'icon'  => 'eicon-editor-h4',
					],
					'h5'   => [
						'title' => esc_html__( 'H5 Tag', 'onnat-extra' ),
						'icon'  => 'eicon-editor-h5',
					],
					'h6'   => [
						'title' => esc_html__( 'H6 Tag', 'onnat-extra' ),
						'icon'  => 'eicon-editor-h6',
					],
					'div'  => [
						'title' => esc_html__( 'div Tag', 'onnat-extra' ),
						'icon'  => 'kfw-ctrl-icon  kfw-ctrl-div',
					],
					'span' => [
						'title' => esc_html__( 'span Tag', 'onnat-extra' ),
						'icon'  => 'kfw-ctrl-icon kfw-ctrl-span',
					],
					'p'    => [
						'title' => esc_html__( 'p Tag', 'onnat-extra' ),
						'icon'  => 'eicon-editor-paragraph',
					],
				],
				'default'   => 'h2',
				'toggle'    => false,
			]);
            $this->add_control('use_link',[
                'label'        => esc_html__( 'Enable Link?', 'onnat-extra' ),
                'type'         => KFCM::SWITCHER,
                'return_value' => 'yes',
                'default'      => 'no'
            ]);
		$this->end_controls_section();
	}

	public function _register_title_style_section() {
		$this->start_controls_section( 'section_title_style', [
			'label' => esc_html__( 'Title', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_responsive_control('align',[
				'label'        => esc_html__( 'Alignment', 'onnat-extra' ),
				'type'         => KFCM::CHOOSE,
				'options'      => [
					'left'    => [
						'title' => esc_html__( 'Left', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-left',
					],
					'center'  => [
						'title' => esc_html__( 'Center', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-center',
					],
					'right'   => [
						'title' => esc_html__( 'Right', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-right',
					],
					'justify' => [
						'title' => esc_html__( 'Justified', 'onnat-extra' ),
						'icon'  => 'eicon-text-align-justify',
					],
				],
				'toggle'       => false,
				'default'      => 'center',
				'selectors'    => [
					'{{WRAPPER}} .kfw-ele-widget-post-title' => 'text-align: {{VALUE}};',
				],
			]);
            $this->add_group_control(KFGCT::get_type(), [
                'name'      => 'title_typo',
                'label'     => esc_html__( 'Typography', 'onnat-extra' ),
                'selector'  => '{{WRAPPER}} .kfw-ele-widget-post-title',
            ]);
			$this->add_control( 'title_color', [
				'label'     => esc_html__('Text Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-post-title' => 'color: {{VALUE}};'
				],
			]);
			$this->add_control( 'title_hover_color', [
				'label'     => esc_html__('Hover Text Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'condition' => [ 
					'use_link' => 'yes'
				],
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-post-title:hover' => 'color: {{VALUE}};'
				],
			]);			
			$this->add_group_control( KFGCTS::get_type(), [
				'name'      => 'title_txt_shadow',
				'selector'  => '{{WRAPPER}} .kfw-ele-widget-post-title',
			]);
			$this->add_group_control( KFGCTST::get_type(), [
				'name'      => 'title_txt_stroke',
				'selector'  => '{{WRAPPER}} .kfw-ele-widget-post-title',
			]);
			$this->add_control( 'title_blend_mode', [
				'label'     => esc_html__('Blend Mode', 'onnat-extra' ),
				'type'      => KFCM::SELECT,
				'options'   => [
					''            => esc_html__( 'Normal', 'onnat-extra' ),
					'multiply'    => esc_html__( 'Multiply', 'onnat-extra' ),
					'screen'      => esc_html__( 'Screen', 'onnat-extra' ),
					'overlay'     => esc_html__( 'Overlay', 'onnat-extra' ),
					'darken'      => esc_html__( 'Darken', 'onnat-extra' ),
					'lighten'     => esc_html__( 'Lighten', 'onnat-extra' ),
					'color-dodge' => esc_html__( 'Color Dodge', 'onnat-extra' ),
					'saturation'  => esc_html__( 'Saturation', 'onnat-extra' ),
					'color'       => esc_html__( 'Color', 'onnat-extra' ),
					'difference'  => esc_html__( 'Difference', 'onnat-extra' ),
					'exclusion'   => esc_html__( 'Exclusion', 'onnat-extra' ),
					'hue'         => esc_html__( 'Hue', 'onnat-extra' ),
					'luminosity'  => esc_html__( 'Luminosity', 'onnat-extra' ),
				],
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-widget-post-title' => 'mix-blend-mode: {{VALUE}};'
				],
			]);
		$this->end_controls_section();
	}

    public function _tag( $tag ) {
        $tags = [
            'div',
            'span',
            'p',
            'h1',
            'h2',
            'h3',
            'h4',
            'h5',
            'h6',
        ];

        return in_array( strtolower( $tag ), $tags) ? $tag : 'div';
    }

    public function _title() {
		$post_id  = get_the_id();

		$settings = $this->get_settings();
		$tag      = $this->_tag( $settings['title_tag'] );
		$mode     = $this->_is_editor_mode();

		if( $mode ) {
			$post_id = kf_extra_plugin_elementor()->get_latest_post_id( 'kinfw-project' );
		}

		$post_title = get_the_title($post_id);

		/**
		 * Class attribures
		 */
			$class = [
				'kfw-ele-widget-post-title',
				( $settings['use_link'] === 'yes' ) ? 'kfw-ele-widget-linked-post-title' : ''
			];

			$this->add_render_attribute( 'title', 'class', $class );

		/**
		 * Linked Title
		 */
			if( $settings['use_link'] === 'yes' && !$mode ) {
				$link = get_permalink( $post_id );
				$this->add_render_attribute( 'title', 'onclick', 'window.open("'. esc_url( $link ) .'", "_self")' );
			}

		if( !empty( $post_title ) ) {
			printf(
				'<%1$s %2$s>%3$s</%1$s>',
				$tag,
				$this->get_render_attribute_string( 'title' ),
				esc_html( $post_title ),
				$tag,
			);
		}

	}

    public function _is_editor_mode() {
        $preview = KFEl::instance()->preview->is_preview_mode();
        $editor  = KFEl::instance()->editor->is_edit_mode();

        return ( $preview || $editor ) ? true : false;
    }

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-cpt-project-post-title");
		include $render;
    }
}