<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}
use \Elementor\Plugin as KFEl;
use \Elementor\Widget_Base as KFWB;
use \Elementor\Controls_Manager as KFCM;

use \Elementor\Group_Control_Typography as KFGCT;
use \Elementor\Group_Control_Text_Shadow as KFGCTS;
use \Elementor\Group_Control_Text_Stroke as KFGCTST;

class KFW_Custom_Text extends KFWB {
	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-custom-text';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Custom Text', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_core_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-t-letter';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'kfw', 'kin', 'custom', 'text', 'letter'];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
        return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/custom-text';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

		wp_register_style( 'kfw-custom-text',
			KFE_CONST_URL . 'libraries/elementor/assets/css/custom-text' . KFE_CONST_DEBUG_SUFFIX . '.css',
			[],
			KFE_CONST_VERSION
		);

        if ( ! wp_script_is( 'kfw-gsap', 'enqueued' ) ) {
            wp_register_script( 'kfw-gsap', KFE_CONST_URL . 'assets/js/gsap.min.js', [], KFE_CONST_VERSION, true );
        }

        if ( ! wp_script_is( 'kfw-gsap-scroll-trigger', 'enqueued' ) ) {
            wp_register_script( 'kfw-gsap-scroll-trigger', KFE_CONST_URL . 'assets/js/ScrollTrigger.min.js', ['kfw-gsap'], KFE_CONST_VERSION, true );
        }

        if ( ! wp_script_is( 'kfw-gsap-split-text', 'enqueued' ) ) {
            wp_register_script( 'kfw-gsap-split-text', KFE_CONST_URL . 'assets/js/SplitText.min.js', ['kfw-gsap'], KFE_CONST_VERSION, true );
        }

		wp_register_script( 'kfw-custom-text',
			KFE_CONST_URL . 'libraries/elementor/assets/js/custom-text' . KFE_CONST_DEBUG_SUFFIX . '.js',
			[ 'jquery', 'elementor-frontend' ],
			KFE_CONST_VERSION,
			true
		);

    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'kfw-custom-text' ];
	}

	/**
	 * Get script dependencies.
	 *
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @return array Element scripts dependencies.
	 */
	public function get_script_depends() {
        return [ 'kfw-gsap', 'kfw-gsap-scroll-trigger', 'kfw-gsap-split-text', 'kfw-custom-text' ];
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
        $this->_register_items_section();
		$this->_register_settings_section();

		$this->_register_anim_6_style_section();
		$this->_register_primary_typo_style_section();
		$this->_register_secondary_typo_style_section();
    }

    public function _register_items_section() {
		$this->start_controls_section( 'section_items', [
			'label' => esc_html__( 'Items', 'onnat-extra' ),
			'tab'   => KFCM::TAB_CONTENT,
		]);
            $this->add_control( 'items', [
                    'type'    => KFCM::REPEATER,
                    'fields'  => [
                        [
                            'name'    => 'source',
                            'label'   => esc_html__( 'Content Source', 'onnat-extra' ),
                            'type'    => KFCM::CHOOSE,
                            'dynamic' => [ 'active' => true ],
                            'options' => [
                                'editor'  => [
                                    'title' => esc_html__('Editor', 'onnat-extra'),
                                    'icon'  => 'eicon-text-area',
                                ],
                                'template'  => [
                                    'title' => esc_html__('Template', 'onnat-extra'),
                                    'icon'  => 'eicon-t-letter',
                                ],
                            ],
                            'default' => 'editor',
                            'toggle'  => false,
                        ],
                        [
                            'name'        => 'editor',
                            'label'       => esc_html__( 'Editor', 'onnat-extra' ),
                            'type'        => KFCM::TEXTAREA,
                            'label_block' => true,
                            'dynamic'     => [ 'active' => true ],
                            'default'     => esc_html__( 'Enter your title', 'onnat-extra' ),
                            'description' => esc_html__( '"Secondary Typo" Settings will take effect if you use this {{something}} format.', 'onnat-extra' ),
                            'condition'   => [
                                'source' => 'editor',
                            ],
                        ],
                        [
                            'name'        => 'template',
                            'label'       => esc_html__( 'Template', 'onnat-extra' ),
                            'type'        => KFCM::SELECT,
                            'label_block' => true,
                            'dynamic'     => [ 'active' => true ],
                            'options'     => kf_extra_plugin_elementor()->get_elementor_library(),
                            'default'     => 0,
                            'condition'   => [
                                'source' => 'template',
                            ],
                        ],
                    ],
                    'default' => [
                        [
                            'source' => 'editor',
                            'editor' => 'Donec gravida risus sed {{ligula}} scelerisque laoreet',
                        ],
                        [
                            'source' => 'editor',
                            'editor' => 'Etiam interdum orci id mattis {{interdum gravida}} ligula commodo',
                        ],
                        [
                            'source' => 'editor',
                            'editor' => 'ullamcorper non laoreet id maximus ut sapien sollicitudin gravida diam',
                        ],
                        [
                            'source' => 'editor',
                            'editor' => 'vestibulum est ut nisi'
                        ]
                    ],
            ]);
        $this->end_controls_section();
    }

	public function _register_settings_section() {
		$this->start_controls_section( 'section_settings', [
			'label' => esc_html__( 'Settings', 'onnat-extra' )
		]);
            $this->add_responsive_control('align',[
				'label'        => esc_html__( 'Alignment', 'onnat-extra' ),
				'type'         => KFCM::CHOOSE,
				'options'      => [
					'left'    => [
                        'title' => esc_html__( 'Left', 'onnat-extra' ),
                        'icon'  => 'eicon-text-align-left',
                    ],
					'center'  => [
                        'title' => esc_html__( 'Center', 'onnat-extra' ),
                        'icon'  => 'eicon-text-align-center',
                    ],
					'right'   => [
                        'title' => esc_html__( 'Right', 'onnat-extra' ),
                        'icon'  => 'eicon-text-align-right',
                    ],
				],
				'toggle'               => false,
				'default'              => 'center',
				'tablet_default'       => 'center',
				'mobile_default'       => 'center',
				'selectors_dictionary' => [
					'left'   => 'text-align:left;',
					'center' => 'text-align:center;',
					'right'  => 'text-align:right;',
				],
				'selectors'            => [
                    '{{WRAPPER}} .kfw-ele-widget-custom-text-main' => '{{VALUE}};',
                ],
			]);
            $this->add_control('anim_title',[
				'label' => esc_html__( 'Animation?', 'onnat-extra' ),
				'type'  => KFCM::SWITCHER,
			]);
			$this->add_control('anim_type',[
				'label'        => esc_html__( 'Type', 'onnat-extra' ),
				'type'         => KFCM::SELECT,
				'default'      => 'style-1',
				'options'      => [
                    'style-1' => esc_html__('Style 1', 'onnat-extra'),
                    'style-2' => esc_html__('Style 2', 'onnat-extra'),
                    'style-3' => esc_html__('Style 3', 'onnat-extra'),
                    'style-4' => esc_html__('Style 4', 'onnat-extra'),
                    'style-5' => esc_html__('Style 5', 'onnat-extra'),
                    'style-6' => esc_html__('Style 6', 'onnat-extra'),
                ],
                'prefix_class' => 'kfw-custom-text-anim kfw-custom-text-anim-',
                'condition'    => [
					'anim_title' => 'yes'
				],
			]);
		$this->end_controls_section();
    }

    public function _register_anim_6_style_section() {
		$this->start_controls_section( 'section_anim_6_style', [
			'label' => esc_html__( 'Animation 6 Colors', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
            'condition' => [
                'anim_title' => 'yes',
                'anim_type'  => 'style-6'
            ],
		]);
            $this->add_control( 'anim_6_primary_color', [
                'label'     => esc_html__('Primary Color', 'onnat-extra' ),
                'type'      => KFCM::COLOR,
                'selectors' => ['{{WRAPPER}}' => '--kfw-custom-text-anim-primary-color:{{VALUE}};' ],
                'condition' => [
                    'anim_title' => 'yes',
                    'anim_type' => 'style-6'
                ],
            ]);
            $this->add_control( 'anim_6_secondary_color', [
                'label'     => esc_html__('Secondary Color', 'onnat-extra' ),
                'type'      => KFCM::COLOR,
                'selectors' => ['{{WRAPPER}}' => '--kfw-custom-text-anim-secondary-color:{{VALUE}};' ],
                'condition' => [
                    'anim_title' => 'yes',
                    'anim_type' => 'style-6'
                ],
            ]);
		$this->end_controls_section();
    }

    public function _register_primary_typo_style_section() {
		$this->start_controls_section( 'section_primary_style', [
			'label' => esc_html__( 'Primary Text', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_responsive_control( 'primary_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-custom-text-main .elementor-repeater-item' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'primary_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-custom-text-main .elementor-repeater-item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_group_control(KFGCT::get_type(), [
                'name'      => 'primary_typo',
                'label'     => esc_html__( 'Typography', 'onnat-extra' ),
                'selector'  => '{{WRAPPER}} .kfw-ele-widget-custom-text-main .elementor-repeater-item.kfw-ele-widget-custom-text-editor-item',
            ]);
            $this->add_control( 'primary_color', [
                'label'     => esc_html__('Color', 'onnat-extra' ),
                'type'      => KFCM::COLOR,
                'selectors' => ['{{WRAPPER}}' => '--kfw-custom-text-primary-color:{{VALUE}};' ],
                'condition' => [
                    'anim_type' => ['style-1','style-2','style-3','style-4','style-5']
                ],
            ]);
            $this->add_group_control( KFGCTS::get_type(), [
                'name'      => 'primary_txt_shadow',
                'selector'  => '{{WRAPPER}} .kfw-ele-widget-custom-text-main .elementor-repeater-item.kfw-ele-widget-custom-text-editor-item',
            ]);
            $this->add_group_control( KFGCTST::get_type(), [
                'name'      => 'primary_txt_stroke',
                'selector'  => '{{WRAPPER}} .kfw-ele-widget-custom-text-main .elementor-repeater-item.kfw-ele-widget-custom-text-editor-item',
            ]);
		$this->end_controls_section();
    }

    public function _register_secondary_typo_style_section() {
		$this->start_controls_section( 'section_secondary_style', [
			'label' => esc_html__( 'Secondary Text', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_responsive_control( 'secondary_margin', [
                'label'      => esc_html__('Margin', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-custom-text-main kfw-ele-widget-custom-text-secondary-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_responsive_control( 'secondary_padding', [
                'label'      => esc_html__('Padding', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-custom-text-main kfw-ele-widget-custom-text-secondary-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_group_control(KFGCT::get_type(), [
                'name'      => 'secondary_typo',
                'label'     => esc_html__( 'Typography', 'onnat-extra' ),
                'selector'  => '{{WRAPPER}} .kfw-ele-widget-custom-text-main .kfw-ele-widget-custom-text-editor-item kfw-ele-widget-custom-text-secondary',
            ]);
            $this->add_control( 'secondary_color', [
                'label'     => esc_html__('Color', 'onnat-extra' ),
                'type'      => KFCM::COLOR,
                'selectors' => ['{{WRAPPER}}' => '--kfw-custom-text-secondary-color:{{VALUE}};' ],
                'condition' => [
                    'anim_type' => ['style-1','style-2','style-3','style-4','style-5']
                ],
            ]);
            $this->add_group_control( KFGCTS::get_type(), [
                'name'      => 'secondary_txt_shadow',
                'selector'  => '{{WRAPPER}} .kfw-ele-widget-custom-text-main .kfw-ele-widget-custom-text-editor-item kfw-ele-widget-custom-text-secondary',
            ]);
            $this->add_group_control( KFGCTST::get_type(), [
                'name'      => 'secondary_txt_stroke',
                'selector'  => '{{WRAPPER}} .kfw-ele-widget-custom-text-main .kfw-ele-widget-custom-text-editor-item kfw-ele-widget-custom-text-secondary',
            ]);
		$this->end_controls_section();
    }

	/**
	 * Generate setting json
	 */
	protected function _generate_setting_json( array $settings ) {

        $options['anim_title'] = !empty( $settings['anim_title'] ) ? $settings['anim_title'] : 'no';
        $options['anim_type']  = $settings['anim_type'];
        $options['split']      = ( 'style-6' === $settings['anim_type'] ) ? false : true;

		return wp_json_encode( $options );
    }

	/**
	 * Get template content
	 */
	public function _generate_template_content( $template_id ) {
		return KFEl::instance()->frontend->get_builder_content_for_display( $template_id );
	}

    public function _parse_text_editor( $content ) {
        $content = $this->parse_text_editor( $content );

        $content = str_replace( array( '{{', '}}' ), [
            '<kfw-ele-widget-custom-text-secondary-wrap> <kfw-ele-widget-custom-text-secondary>',
            '</kfw-ele-widget-custom-text-secondary></kfw-ele-widget-custom-text-secondary-wrap>'
        ], $content );

        return $content;
    }

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-custom-text");
		include $render;
    }
}