<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use Elementor\Widget_Base as WB;
use Elementor\Controls_Manager as KFCM;

use Elementor\Group_Control_Border as KFGCBR;
use Elementor\Group_Control_Css_Filter as KFGCSSF;
use Elementor\Group_Control_Typography as KFGCT;

class KFW_Google_Map extends WB {

    public $gmap_api_key;

	public $gmap_id;

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-google-map';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Google Map', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_core_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-google-maps';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'kfw', 'kin', 'map', 'google', 'earth' ];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
        return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/google-map';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

        $this->gmap_api_key = kinfw_onnat_theme_options()->kinfw_get_option( 'gmap_api_key' );
		$this->gmap_id = isset( $data['id'] ) ? $data['id'] : rand( 1, 100 );

		wp_register_script( 'kfw-google-map-api', add_query_arg([ 'key' => $this->gmap_api_key, ], 'https://maps.googleapis.com/maps/api/js'), KFE_CONST_VERSION, true  );

		wp_register_script( 'kfw-google-map',
			KFE_CONST_URL . 'libraries/elementor/assets/js/google-map' . KFE_CONST_DEBUG_SUFFIX . '.js',
			[ 'jquery', 'elementor-frontend' ],
			KFE_CONST_VERSION,
			true
		);

		wp_register_style( 'kfw-google-map',
			KFE_CONST_URL . 'libraries/elementor/assets/css/google-map' . KFE_CONST_DEBUG_SUFFIX . '.css',
			[],
			KFE_CONST_VERSION
		);

    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'kfw-google-map' ];
	}

	/**
	 * Get script dependencies.
	 *
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @return array Element scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'kfw-google-map-api', 'kfw-google-map' ];
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
		$this->_register_map_settings_section();
		$this->_register_map_markers_section();
		$this->_register_map_style_section();
		$this->_register_marker_style_section();
	}

    public function _register_map_settings_section() {
		$this->start_controls_section('section_map_settings', [
			'label' => esc_html__('Map Settings', 'onnat-extra' )
		]);
            $this->add_control('center_latitude', [
				'label'       => esc_html__( 'Map Center Latitude', 'onnat-extra' ),
				'label_block' => true,
				'type'        => KFCM::TEXT,
				'default'     => 11,
				'description' => sprintf(
                    esc_html__('Easily obtain location coordinates by clicking %s.', 'onnat-extra' ),
                    sprintf( '<a href="https://www.latlong.net/" target="_blank">%s</a>', esc_html__('here', 'onnat-extra') )
                ),
            ]);
            $this->add_control('center_longitude', [
                'label'       => esc_html__( 'Map Center Longitude', 'onnat-extra' ),
                'label_block' => true,
                'type'        => KFCM::TEXT,
				'default'     => 77,
                'description' => sprintf(
                    esc_html__('Easily obtain location coordinates by clicking %s.', 'onnat-extra' ),
                    sprintf( '<a href="https://www.latlong.net/" target="_blank">%s</a>', esc_html__('here', 'onnat-extra') )
                ),
            ]);
            $this->add_control('zoom', [
                'label'   => esc_html__( 'Initial Zoom', 'onnat-extra' ),
				'type'    => KFCM::SLIDER,
				'default' => [
                    'unit' => 'zoom',
                    'size' => 10
                ],
				'range'   => [
                    'zoom' => [ 'min' => 1, 'max' => 18, 'step' => 1 ]
                ]
            ]);
			$this->add_control('map_type',[
				'label'              => esc_html__( 'Inital Map Type', 'onnat-extra' ),
				'type'               => KFCM::CHOOSE,
				'options'            => [
					'hybrid'    => [
						'title' => esc_html__('Hybrid View', 'onnat-extra'),
						'icon'  => 'kfw-ctrl-icon kfw-ctrl-h-icon',
					],
					'roadmap'   => [
						'title' => esc_html__('Roadmap View', 'onnat-extra'),
						'icon'  => 'kfw-ctrl-icon kfw-ctrl-r-icon',
					],
					'satellite' => [
						'title' => esc_html__('Satellite View', 'onnat-extra'),
						'icon'  => 'kfw-ctrl-icon kfw-ctrl-s-icon',
					],
					'terrain'   => [
						'title' => esc_html__('Terrain View', 'onnat-extra'),
						'icon'  => 'kfw-ctrl-icon kfw-ctrl-t-icon',
					],
				],
				'default'            => 'roadmap',
				'toggle'             => false,
				'frontend_available' => true,
            ]);

			$this->add_control('map_controls_before_hr', [
				'type' => KFCM::DIVIDER,
			]);
			$this->add_control('map_controls_heading',[
				'label' => esc_html__( 'CONTROLS', 'onnat-extra' ),
				'type'  => KFCM::HEADING,
			]);
			$this->add_control('draggable_control', [
				'label'        => esc_html__( 'Is Map Draggable ?', 'onnat-extra' ),
				'type'         => KFCM::SWITCHER,
				'return_value' => 'yes',
				'default'      => 'yes'
			]);
            $this->add_control('fullscreen_control', [
				'label'        => esc_html__( 'Full Screen', 'onnat-extra' ),
				'type'         => KFCM::SWITCHER,
				'label_on'     => esc_html__( 'On', 'onnat-extra' ),
				'label_off'    => esc_html__( 'Off', 'onnat-extra' ),
				'return_value' => 'yes',
				'default'      => 'yes'
			]);
            $this->add_control('map_type_control', [
				'label'        => esc_html__( 'Map Type', 'onnat-extra' ),
				'type'         => KFCM::SWITCHER,
				'label_on'     => esc_html__( 'On', 'onnat-extra' ),
				'label_off'    => esc_html__( 'Off', 'onnat-extra' ),
				'return_value' => 'yes',
				'default'      => 'yes'
			]);
            $this->add_control('rotate_control', [
				'label'        => esc_html__( 'Rotate', 'onnat-extra' ),
				'type'         => KFCM::SWITCHER,
				'label_on'     => esc_html__( 'On', 'onnat-extra' ),
				'label_off'    => esc_html__( 'Off', 'onnat-extra' ),
				'return_value' => 'yes',
				'default'      => 'yes'
			]);
            $this->add_control('scale_control', [
				'label'        => esc_html__( 'Scale', 'onnat-extra' ),
				'type'         => KFCM::SWITCHER,
				'label_on'     => esc_html__( 'On', 'onnat-extra' ),
				'label_off'    => esc_html__( 'Off', 'onnat-extra' ),
				'return_value' => 'yes',
				'default'      => 'yes'
			]);
            $this->add_control('scrollwheel_control', [
				'label'        => esc_html__( 'Scrollwheel Zoom', 'onnat-extra' ),
				'type'         => KFCM::SWITCHER,
				'label_on'     => esc_html__( 'On', 'onnat-extra' ),
				'label_off'    => esc_html__( 'Off', 'onnat-extra' ),
				'return_value' => 'yes',
				'default'      => 'yes'
			]);
            $this->add_control('street_view_control', [
				'label'        => esc_html__( 'Street View', 'onnat-extra' ),
				'type'         => KFCM::SWITCHER,
				'label_on'     => esc_html__( 'On', 'onnat-extra' ),
				'label_off'    => esc_html__( 'Off', 'onnat-extra' ),
				'return_value' => 'yes',
				'default'      => 'no'
			]);
            $this->add_control('zoom_control', [
				'label'        => esc_html__( 'Zoom', 'onnat-extra' ),
				'type'         => KFCM::SWITCHER,
				'label_on'     => esc_html__( 'On', 'onnat-extra' ),
				'label_off'    => esc_html__( 'Off', 'onnat-extra' ),
				'return_value' => 'yes',
				'default'      => 'yes'
			]);
		$this->end_controls_section();
    }

	public function _register_map_markers_section() {
		$this->start_controls_section('section_map_markers', [
			'label' => esc_html__('Markers', 'onnat-extra' ),
		]);
			$this->add_control( 'items', [
				'type'        => KFCM::REPEATER,
				'fields'      => [
					[
						'name'        => 'name',
						'label'       => esc_html__( 'Name', 'onnat-extra' ),
						'type'        => KFCM::TEXT,
						'label_block' => true,
						'dynamic'     => [ 'active' => true ],
					],
					[
						'name'        => 'latitude',
						'label'       => esc_html__( 'Latitude', 'onnat-extra' ),
						'type'        => KFCM::TEXT,
						'label_block' => false,
						'dynamic'     => [ 'active' => true ],
					],
					[
						'name'        => 'longitude',
						'label'       => esc_html__( 'Longitude', 'onnat-extra' ),
						'type'        => KFCM::TEXT,
						'label_block' => false,
						'dynamic'     => [ 'active' => true ],
					],
					[
						'name'        => 'content',
						'label'       => esc_html__( 'Content', 'onnat-extra' ),
						'type'        => KFCM::WYSIWYG,
						'label_block' => true,
					],
					[
						'name'        => 'phone',
						'label'       => esc_html__( 'Phone Number', 'onnat-extra' ),
						'type'        => KFCM::TEXT,
						'label_block' => false,
						'dynamic'     => [ 'active' => true ],
					],
					[
						'name'        => 'website',
						'label'       => esc_html__( 'Website', 'onnat-extra' ),
						'type'        => KFCM::URL,
						'label_block' => true,
						'dynamic'     => [ 'active' => true ],
						'options'     => false,
						'placeholder' => 'https://your-link.com',
					],
					[
						'name'        => 'state',
						'label'       => esc_html__( 'Initial State?', 'onnat-extra' ),
						'type'        => KFCM::CHOOSE,
						'label_block' => false,
						'dynamic'     => [ 'active' => true ],
						'options'     => [
							'visible' => [
								'title' => esc_html__('Visible', 'onnat-extra'),
								'icon'  => 'eicon-dot-circle-o'
							],
							'hidden'  => [
								'title' => esc_html__('Hidden', 'onnat-extra'),
								'icon'  => 'eicon-circle-o'
							],
						],
						'default'     => 'hidden',
						'toggle'      => false,
					],
					[
						'name'        => 'pin',
						'label'       => esc_html__( 'Custom Pin', 'onnat-extra' ),
						'type'        => KFCM::MEDIA,
						'label_block' => true,
						'dynamic'     => [ 'active' => true ],
						'default'     => [],
						'description' => esc_html__('Use max 32x32 px size icon for better result.', 'onnat-extra'),
					]
				],
				'title_field' => '{{{name}}}',
				'default'     => []
            ]);
		$this->end_controls_section();
	}

	public function _register_map_style_section() {
		$this->start_controls_section('section_map_style', [
			'label' => esc_html__('Map', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_responsive_control('map_height',[
				'label'     => esc_html__('Height', 'onnat-extra' ),
				'type'      => KFCM::SLIDER,
				'range'     => [
					'px' => [
						'min' => 100,
						'max' => 1500,
					]
				],
				'default'   => ['size' => 300,],
				'selectors' => [
					'{{WRAPPER}}' => '--kfw-map-height: {{SIZE}}{{UNIT}};',
				],
			]);
            $this->add_control('map_color_scheme', [
				'label'       => esc_html__('Color Scheme', 'onnat-extra' ),
				'type'        => KFCM::TEXTAREA,
				'default'     => '',
				'description' => sprintf(
					esc_html__('Visit %1s snazzymaps.com %2s, choose a style, copy the entire JSON code, and paste it here.', 'onnat-extra' ),
					'<a href="https://snazzymaps.com/" target="_blank">',
					'</a>'
				)
			]);
			$this->start_controls_tabs( 'map_customize_tabs' );
				$this->start_controls_tab( 'map_normal_state_tab', [
					'label' => esc_html__( 'Normal', 'onnat-extra' ),
				]);
					$this->add_group_control( KFGCSSF::get_type(), [
						'name'      => 'normal_css_filters',
						'selector'  => '{{WRAPPER}} .kfw-ele-widget-google-map-main',
					]);
					$this->add_group_control( KFGCBR::get_type(), [
						'name'           => 'map_border',
						'fields_options' => [
							'width' => [ 'label' => _x( 'Border Width', 'Border Control', 'onnat-extra' ) ],
							'color' => [ 'label' => _x( 'Border Color', 'Border Control', 'onnat-extra' ) ],
						],
						'selector'       => '{{WRAPPER}} .kfw-ele-widget-google-map-main',
					]);
					$this->add_responsive_control( 'map_border_radius', [
						'label'      => esc_html__('Border Radius', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', '%' ],
						'selectors'  => [ '{{WRAPPER}} .kfw-ele-widget-google-map-main' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};' ],
					]);
				$this->end_controls_tab();
				$this->start_controls_tab( 'map_hover_state_tab', [
					'label' => esc_html__( 'Hover', 'onnat-extra' ),
				]);
					$this->add_group_control( KFGCSSF::get_type(), [
						'name'      => 'hover_css_filters',
						'selector'  => '{{WRAPPER}} .kfw-ele-widget-google-map-main:hover',
					]);
				$this->end_controls_tab();
			$this->end_controls_tabs();
		$this->end_controls_section();
	}

	public function _register_marker_style_section() {
		$this->start_controls_section('section_marker_style', [
			'label' => esc_html__('Marker', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_responsive_control( 'marker_padding', [
				'label'      => esc_html__( 'Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-google-map .gm-style .gm-style-iw-c' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
			]);
			$this->start_controls_tabs( 'marker_field_customize_tabs' );
				$this->start_controls_tab( 'name_field_normal_state', [
					'label' => esc_html__( 'Name', 'onnat-extra' ),
				]);
					$this->add_group_control( KFGCT::get_type(), [
						'name'        => 'name_field_typo',
						'description' => esc_html__( 'Typography', 'onnat-extra' ),
						'selector'    => '{{WRAPPER}} .kfw-ele-widget-google-map .kfw-google-map-name',
					]);
					$this->add_control( 'name_field_color', [
						'label'     => esc_html__('Text Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => ['{{WRAPPER}} .kfw-ele-widget-google-map .kfw-google-map-name' => 'color: {{VALUE}};' ],
					]);
					$this->add_responsive_control( 'name_field_padding', [
						'label'      => esc_html__( 'Padding', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', 'em', '%' ],
						'selectors'  => [
							'{{WRAPPER}} .kfw-ele-widget-google-map .kfw-google-map-name' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
						],
					]);
					$this->add_responsive_control( 'name_field_margin', [
						'label'      => esc_html__( 'Margin', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', 'em', '%' ],
						'separator'  => 'after',
						'selectors'  => [
							'{{WRAPPER}} .kfw-ele-widget-google-map .kfw-google-map-name' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
						],
					]);
				$this->end_controls_tab();
				$this->start_controls_tab( 'content_field_normal_state', [
					'label' => esc_html__( 'Content', 'onnat-extra' ),
				]);
					$this->add_group_control( KFGCT::get_type(), [
						'name'        => 'content_field_typo',
						'description' => esc_html__( 'Typography', 'onnat-extra' ),
						'selector'    => '{{WRAPPER}} .kfw-ele-widget-google-map .kfw-google-map-content',
					]);
					$this->add_control( 'content_field_color', [
						'label'     => esc_html__('Text Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => ['{{WRAPPER}} .kfw-ele-widget-google-map .kfw-google-map-content' => 'color: {{VALUE}};' ],
					]);
					$this->add_responsive_control( 'content_field_padding', [
						'label'      => esc_html__( 'Padding', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', 'em', '%' ],
						'selectors'  => [
							'{{WRAPPER}} .kfw-ele-widget-google-map .kfw-google-map-content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
						],
					]);
					$this->add_responsive_control( 'content_field_margin', [
						'label'      => esc_html__( 'Margin', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', 'em', '%' ],
						'separator'  => 'after',
						'selectors'  => [
							'{{WRAPPER}} .kfw-ele-widget-google-map .kfw-google-map-content' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
						],
					]);
				$this->end_controls_tab();
				$this->start_controls_tab( 'pno_field_normal_state', [
					'label' => esc_html__( 'Phone No', 'onnat-extra' ),
				]);
					$this->add_group_control( KFGCT::get_type(), [
						'name'        => 'pno_field_typo',
						'description' => esc_html__( 'Typography', 'onnat-extra' ),
						'selector'    => '{{WRAPPER}} .kfw-ele-widget-google-map .kfw-google-map-phone',
					]);
					$this->add_control( 'pno_field_color_settings', [
						'label'        => esc_html__('Text Color', 'onnat-extra'),
						'type'         => KFCM::POPOVER_TOGGLE,
						'return_value' => 'yes',
					]);
					$this->start_popover();
						$this->add_control( 'pno_field_color', [
							'label'     => esc_html__('Normal', 'onnat-extra' ),
							'type'      => KFCM::COLOR,
							'selectors' => ['{{WRAPPER}} .kfw-ele-widget-google-map .kfw-google-map-phone a' => 'color: {{VALUE}};' ],
						]);
						$this->add_control( 'pno_field_hover_color', [
							'label'     => esc_html__('Hover', 'onnat-extra' ),
							'type'      => KFCM::COLOR,
							'selectors' => ['{{WRAPPER}} .kfw-ele-widget-google-map .kfw-google-map-phone a:hover' => 'color: {{VALUE}};' ],
						]);
					$this->end_popover();
					$this->add_responsive_control( 'pno_field_padding', [
						'label'      => esc_html__( 'Padding', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', 'em', '%' ],
						'selectors'  => [
							'{{WRAPPER}} .kfw-ele-widget-google-map .kfw-google-map-phone' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
						],
					]);
					$this->add_responsive_control( 'pno_field_margin', [
						'label'      => esc_html__( 'Margin', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', 'em', '%' ],
						'separator'  => 'after',
						'selectors'  => [
							'{{WRAPPER}} .kfw-ele-widget-google-map .kfw-google-map-phone' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
						],
					]);
				$this->end_controls_tab();
				$this->start_controls_tab( 'website_field_normal_state', [
					'label' => esc_html__( 'Website', 'onnat-extra' ),
				]);
					$this->add_group_control( KFGCT::get_type(), [
						'name'        => 'website_field_typo',
						'description' => esc_html__( 'Typography', 'onnat-extra' ),
						'selector'    => '{{WRAPPER}} .kfw-ele-widget-google-map .kfw-google-map-website',
					]);
					$this->add_control( 'website_field_color_settings', [
						'label'        => esc_html__('Text Color', 'onnat-extra'),
						'type'         => KFCM::POPOVER_TOGGLE,
						'return_value' => 'yes',
					]);
					$this->start_popover();
						$this->add_control( 'website_field_color', [
							'label'     => esc_html__('Normal', 'onnat-extra' ),
							'type'      => KFCM::COLOR,
							'selectors' => ['{{WRAPPER}} .kfw-ele-widget-google-map .kfw-google-map-website a' => 'color: {{VALUE}};' ],
						]);
						$this->add_control( 'website_field_hover_color', [
							'label'     => esc_html__('Hover', 'onnat-extra' ),
							'type'      => KFCM::COLOR,
							'selectors' => ['{{WRAPPER}} .kfw-ele-widget-google-map .kfw-google-map-website a:hover' => 'color: {{VALUE}};' ],
						]);
					$this->end_popover();
					$this->add_responsive_control( 'website_field_padding', [
						'label'      => esc_html__( 'Padding', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', 'em', '%' ],
						'selectors'  => [
							'{{WRAPPER}} .kfw-ele-widget-google-map .kfw-google-map-website' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
						],
					]);
					$this->add_responsive_control( 'website_field_margin', [
						'label'      => esc_html__( 'Margin', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', 'em', '%' ],
						'separator'  => 'after',
						'selectors'  => [
							'{{WRAPPER}} .kfw-ele-widget-google-map .kfw-google-map-website' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
						],
					]);
				$this->end_controls_tab();
			$this->end_controls_tabs();

		$this->end_controls_section();
	}

	public function _message( $message, $echo = true ) {
		$return = sprintf( '
			<div class="elementor-alert elementor-alert-danger" role="alert">
				<span class="elementor-alert-description">%s</span>
			</div>',
			$message
		);

		if( $echo ) {
			echo $return;
			return;
		}

		return $return;
	}

	/**
	 * Generate setting json
	 */
	protected function _generate_setting_json( array $settings ) {
		$options = [];
		$lat     = filter_var( $settings['center_latitude'], FILTER_VALIDATE_FLOAT );
		$lon     = filter_var( $settings['center_longitude'], FILTER_VALIDATE_FLOAT );

		if( false !== $lat && false !== $lon ) {

			$options = [
				'center'            => [
					'lat' => $lat,
					'lng' => $lon
				],
				'zoom'              => filter_var( $settings['zoom']['size'], FILTER_VALIDATE_INT ),
				'mapTypeId'         => $settings['map_type'],
				'zoomControl'       => filter_var( $settings['zoom_control'], FILTER_VALIDATE_BOOLEAN ),
				'mapTypeControl'    => filter_var( $settings['map_type_control'], FILTER_VALIDATE_BOOLEAN ),
				'scaleControl'      => filter_var( $settings['scale_control'], FILTER_VALIDATE_BOOLEAN ),
				'streetViewControl' => filter_var( $settings['street_view_control'], FILTER_VALIDATE_BOOLEAN ),
				'rotateControl'     => filter_var( $settings['rotate_control'], FILTER_VALIDATE_BOOLEAN ),
				'fullscreenControl' => filter_var( $settings['fullscreen_control'], FILTER_VALIDATE_BOOLEAN ),
				'scrollwheel'       => filter_var( $settings['scrollwheel_control'], FILTER_VALIDATE_BOOLEAN ),
				'draggable'         => filter_var( $settings['draggable_control'], FILTER_VALIDATE_BOOLEAN ),
			];

			if( isset( $settings['map_color_scheme'] ) ) {
				$style = trim(preg_replace('/\s+/', ' ', $settings['map_color_scheme']));
				if( !empty( $style ) ) {
					$options['styles'] = json_decode( $style );
				}
			}
		}

		return wp_json_encode( $options );
	}

	/**
	 * Generate markers json
	 */
	protected function _generate_markers( array $settings ) {
		$markers = [];

		foreach( $settings['items'] as $marker ) {

			$lat = filter_var( $marker['latitude'], FILTER_VALIDATE_FLOAT );
			$lon = filter_var( $marker['longitude'], FILTER_VALIDATE_FLOAT );

			if( false !== $lat && false !== $lon ) {
				$current   = [
					'key'          => $marker['_id'],
					'latitude'     => $lat,
					'longitude'    => $lon,
					'name'         => $marker['name'],
					'content'      => $marker['content'],
					'phone'        => $marker['phone'],
					'websiteLink' => $marker['website']['url'],
					'websiteName' => preg_replace("(^https?://)", "", $marker['website']['url']),
					'state'        => $marker['state'],
				];

                if ( ! empty( $marker['pin']['url'] ) ) {
                    $current['pin'] = esc_url( $marker['pin']['url'] );
                }

				$markers[] = $current;
			}

		}

		return wp_json_encode( $markers );
	}

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-google-map");
		include $render;
    }
}