<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use \Elementor\Widget_Base as KFWB;
use \Elementor\Controls_Manager as KFCM;
use \Elementor\Group_Control_Image_Size as KFGCIS;
use \Elementor\Group_Control_Typography as KFGCT;
use \Elementor\Group_Control_Border as KFGCBR;
use \Elementor\Group_Control_Background as KFGCBG;
use \Elementor\Group_Control_Box_Shadow as KFGCBS;
use \Elementor\Group_Control_Text_Shadow as KFGCTS;
use \Elementor\Group_Control_Text_Stroke as KFGCTST;
use \Elementor\Repeater as KFRTR;

class KFW_Image_Expand extends KFWB {
	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-image-expand';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Image Expand', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_core_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-spacer';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'kin', 'kf', 'images', 'expand', 'image'  ];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
		return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/image-expand';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

		wp_register_script( 'kfw-img-expand',
			KFE_CONST_URL . 'libraries/elementor/assets/js/image-expand' . KFE_CONST_DEBUG_SUFFIX . '.js',
			[ 'jquery', 'elementor-frontend' ],
			KFE_CONST_VERSION,
			true
		);

		wp_register_style( 'kfw-img-expand',
			KFE_CONST_URL . 'libraries/elementor/assets/css/image-expand' . KFE_CONST_DEBUG_SUFFIX . '.css',
			[],
			KFE_CONST_VERSION
		);
    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'kfw-img-expand' ];
	}

	/**
	 * Get script dependencies.
	 *
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @return array Element scripts dependencies.
	 */
	public function get_script_depends() {
		$scripts = [ 'imagesloaded', 'kfw-img-expand' ];

		if( $this->_is_edit_mode() ) {
			$scripts = ['imagesloaded', 'kfe-elementor-editor', 'kfw-img-expand' ];
		}

		return $scripts;
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
		$this->_register_items_section();
		$this->_register_settings_section();

		$this->_register_container_style_section();
		$this->_register_item_style_section();
		$this->_register_item_title_style_section();
		$this->_register_item_sub_title_style_section();
		$this->_register_item_content_style_section();
		$this->_register_item_content_boxed_style_section();
		$this->_register_item_action_btn_style_section();
    }

	public function _register_items_section() {
		$this->start_controls_section( 'section_items', [
			'label' => esc_html__( 'Items', 'onnat-extra' )
        ]);

			$repeater = new KFRTR();

			$repeater->start_controls_tabs('tabs');

				/**
				 * Tab: Image
				 */
				$repeater->start_controls_tab('tab_image',[
					'label' => esc_html__( 'Image', 'onnat-extra' )
				]);
					$repeater->add_control('image',[
						'label'   => esc_html__( 'Image', 'onnat-extra' ),
						'type'    => KFCM::MEDIA,
						'default' => [ 'url' => KFE_CONST_URL .'libraries/elementor/assets/images/image-placeholder-'.rand(1,5).'.svg' ],
						'dynamic' => [ 'active' => true ],
					]);
				$repeater->end_controls_tab();

				/**
				 * Tab: Content
				 */
				$repeater->start_controls_tab('tab_content',[
					'label' => esc_html__( 'Content', 'onnat-extra' )
				]);
					$repeater->add_control('title',[
						'label'       => esc_html__( 'Title', 'onnat-extra' ),
						'type'        => KFCM::TEXT,
						'dynamic'     => [ 'active' => true ],
						'default'     => esc_html__( 'Title' , 'onnat-extra' ),
						'label_block' => true,
					]);
					$repeater->add_control('sub_title',[
						'label'       => esc_html__( 'Sub Title', 'onnat-extra' ),
						'type'        => KFCM::TEXT,
						'dynamic'     => [ 'active' => true ],
						'label_block' => true,
					]);
					$repeater->add_control('button',[
						'label' => esc_html__('Button', 'onnat-extra'),
						'type'  => KFCM::POPOVER_TOGGLE,
					]);
						$repeater->start_popover();
							$repeater->add_control('button_txt',[
								'label'   => esc_html__( 'Button Text', 'onnat-extra' ),
								'type'    => KFCM::TEXT,
								'dynamic' => [ 'active' => true ],
								'default' => esc_html__( 'Read More' , 'onnat-extra' ),
							]);
							$repeater->add_control('button_link',[
								'label'     => esc_html__( 'Button Link', 'onnat-extra' ),
								'type'      => KFCM::URL,
								'dynamic'   => [
									'active' => true
								],
								'options'   => [
									'url',
									'is_external',
									'nofollow'
								],
								'default'   => [
									'url'         => '#',
									'is_external' => true,
									'nofollow'    => true,
								],
								'condition' => [
									'button_txt!' => ''
								]
							]);
						$repeater->end_popover();
					$repeater->add_control('content',[
						'label'   => esc_html__( 'Content', 'onnat-extra' ),
						'type'    => KFCM::WYSIWYG,
						'default' => 'Neque porro quisquam est qui dolorem ipsum quia dolor sit amet, consectetur, adipisci velit.',
						'dynamic' => [ 'active' => true ],
					]);
				$repeater->end_controls_tab();

			$repeater->end_controls_tabs();

			$this->add_control( 'items', [
				'type'    => KFCM::REPEATER,
				'fields'  => $repeater->get_controls(),
				'default' => [
					[
						'image' => [
							'url' => KFE_CONST_URL .'libraries/elementor/assets/images/image-placeholder-1.svg'
						]
					],
					[
						'image' => [
							'url' => KFE_CONST_URL .'libraries/elementor/assets/images/image-placeholder-2.svg'
						]
					],
					[
						'image' => [
							'url' => KFE_CONST_URL .'libraries/elementor/assets/images/image-placeholder-3.svg'
						]
					],
					[
						'image' => [
							'url' => KFE_CONST_URL .'libraries/elementor/assets/images/image-placeholder-4.svg'
						]
					],
					[
						'image' => [
							'url' => KFE_CONST_URL .'libraries/elementor/assets/images/image-placeholder-5.svg'
						]
					],
				]
			]);

		$this->end_controls_section();
	}

	public function _register_settings_section() {

        /**
         * Devices
         */
            $devices     = [];
            $breakpoints = \Elementor\Plugin::$instance->breakpoints->get_active_devices_list( [
                'reverse'       => true,
                'desktop_first' => true,
            ]);

            foreach( $breakpoints as $key => $device ) {
                $devices[ 'kfw-ele-hide-on-'. $device ] = ucwords( $device );
            }

        /**
         * Settings
         */
		$this->start_controls_section( 'section_settings', [
			'label' => esc_html__( 'Settings', 'onnat-extra' )
        ]);
			$this->add_group_control( KFGCIS::get_type(), [
				'name'    => 'image_size',
				'default' => 'full',
			]);
			$this->add_responsive_control('orientation',[
				'label'                => esc_html__( 'Orientation', 'onnat-extra' ),
				'type'                 => KFCM::CHOOSE,
				'options'              => [
					'vertical'   => [
						'title' => esc_html__('Vertical','onnat-extra'),
						'icon'  => 'eicon-navigation-vertical'
					],
					'horizontal' => [
						'title' => esc_html__('Horizontal','onnat-extra'),
						'icon'  => 'eicon-navigation-horizontal'
					],
				],
				'default'              => 'horizontal',
				'tablet_default'       => 'horizontal',
				'mobile_default'       => 'horizontal',
				'toggle'               => false,
				'prefix_class'         => 'kfw-ele-img-expand%s-orientation-',
				'selectors_dictionary' => [
                    'horizontal' => 'flex-direction: unset;',
                    'vertical'   => 'flex-direction: column;',
				],
				'selectors'            => [
					'{{WRAPPER}} .kfw-elementor-widget-img-expand-wrapper' => '{{VALUE}};',
				]
			]);
			$this->add_control('active_item',[
				'label' => esc_html__( 'Active Item?', 'onnat-extra' ),
				'type'  => KFCM::SWITCHER,
			]);
			$this->add_control('active_item_number',[
				'label'       => esc_html__( 'Item Number', 'onnat-extra' ),
				'description' => esc_html__( 'Type your item number', 'onnat-extra' ),
				'type'        => KFCM::NUMBER,
				'default'     => 1,
				'min'         => 1,
				'setp'        => 1,
				'condition'   => [
					'active_item' => 'yes'
				]
			]);
			$this->add_control('content_heading',[
				'label'     => esc_html__( 'Data Settings', 'onnat-extra' ),
				'type'      => KFCM::HEADING,
				'separator' => 'before',
			]);
			$this->add_control('show_title',[
                'label'   => esc_html__( 'Show Title?', 'onnat-extra' ),
                'type'    => KFCM::SWITCHER,
                'default' => 'yes'
			]);
			$this->add_control('show_sub_title',[
                'label'   => esc_html__( 'Show Sub Title?', 'onnat-extra' ),
                'type'    => KFCM::SWITCHER,
                'default' => 'yes'
			]);
			$this->add_control('show_button',[
                'label'   => esc_html__( 'Show Button?', 'onnat-extra' ),
                'type'    => KFCM::SWITCHER,
                'default' => 'yes'
			]);
			$this->add_control('show_content',[
                'label'   => esc_html__( 'Show Content?', 'onnat-extra' ),
                'type'    => KFCM::SWITCHER,
                'default' => 'yes'
			]);
			$this->add_control('hide_content_on',[
                'label'              => esc_html__( 'Hide Content on', 'onnat-extra' ),
                'type'               => KFCM::SELECT2,
                'multiple'           => true,
                'label_block'        => true,
                'options'            => $devices,
                'frontend_available' => true,
                'condition'          => [
                    'show_content' => 'yes'
                ]
            ]);
		$this->end_controls_section();
	}

	public function _register_container_style_section() {
		$this->start_controls_section( 'section_container_style',[
			'label' => esc_html__( 'Container', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_responsive_control('container_height',[
				'label'     => esc_html__( 'Height', 'onnat-extra' ),
				'type'      => KFCM::SLIDER,
				'range'     => [ 'px' => ['min' => 100, 'max' => 1000  ] ],
				'default'   => [
					'unit' => 'px',
					'size' => 600,
				],
				'selectors' => [
					'{{WRAPPER}} .kfw-elementor-widget-img-expand-wrapper' => 'height:{{SIZE}}{{UNIT}};'
				],
			]);
			$this->add_group_control( KFGCBG::get_type(), [
				'name'     => 'container_bg',
				'types'    => ['classic', 'gradient'],
				'exclude'  => [ 'image'],
				'selector' => '{{WRAPPER}} .kfw-elementor-widget-img-expand-wrapper',
			]);
			$this->add_group_control( KFGCBR::get_type(), [
				'name'     => 'border',
				'selector' => '{{WRAPPER}} .kfw-elementor-widget-img-expand-wrapper'
			]);
			$this->add_responsive_control( 'container_border_radius', [
				'label'      => esc_html__('Border Radius', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-elementor-widget-img-expand-wrapper' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_group_control( KFGCBS::get_type(), [
				'name'     => 'container_box_shadow',
				'selector' => '{{WRAPPER}} .kfw-elementor-widget-img-expand-wrapper'
			]);
			$this->add_responsive_control( 'container_padding',[
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-elementor-widget-img-expand-wrapper' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
		$this->end_controls_section();
	}

	public function _register_item_style_section() {
		$this->start_controls_section( 'section_item_style',[
			'label' => esc_html__( 'Gutter', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_responsive_control('guter_width_horizontal',[
				'label'          => esc_html__( 'Width', 'onnat-extra' ),
				'type'           => KFCM::SLIDER,
				'range'          => [
					'px' => [
						'min' => 0,
						'max' => 100
					]
				],
				'default'        => [
					'unit' => 'px',
					'size' => 5,
				],
				'tablet_default' => [
					'size' => 5,
					'unit' => 'px',
				],
				'mobile_default' => [
					'size' => 5,
					'unit' => 'px',
				],
				'condition'      => [
					'orientation' => 'horizontal'
				],
				'selectors'      => [
					'{{WRAPPER}} .kfw-ele-img-expand-item:after' => 'width:{{SIZE}}{{UNIT}}; height:100%;'
				],
			]);
			$this->add_responsive_control('guter_width_vertical',[
				'label'          => esc_html__( 'Width', 'onnat-extra' ),
				'type'           => KFCM::SLIDER,
				'range'          => [
					'px' => [
						'min' => 0,
						'max' => 100
					]
				],
				'default'        => [
					'unit' => 'px',
					'size' => 5,
				],
				'tablet_default' => [
					'size' => 5,
					'unit' => 'px',
				],
				'mobile_default' => [
					'size' => 5,
					'unit' => 'px',
				],
				'condition'      => [
					'orientation' => 'vertical'
				],
				'selectors'      => [
					'{{WRAPPER}} .kfw-ele-img-expand-item:after' => 'height:{{SIZE}}{{UNIT}}; width:100%;'
				],
			]);
			$this->add_control( 'guter_color', [
				'label'     => esc_html__('Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-img-expand-item:after' => 'background:{{VALUE}};'
				]
			]);
		$this->end_controls_section();
	}

	public function _register_item_title_style_section() {
		$this->start_controls_section( 'section_item_title_style',[
			'label' => esc_html__( 'Item::Title', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_group_control(KFGCT::get_type(), [
				'name'     => 'title_typo',
				'label'    => esc_html__( 'Typography', 'onnat-extra' ),
				'selector' => '{{WRAPPER}} .kfw-ele-img-expand-title'
			]);
			$this->add_control( 'title_color', [
				'label'     => esc_html__('Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-img-expand-title' => 'color:{{VALUE}};'
				]
			]);
			$this->add_group_control( KFGCTS::get_type(), [
				'name'     => 'title_text_shadow',
				'selector' => '{{WRAPPER}} .kfw-ele-img-expand-title'
			]);
			$this->add_group_control( KFGCTST::get_type(), [
				'name'     => 'title_text_stroke',
				'selector' => '{{WRAPPER}} .kfw-ele-img-expand-title'
			]);
			$this->add_responsive_control( 'title_padding',[
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-img-expand-title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_responsive_control( 'title_margin',[
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-img-expand-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
		$this->end_controls_section();
	}

	public function _register_item_sub_title_style_section() {
		$this->start_controls_section( 'section_item_stitle_style',[
			'label' => esc_html__( 'Item::Sub Title', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_group_control(KFGCT::get_type(), [
				'name'     => 'sub_title_typo',
				'label'    => esc_html__( 'Typography', 'onnat-extra' ),
				'selector' => '{{WRAPPER}} .kfw-ele-img-expand-sub-title'
			]);
			$this->add_control( 'sub_title_color', [
				'label'     => esc_html__('Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-img-expand-sub-title' => 'color:{{VALUE}};'
				]
			]);
			$this->add_group_control( KFGCTS::get_type(), [
				'name'     => 'sub_title_text_shadow',
				'selector' => '{{WRAPPER}} .kfw-ele-img-expand-sub-title'
			]);
			$this->add_group_control( KFGCTST::get_type(), [
				'name'     => 'sub_title_text_stroke',
				'selector' => '{{WRAPPER}} .kfw-ele-img-expand-sub-title'
			]);
			$this->add_responsive_control( 'sub_title_padding',[
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-img-expand-sub-title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_responsive_control( 'sub_title_margin',[
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-img-expand-sub-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
		$this->end_controls_section();
	}

	public function _register_item_content_style_section() {
		$this->start_controls_section( 'section_item_content_style',[
			'label' => esc_html__( 'Item::Content', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_control( 'content_style', [
				'label'        => esc_html__('Style', 'onnat-extra' ),
				'type'         => KFCM::SWITCHER,
				'label_on'     => esc_html__( 'Fill', 'onnat-extra' ),
				'label_off'    => esc_html__( 'Boxed', 'onnat-extra' ),
				'return_value' => 'yes',
				'default'      => 'yes',
			]);
			$this->add_group_control(KFGCT::get_type(), [
				'name'     => 'content_typo',
				'label'    => esc_html__( 'Typography', 'onnat-extra' ),
				'selector' => '{{WRAPPER}} .kfw-ele-img-expand-content'
			]);
			$this->add_control( 'content_color', [
				'label'     => esc_html__('Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-img-expand-content' => 'color:{{VALUE}};'
				]
			]);
			$this->add_group_control( KFGCTS::get_type(), [
				'name'     => 'content_text_shadow',
				'selector' => '{{WRAPPER}} .kfw-ele-img-expand-content'
			]);
			$this->add_group_control( KFGCTST::get_type(), [
				'name'     => 'content_text_stroke',
				'selector' => '{{WRAPPER}} .kfw-ele-img-expand-content'
			]);
			$this->add_responsive_control( 'content_padding',[
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-img-expand-content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_responsive_control( 'content_margin',[
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-img-expand-content' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_group_control( KFGCBG::get_type(), [
				'name'           => 'content_overlay',
				'types'          => ['classic', 'gradient'],
				'exclude'        => [ 'image'],
				'selector'       => '{{WRAPPER}} .kfw-ele-img-expand-content-boxed-style .kfw-ele-img-expand-content-wrapper, {{WRAPPER}} .kfw-ele-img-expand-content-fill-style .kfw-ele-img-expand-item:before',
				'fields_options' => [
					'background' => [
						'label'       => esc_html__( 'Overlay Type', 'onnat-extra' ),
						'type'        => KFCM::CHOOSE,
						'render_type' => 'ui',
					]
				],
			]);
			$this->add_responsive_control('content_width',[
				'label'     => esc_html__( 'Width', 'onnat-extra' ),
				'type'      => KFCM::SLIDER,
				'range'     => [ 'px' => ['min' => 100, 'max' => 1000  ] ],
				'default'   => [
					'unit' => 'px',
					'size' => 350,
				],
				'selectors' => [
					'{{WRAPPER}} .kfw-ele-img-expand-content-wrapper' => 'width:{{SIZE}}{{UNIT}};'
				],
			]);
		$this->end_controls_section();
	}

	public function _register_item_content_boxed_style_section() {
		$this->start_controls_section( 'section_item_boxed_content_style',[
			'label'     => esc_html__( 'Item::Content Boxed Style', 'onnat-extra' ),
			'tab'       => KFCM::TAB_STYLE,
			'condition' => [ 'content_style!' => 'yes' ],
		]);
			$this->add_group_control( KFGCBR::get_type(), [
				'name'      => 'boxed_content_border',
				'condition' => [ 'content_style!' => 'yes' ],
				'selector'  => '{{WRAPPER}} .kfw-ele-img-expand-content-boxed-style .kfw-ele-img-expand-content-wrapper'
			]);
			$this->add_responsive_control( 'boxed_content_border_radius', [
				'label'      => esc_html__('Border Radius', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'condition'  => [ 'content_style!' => 'yes' ],
				'size_units' => [ 'px', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-img-expand-content-boxed-style .kfw-ele-img-expand-content-wrapper' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_group_control( KFGCBS::get_type(), [
				'name'      => 'boxed_content_box_shadow',
				'condition' => [ 'content_style!' => 'yes' ],
				'selector'  => '{{WRAPPER}} .kfw-ele-img-expand-content-boxed-style .kfw-ele-img-expand-content-wrapper'
			]);
		$this->end_controls_section();
	}

	public function _register_item_action_btn_style_section() {
		$this->start_controls_section( 'section_item_action_btn_style',[
			'label' => esc_html__( 'Item::Action Button', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
			$this->add_group_control(KFGCT::get_type(), [
				'name'     => 'content_btn_typo',
				'label'    => esc_html__( 'Typography', 'onnat-extra' ),
				'selector' => '{{WRAPPER}} .kfw-ele-img-expand-button'
			]);
			$this->add_responsive_control( 'content_btn_padding',[
				'label'      => esc_html__('Padding', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-img-expand-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->add_responsive_control( 'content_btn_margin',[
				'label'      => esc_html__('Margin', 'onnat-extra' ),
				'type'       => KFCM::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .kfw-ele-img-expand-button' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]);
			$this->start_controls_tabs('tabs_content_btn_style');
				$this->start_controls_tab('tab_content_btn_normal_style',[
					'label' => esc_html__( 'Normal', 'onnat-extra' )
				]);
					$this->add_control( 'btn_normal_color', [
						'label'     => esc_html__('Text Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-img-expand-button' => 'color:{{VALUE}};'
						]
					]);
					$this->add_control( 'btn_normal_bg_color', [
						'label'     => esc_html__('Background Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-img-expand-button' => 'background:{{VALUE}};'
						]
					]);
					$this->add_group_control( KFGCTS::get_type(), [
						'name'     => 'btn_normal_text_shadow',
						'selector' => '{{WRAPPER}} .kfw-ele-img-expand-button'
					]);
					$this->add_group_control( KFGCTST::get_type(), [
						'name'     => 'btn_normal_text_stroke',
						'selector' => '{{WRAPPER}} .kfw-ele-img-expand-button'
					]);
					$this->add_group_control( KFGCBR::get_type(), [
						'name'     => 'btn_normal_border',
						'selector' => '{{WRAPPER}} .kfw-ele-img-expand-button'
					]);
					$this->add_responsive_control( 'btn_normal_border_radius', [
						'label'      => esc_html__('Border Radius', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', '%' ],
						'selectors'  => [
							'{{WRAPPER}} .kfw-ele-img-expand-button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
						]
					]);
					$this->add_group_control( KFGCBS::get_type(), [
						'name'     => 'btn_normal_box_shadow',
						'selector' => '{{WRAPPER}} .kfw-ele-img-expand-button'
					]);
				$this->end_controls_tab();
				$this->start_controls_tab('tab_content_btn_hover_style',[
					'label' => esc_html__( 'Hover', 'onnat-extra' )
				]);
					$this->add_control( 'btn_hover_color', [
						'label'     => esc_html__('Text Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-img-expand-button:hover' => 'color:{{VALUE}};'
						]
					]);
					$this->add_control( 'btn_hover_bg_color', [
						'label'     => esc_html__('Background Color', 'onnat-extra' ),
						'type'      => KFCM::COLOR,
						'selectors' => [
							'{{WRAPPER}} .kfw-ele-img-expand-button:hover' => 'background:{{VALUE}};'
						]
					]);
					$this->add_group_control( KFGCTS::get_type(), [
						'name'     => 'btn_hover_text_shadow',
						'selector' => '{{WRAPPER}} .kfw-ele-img-expand-button:hover'
					]);
					$this->add_group_control( KFGCTST::get_type(), [
						'name'     => 'btn_hover_text_stroke',
						'selector' => '{{WRAPPER}} .kfw-ele-img-expand-button:hover'
					]);
					$this->add_group_control( KFGCBR::get_type(), [
						'name'     => 'btn_hover_border',
						'selector' => '{{WRAPPER}} .kfw-ele-img-expand-button:hover'
					]);
					$this->add_responsive_control( 'btn_hover_border_radius', [
						'label'      => esc_html__('Border Radius', 'onnat-extra' ),
						'type'       => KFCM::DIMENSIONS,
						'size_units' => [ 'px', '%' ],
						'selectors'  => [
							'{{WRAPPER}} .kfw-ele-img-expand-button:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
						]
					]);
					$this->add_group_control( KFGCBS::get_type(), [
						'name'     => 'btn_hover_box_shadow',
						'selector' => '{{WRAPPER}} .kfw-ele-img-expand-button:hover'
					]);
				$this->end_controls_tab();
			$this->end_controls_tabs();
		$this->end_controls_section();
	}

	/**
	 * Generate setting json
	 */
	protected function _generate_setting_json( array $settings ) {
		extract( $settings );

		$options = [
			'id'             => 'kfw-ele-widget-img-expand-'.esc_attr( $this->get_id() ),
			'active_item'    => ( $active_item === 'yes' ) ? true : false,
			'active_item_no' => $active_item_number,
			'total_items'    => count( $items ),
		];

		return wp_json_encode( $options );
	}

	/**
	 * Get attachment image src.
	 */
	public static function _get_image_src( $attachment_id, $image_size_key, array $settings ) {
        return KFGCIS::get_attachment_image_src( $attachment_id, $image_size_key, $settings );
	}

    /**
     * Get attachment image html
     */
	public static function _get_image_html( $settings ) {
		return KFGCIS::get_attachment_image_html( $settings );
	}

    /**
     * Get attachment image title
     */
	public function _get_image_title( $attachment ) {
		if ( empty( $attachment['id'] ) ) {
			return '';
		}

		return get_the_title( $attachment['id'] );
	}

    /**
     * Get attachment image alt
     */
	public function _get_image_alt( $instance ) {
		if ( empty( $instance['id'] ) ) {
			// For `Insert From URL` images.
			return isset( $instance['alt'] ) ? trim( strip_tags( $instance['alt'] ) ) : '';
		}

		$attachment_id = $instance['id'];
		if ( ! $attachment_id ) {
			return '';
		}

		$attachment = get_post( $attachment_id );
		if ( ! $attachment ) {
			return '';
		}

		$alt = get_post_meta( $attachment_id, '_wp_attachment_image_alt', true );
		if ( ! $alt ) {
			$alt = $attachment->post_excerpt;
			if ( ! $alt ) {
				$alt = $attachment->post_title;
			}
		}
		return trim( strip_tags( $alt ) );
	}

	public function _is_edit_mode() {

		if( \Elementor\Plugin::$instance->preview->is_preview_mode() || \Elementor\Plugin::$instance->editor->is_edit_mode() ) {
			return true;
		}

		return false;
	}

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-image-expand");
		include $render;
    }
}