<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

use \Elementor\Plugin as KFEl;
use \Elementor\Widget_Base as WB;
use \Elementor\Controls_Manager as KFCM;
use \Elementor\Icons_Manager as KFIM;
use \Elementor\Group_Control_Background as KFGCBG;
use \Elementor\Group_Control_Border as KFGCBR;
use \Elementor\Group_Control_Box_Shadow as KFGCBS;
use \Elementor\Group_Control_Css_Filter as KFGCSSF;
use \Elementor\Group_Control_Typography as KFGCT;
use \Elementor\Group_Control_Text_Shadow as KFGCTS;
use \Elementor\Group_Control_Text_Stroke as KFGCTST;
use \Elementor\Utils as KFUtils;

class KFW_Image_Stack extends WB {

	/**
	 * Get element name.
	 *
	 * Retrieve the element name.
	 *
	 * @return string The name.
	 */
	public function get_name() {
        return 'kfw-image-stack';
    }

	/**
	 * Get element title.
	 *
	 * Retrieve the element title.
	 *
	 * @return string Element title.
	 */
	public function get_title() {
		return esc_html__('Image Stack', 'onnat-extra');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the widget categories.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ '_kfw_core_widgets' ];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'kfw-badge eicon-slideshow kfw-image-stack';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the widget keywords.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'kfw', 'kin', 'image', 'stack' ];
	}

	/**
	 * Get widget help url.
	 *
	 * Retrieve the widget help url.
	 *
	 * @return string Widget help url.
	 */
	public function get_custom_help_url() {
        return KFE_CONST_THEME_DOC . 'elementor-custom-widgets/image-stack';
    }

	/**
	 * Get HTML wrapper class.
     *
	 * Retrieve the widget container class.
     *
     * @return string Widget container css classes.
	 */
	public function get_html_wrapper_class() {
        $name = $this->get_name();

		return sprintf( 'kfw-ele-%s elementor-widget-%s', $name, $name );
	}

	/**
	 * Widget base constructor.
	 *
	 * Initializing the widget base class.
	 */
	public function __construct( $data = [], $args = null ) {
        parent::__construct( $data, $args );

		wp_register_style( 'kfw-image-stack',
			KFE_CONST_URL . 'libraries/elementor/assets/css/image-stack' . KFE_CONST_DEBUG_SUFFIX . '.css',
			[],
			KFE_CONST_VERSION
		);
    }

	/**
	 * Get style dependencies.
	 *
	 * Retrieve the list of style dependencies the element requires.
	 *
	 * @return array Element styles dependencies.
	 */
	public function get_style_depends() {
		return [ 'kfw-image-stack' ];
	}

	/**
	 * Register controls.
	 *
	 * Used to add new controls to any element type.
	 */
	protected function register_controls() {
		$this->_register_items_section();
		$this->_register_last_item_section();
		$this->_register_settings_section();

		$this->_register_shape_style_section();
		$this->_register_items_style_section();
        $this->_register_last_item_style_section();
    }

	public function _register_items_section() {
		$this->start_controls_section( 'section_items', [
			'label' => esc_html__( 'Stack Elements', 'onnat-extra' )
		]);
			$this->add_control( 'stack_element', [
				'label'     => esc_html__( 'Type', 'onnat-extra' ),
				'type'      => KFCM::CHOOSE,
				'options'   => [
					'icon'  => [
						'title' => esc_html__('Icon', 'onnat-extra'),
						'icon'  => 'eicon-plus-circle-o',
					],
					'image'  => [
						'title' => esc_html__('Image', 'onnat-extra'),
						'icon'  => 'eicon-image',
					],
					'text'  => [
						'title' => esc_html__('Text', 'onnat-extra'),
						'icon'  => 'eicon-animation-text',
					],
				],
				'default'   => 'text',
				'toggle'    => false
			]);
            $this->add_control( 'icon_items', [
                'type'      => KFCM::REPEATER,
                'fields'    => [
                    [
                        'name'    => 'icon',
                        'label'   => esc_html__( 'Icon', 'onnat-extra' ),
                        'type'    => KFCM::ICONS,
                        'default' => [
                            'value'   => 'fas fa-heart',
                            'library' => 'fa-solid',
                        ],
                    ]
                ],
                'condition' => [ 'stack_element' => 'icon' ],
                'default'   => []
            ]);
            $this->add_control( 'image_items', [
                'type'      => KFCM::REPEATER,
                'fields'    => [
                    [
                        'name'    => 'image',
                        'label'   => esc_html__( 'Image', 'onnat-extra' ),
                        'type'    => KFCM::MEDIA,
                        'default' => [
                            'url' => KFUtils::get_placeholder_image_src(),
                            'id'  => -1
                        ],
                        'dynamic' => [ 'active' => true ],
                    ]
                ],
                'condition' => [ 'stack_element' => 'image' ],
                'default'   => []
            ]);
            $this->add_control( 'text_items', [
                'type'      => KFCM::REPEATER,
                'fields'    => [
                    [
                        'name'    => 'text',
                        'label'   => esc_html__( 'Text', 'onnat-extra' ),
                        'type'    => KFCM::TEXT,
                        'default' => '1',
                        'dynamic' => [
                            'active' => true,
                        ],
                    ]
                ],
                'condition' => [ 'stack_element' => 'text' ],
                'default'   => [
                    [ 'text' => '1' ],
                    [ 'text' => '2' ],
                    [ 'text' => '3' ],
                    [ 'text' => '4' ],
                    [ 'text' => '5' ],
                ]
            ]);
		$this->end_controls_section();
    }

    public function _register_last_item_section() {
		$this->start_controls_section( 'section_last_item', [
			'label' => esc_html__( 'Stack Last Element', 'onnat-extra' )
		]);
            $this->add_control( 'last_element', [
                'label'     => esc_html__( 'Type', 'onnat-extra' ),
                'type'      => KFCM::CHOOSE,
                'options'   => [
                    'icon'  => [
                        'title' => esc_html__('Icon', 'onnat-extra'),
                        'icon'  => 'eicon-plus-circle-o',
                    ],
                    'image'  => [
                        'title' => esc_html__('Image', 'onnat-extra'),
                        'icon'  => 'eicon-image',
                    ],
                    'text'  => [
                        'title' => esc_html__('Text', 'onnat-extra'),
                        'icon'  => 'eicon-animation-text',
                    ],
                ],
                'default'   => 'icon',
                'toggle'    => false
            ]);
            $this->add_control( 'le_icon', [
                'label'     => esc_html__( 'Icon', 'onnat-extra' ),
                'type'      => KFCM::ICONS,
                'condition' => [ 'last_element' => 'icon' ],
                'default'   => [
                    'value'   => 'fas fa-heart',
                    'library' => 'fa-solid',
                ],
            ]);
            $this->add_control( 'le_image', [
                'label'     => esc_html__( 'Image', 'onnat-extra' ),
                'type'      => KFCM::MEDIA,
                'dynamic'   => [
                    'active' => true,
                ],
                'default'   => [
                    'url' => KFUtils::get_placeholder_image_src(),
                ],
                'condition' => [ 'last_element' => 'image' ],
            ]);
            $this->add_control( 'le_text', [
                'label'     => esc_html__( 'Text', 'onnat-extra' ),
                'type'      => KFCM::TEXT,
                'dynamic'   => [
                    'active' => true,
                ],
                'condition' => [ 'last_element' => 'text' ],
                'default'   => '1',
            ]);
            $this->add_control('le_link',[
                'label'       => esc_html__( 'Link', 'onnat-extra' ),
                'label_block' => true,
                'type'        => KFCM::URL,
                'dynamic'     => [
                    'active' => true
                ],
                'options'     => [
                    'url',
                    'is_external',
                    'nofollow'
                ],
                'default'     => [
                    'url'         => '',
                    'is_external' => true,
                    'nofollow'    => true,
                ],
            ]);
		$this->end_controls_section();
    }

    public function _register_settings_section() {
		$this->start_controls_section( 'section_settings', [
			'label' => esc_html__( 'Settings', 'onnat-extra' ),
			'tab'   => KFCM::TAB_CONTENT,
		]);
            $this->add_control( 'style', [
                'label'        => esc_html__( 'Orientation', 'onnat-extra' ),
                'type'         => KFCM::CHOOSE,
                'options'      => [
                    'row' => [ 
                        'title' => esc_html__('Show in Row wise', 'onnat-extra'),
                        'icon'  => 'eicon-section'
                    ],
                    'col' => [
                        'title' => esc_html__('Show in Column wise', 'onnat-extra'),
                        'icon'  => 'eicon-column'
                    ],
                ],
                'default'      => 'row',
                'prefix_class' => 'kfw-ele-image-stack-style-',
                'toggle'       => false,
            ]);
            $this->add_control( 'hover_style', [
                'label'        => esc_html__( 'Style', 'onnat-extra' ),
                'type'         => KFCM::CHOOSE,
                'options'      => [
                    'style-1' => [
                        'title' => esc_html__('Type 1', 'onnat-extra'),
                        'icon'  => 'kfw-ctrl-icon kfw-ctrl-type-1-icon',
                    ],
                    'style-2' => [
                        'title' => esc_html__('Type 2', 'onnat-extra'),
                        'icon'  => 'kfw-ctrl-icon kfw-ctrl-type-2-icon',
                    ],
                ],
                'default'      => 'style-1',
                'prefix_class' => 'kfw-ele-image-stack-hover-',
                'toggle'       => false,
            ]);
            $this->add_control( 'reverse_it', [
                'label'        => esc_html__( 'Reverse it?', 'onnat-extra' ),
                'type'         => KFCM::SWITCHER,
                'return_value' => 'yes',
            ]);
		$this->end_controls_section();
    }

    public function _register_shape_style_section() {
		$this->start_controls_section( 'section_shape_style', [
			'label' => esc_html__( 'Shape', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_responsive_control('stack_align', [
                'label'                => esc_html__( 'Alignment', 'onnat-extra' ),
                'type'                 => KFCM::CHOOSE,
                'toggle'               => false,
                'options'              => [
                    'left'   => [
                        'title' => esc_html__( 'Left', 'onnat-extra' ),
                        'icon'  => 'eicon-text-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__( 'Center', 'onnat-extra' ),
                        'icon'  => 'eicon-text-align-center',
                    ],
                    'right'  => [
                        'title' => esc_html__( 'Right', 'onnat-extra' ),
                        'icon'  => 'eicon-text-align-right',
                    ],
                ],
                'default'              => 'center',
                'selectors_dictionary' => [
					'left'   => 'justify-content:start;',
					'center' => 'justify-content:center;',
					'right'  => 'justify-content:end;',
                ],
                'selectors'            => [
                    '{{WRAPPER}} .kfw-ele-widget-image-stack-main' => '{{VALUE}};'
                ],
            ]);
            $this->add_responsive_control('size',[
                'label'     => esc_html__( 'Size', 'onnat-extra' ),
                'type'      => KFCM::SLIDER,
                'default'   => [
                    'size' => 100,
                ],
				'range'          => [
					'px' => [
						'min' => 50,
						'max' => 1000,
					],
                ],
                'selectors' => [
                    '{{WRAPPER}}' => '--kfw-img-stack-shape-size:{{SIZE}}{{UNIT}};--kfw-img-stack-shape-inset:calc( var(--kfw-img-stack-shape-size) * 0.25 );',
                ],
            ]);
            $this->add_group_control( KFGCBG::get_type(), [
                'name'           => 'shape_bg',
                'types'          => ['classic', 'gradient'],
                'exclude'        => [ 'image'],
                'fields_options' => [
                    'background' => [ 'default' => 'classic' ],
                    'color'      => [ 'default' => '' ],
                ],
                'selector'       => '{{WRAPPER}} .kfw-ele-widget-image-stack-item',
            ]);
            $this->add_group_control( KFGCBR::get_type(), [
                'name'     => 'shape_border',
                'selector' => '{{WRAPPER}} .kfw-ele-widget-image-stack-item',
            ]);
            $this->add_responsive_control( 'shape_border_radius', [
                'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', '%' ],
                'default'    => [
                    'top'      => '',
                    'right'    => '',
                    'bottom'   => '',
                    'left'     => '',
                    'unit'     => 'px',
                    'isLinked' => false
                ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-image-stack-item' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_group_control( KFGCBS::get_type(), [
                'name'     => 'shape_box_shadow',
                'selector' => '{{WRAPPER}} .kfw-ele-widget-image-stack-item'
            ]);
            $this->add_control('shape_rotate',[
                'label'     => esc_html__( 'Rotate', 'onnat-extra' ),
                'type'      => KFCM::SLIDER,
                'default'   => [
                    'size' => 0,
                    'unit' => 'deg',
                ],
                'range'     => [
                    'deg' => [
                        'max' => 360,
                        'min' => -360,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .kfw-ele-widget-image-stack-item' => 'transform: rotate({{SIZE}}{{UNIT}});',
                ],
            ]);
		$this->end_controls_section();
    }

    public function _register_items_style_section() {
		$this->start_controls_section( 'section_items_style', [
			'label' => esc_html__( 'Stack Elements', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_control( 'icon_color', [
                'label'     => esc_html__('Color', 'onnat-extra' ),
                'type'      => KFCM::COLOR,
                'default'   => '',
                'condition' => [ 'stack_element' => 'icon' ],
                'selectors' => [
                    '{{WRAPPER}} .kfw-ele-widget-image-stack-item.kfw-ele-widget-image-stack-icon-type .kfw-elementor-icon-wrap i' => 'color:{{VALUE}};',
                    '{{WRAPPER}} .kfw-ele-widget-image-stack-item.kfw-ele-widget-image-stack-icon-type .kfw-elementor-icon-wrap svg' => 'fill:{{VALUE}};',
                ],
            ]);
            $this->add_responsive_control( 'icon_size', [
                'label'      => esc_html__('Size', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'size_units' => [ 'px' ],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'condition'  => [ 'stack_element' => 'icon' ],
                'selectors'  => [
					'{{WRAPPER}} .kfw-ele-widget-image-stack-item.kfw-ele-widget-image-stack-icon-type .kfw-elementor-icon-wrap' => 'font-size: {{SIZE}}{{UNIT}};'
                ],
            ]);
            $this->add_responsive_control( 'img_size', [
                'label'      => esc_html__('Size', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'size_units' => [ '%' ],
                'range'      => [
                    '%' => [
                        'min' => 5,
                        'max' => 100,
                    ],
                ],
                'condition'  => [ 'stack_element' => 'image' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-image-stack-item.kfw-ele-widget-image-stack-image-type .kfw-elementor-icon-wrap img' => 'width: {{SIZE}}%;',
                ],
            ]);
            $this->add_responsive_control( 'img_border_radius', [
                'label'      => esc_html__('Border Radius', 'onnat-extra' ),
                'type'       => KFCM::DIMENSIONS,
                'size_units' => [ 'px', '%' ],
                'default'    => [
                    'top'      => '',
                    'right'    => '',
                    'bottom'   => '',
                    'left'     => '',
                    'unit'     => 'px',
                    'isLinked' => false
                ],
                'condition'  => [ 'stack_element' => 'image' ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-image-stack-item.kfw-ele-widget-image-stack-image-type img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]);
            $this->add_group_control( KFGCSSF::get_type(), [
                'name'     => 'img_css_filters',
                'selector' => '{{WRAPPER}} .kfw-ele-widget-image-stack-item.kfw-ele-widget-image-stack-image-type .kfw-elementor-icon-wrap img',
                'condition' => [ 'stack_element' => 'image' ],
            ]);
            $this->add_control('img_opacity',[
                'label'     => esc_html__( 'Opacity', 'onnat-extra' ),
                'type'      => KFCM::SLIDER,
                'range'     => [
                    'px' => [
                        'max'  => 1,
                        'min'  => 0.10,
                        'step' => 0.01,
                    ],
                ],
                'condition' => [ 'stack_element' => 'image' ],
                'selectors' => [
                    '{{WRAPPER}} .kfw-ele-widget-image-stack-item.kfw-ele-widget-image-stack-image-type .kfw-elementor-icon-wrap img' => 'opacity: {{SIZE}};'
                ],
            ]);
			$this->add_group_control(KFGCT::get_type(), [
                'name'      => 'text_typo',
                'label'     => esc_html__( 'Typography', 'onnat-extra' ),
                'exclude'   => [ 'letter_spacing', 'word_spacing' ],
                'selector'  => '{{WRAPPER}} .kfw-ele-widget-image-stack-item.kfw-ele-widget-image-stack-text-type .kfw-elementor-icon-wrap',
                'condition' => [ 'stack_element' => 'text' ],
			]);
			$this->add_control( 'text_color', [
				'label'     => esc_html__('Color', 'onnat-extra' ),
				'type'      => KFCM::COLOR,
				'selectors' => [
                    '{{WRAPPER}} .kfw-ele-widget-image-stack-item.kfw-ele-widget-image-stack-text-type .kfw-elementor-icon-wrap' => 'color:{{VALUE}};'
                ],
                'condition' => [ 'stack_element' => 'text' ],
			]);
			$this->add_group_control( KFGCTS::get_type(), [
                'name'      => 'text_shadow',
                'selector'  => '{{WRAPPER}} .kfw-ele-widget-image-stack-item.kfw-ele-widget-image-stack-text-type .kfw-elementor-icon-wrap',
                'condition' => [ 'stack_element' => 'text' ],
			]);
			$this->add_group_control( KFGCTST::get_type(), [
                'name'      => 'text_stroke',
                'selector'  => '{{WRAPPER}} .kfw-ele-widget-image-stack-item.kfw-ele-widget-image-stack-text-type .kfw-elementor-icon-wrap',
                'condition' => [ 'stack_element' => 'text' ],
			]);
            $this->add_control('rotate',[
                'label'     => esc_html__( 'Rotate', 'onnat-extra' ),
                'type'      => KFCM::SLIDER,
                'default'   => [
                    'size' => 0,
                    'unit' => 'deg',
                ],
                'range'     => [
                    'deg' => [
                        'max' => 360,
                        'min' => - 360,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .kfw-ele-widget-image-stack-item .kfw-elementor-icon-wrap' => 'transform: rotate({{SIZE}}{{UNIT}});',
                ],
            ]);
		$this->end_controls_section();
    }

    public function _register_last_item_style_section() {
		$this->start_controls_section( 'section_last_item_style', [
			'label' => esc_html__( 'Stack Last Element', 'onnat-extra' ),
			'tab'   => KFCM::TAB_STYLE,
		]);
            $this->add_control( 'le_icon_color_settings', [
                'label'        => esc_html__('Colors', 'onnat-extra'),
                'type'         => KFCM::POPOVER_TOGGLE,
                'return_value' => 'yes',
                'condition'    => [
                    'last_element'    => 'icon',
                    'le_icon[value]!' => ''
                ],
            ]);
            $this->start_popover();
                $this->add_control('le_icon_normal_color_settings_heading',[
                    'label' => esc_html__( 'Normal Colors', 'onnat-extra' ),
                    'type'  => KFCM::HEADING,
                ]);
                    $this->add_control( 'le_icon_color', [
                        'label'     => esc_html__('Icon Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'default'   => '',
                        'condition' => [
                            'last_element'    => 'icon',
                            'le_icon[value]!' => ''
                        ],
                        'selectors' => [
                            '{{WRAPPER}} .kfw-ele-widget-image-stack-item.kfw-ele-widget-image-stack-last-item.kfw-ele-widget-image-stack-icon-type .kfw-elementor-icon-wrap i'   => 'color:{{VALUE}};',
                            '{{WRAPPER}} .kfw-ele-widget-image-stack-item.kfw-ele-widget-image-stack-last-item.kfw-ele-widget-image-stack-icon-type .kfw-elementor-icon-wrap svg' => 'fill:{{VALUE}};',
                        ],
                    ]);
                    $this->add_control( 'le_bg_color', [
                        'label'     => esc_html__('Background Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'default'   => '',
                        'condition' => [
                            'last_element'    => 'icon',
                            'le_icon[value]!' => ''
                        ],
                        'selectors' => [
                            '{{WRAPPER}} .kfw-ele-widget-image-stack-item.kfw-ele-widget-image-stack-last-item.kfw-ele-widget-image-stack-icon-type' => 'background-color:{{VALUE}};',
                        ],
                    ]);
                $this->add_control('le_icon_hover_color_settings_heading',[
                    'label' => esc_html__( 'Hover Colors', 'onnat-extra' ),
                    'type'  => KFCM::HEADING,
                ]);
                    $this->add_control( 'le_icon_hover_color', [
                        'label'     => esc_html__('Icon Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'default'   => '',
                        'condition' => [
                            'last_element'    => 'icon',
                            'le_icon[value]!' => ''
                        ],
                        'selectors' => [
                            '{{WRAPPER}} .kfw-ele-widget-image-stack-item.kfw-ele-widget-image-stack-last-item.kfw-ele-widget-image-stack-icon-type:hover .kfw-elementor-icon-wrap i'   => 'color:{{VALUE}};',
                            '{{WRAPPER}} .kfw-ele-widget-image-stack-item.kfw-ele-widget-image-stack-last-item.kfw-ele-widget-image-stack-icon-type:hover .kfw-elementor-icon-wrap svg' => 'fill:{{VALUE}};',
                        ],
                    ]);
                    $this->add_control( 'le_icon_bg_hover_color', [
                        'label'     => esc_html__('Background Color', 'onnat-extra' ),
                        'type'      => KFCM::COLOR,
                        'default'   => '',
                        'condition' => [
                            'last_element'    => 'icon',
                            'le_icon[value]!' => ''
                        ],
                        'selectors' => [
                            '{{WRAPPER}} .kfw-ele-widget-image-stack-item.kfw-ele-widget-image-stack-last-item.kfw-ele-widget-image-stack-icon-type:hover' => 'background-color:{{VALUE}};',
                        ],
                    ]);
            $this->end_popover();
            $this->add_responsive_control( 'le_icon_size', [
                'label'      => esc_html__('Size', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'size_units' => [ 'px' ],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'condition'  => [
                    'last_element'    => 'icon',
                    'le_icon[value]!' => ''
                ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-image-stack-item.kfw-ele-widget-image-stack-last-item.kfw-ele-widget-image-stack-icon-type .kfw-elementor-icon-wrap' => 'font-size: {{SIZE}}{{UNIT}};'
                ],
            ]);
            $this->add_responsive_control( 'le_img_size', [
                'label'      => esc_html__('Size', 'onnat-extra' ),
                'type'       => KFCM::SLIDER,
                'size_units' => [ '%' ],
                'range'      => [
                    '%' => [
                        'min' => 5,
                        'max' => 100,
                    ],
                ],
                'condition'  => [
                    'last_element'   => 'image',
                    'le_image[url]!' => ''
                ],
                'selectors'  => [
                    '{{WRAPPER}} .kfw-ele-widget-image-stack-item.kfw-ele-widget-image-stack-last-item.kfw-ele-widget-image-stack-image-type .kfw-elementor-icon-wrap img' => 'width: {{SIZE}}%;',
                ],
            ]);
            $this->add_group_control( KFGCSSF::get_type(), [
                'name'      => 'le_img_css_filters',
                'selector'  => '{{WRAPPER}} .kfw-ele-widget-image-stack-item.kfw-ele-widget-image-stack-last-item.kfw-ele-widget-image-stack-image-type .kfw-elementor-icon-wrap img',
                'condition' => [
                    'last_element'   => 'image',
                    'le_image[url]!' => ''
                ],
            ]);
            $this->add_control('le_img_opacity',[
                'label'     => esc_html__( 'Opacity', 'onnat-extra' ),
                'type'      => KFCM::SLIDER,
                'range'     => [
                    'px' => [
                        'max'  => 1,
                        'min'  => 0.10,
                        'step' => 0.01,
                    ],
                ],
                'condition' => [
                    'last_element'   => 'image',
                    'le_image[url]!' => ''
                ],
                'selectors' => [
                    '{{WRAPPER}} .kfw-ele-widget-image-stack-item.kfw-ele-widget-image-stack-last-item.kfw-ele-widget-image-stack-image-type .kfw-elementor-icon-wrap img' => 'opacity: {{SIZE}};'
                ],
            ]);
			$this->add_group_control(KFGCT::get_type(), [
                'name'      => 'le_text_typo',
                'label'     => esc_html__( 'Typography', 'onnat-extra' ),
                'exclude'   => [ 'letter_spacing', 'word_spacing' ],
                'selector'  => '{{WRAPPER}} .kfw-ele-widget-image-stack-item.kfw-ele-widget-image-stack-last-item.kfw-ele-widget-image-stack-text-type .kfw-elementor-icon-wrap',
                'condition' => [
                    'last_element' => 'text',
                    'le_text!'     => ''
                ],
			]);
            $this->add_control( 'le_text_color_settings', [
                'label'        => esc_html__('Colors', 'onnat-extra'),
                'type'         => KFCM::POPOVER_TOGGLE,
                'return_value' => 'yes',
                'condition'    => [
                    'last_element' => 'text',
                    'le_text!'     => ''
                ],
            ]);
            $this->start_popover();
                $this->add_control('le_text_normal_color_settings_heading',[
                    'label' => esc_html__( 'Normal Colors', 'onnat-extra' ),
                    'type'  => KFCM::HEADING,
                ]);
                $this->add_control( 'le_text_color', [
                    'label'     => esc_html__('Text Color', 'onnat-extra' ),
                    'type'      => KFCM::COLOR,
                    'default'   => '',
                    'condition' => [
                        'last_element' => 'text',
                        'le_text!'     => ''
                    ],
                    'selectors' => [
                        '{{WRAPPER}} .kfw-ele-widget-image-stack-item.kfw-ele-widget-image-stack-last-item.kfw-ele-widget-image-stack-text-type .kfw-elementor-icon-wrap' => 'color:{{VALUE}};',
                    ],
                ]);
                $this->add_control( 'le_text_bg_color', [
                    'label'     => esc_html__('Background Color', 'onnat-extra' ),
                    'type'      => KFCM::COLOR,
                    'default'   => '',
                    'condition' => [
                        'last_element' => 'text',
                        'le_text!'     => ''
                    ],
                    'selectors' => [
                        '{{WRAPPER}} .kfw-ele-widget-image-stack-item.kfw-ele-widget-image-stack-last-item.kfw-ele-widget-image-stack-text-type' => 'background-color:{{VALUE}};',
                    ],
                ]);
                $this->add_control('le_text_hover_color_settings_heading',[
                    'label' => esc_html__( 'Hover Colors', 'onnat-extra' ),
                    'type'  => KFCM::HEADING,
                ]);
                $this->add_control( 'le_text_hover_color', [
                    'label'     => esc_html__('Icon Color', 'onnat-extra' ),
                    'type'      => KFCM::COLOR,
                    'default'   => '',
                    'condition' => [
                        'last_element' => 'text',
                        'le_text!'     => ''
                    ],
                    'selectors' => [
                        '{{WRAPPER}} .kfw-ele-widget-image-stack-item.kfw-ele-widget-image-stack-last-item.kfw-ele-widget-image-stack-text-type:hover .kfw-elementor-icon-wrap' => 'color:{{VALUE}};',
                    ],
                ]);
                $this->add_control( 'le_text_bg_hover_color', [
                    'label'     => esc_html__('Background Color', 'onnat-extra' ),
                    'type'      => KFCM::COLOR,
                    'default'   => '',
                    'condition' => [
                        'last_element' => 'text',
                        'le_text!'     => ''
                    ],
                    'selectors' => [
                        '{{WRAPPER}} .kfw-ele-widget-image-stack-item.kfw-ele-widget-image-stack-last-item.kfw-ele-widget-image-stack-text-type:hover' => 'background-color:{{VALUE}};',
                    ],
                ]);
            $this->end_popover();
			$this->add_group_control( KFGCTS::get_type(), [
                'name'      => 'le_text_shadow',
                'selector'  => '{{WRAPPER}} .kfw-ele-widget-image-stack-item.kfw-ele-widget-image-stack-last-item.kfw-ele-widget-image-stack-text-type .kfw-elementor-icon-wrap',
                'condition' => [
                    'last_element' => 'text',
                    'le_text!'     => ''
                ],
			]);
			$this->add_group_control( KFGCTST::get_type(), [
                'name'      => 'le_text_stroke',
                'selector'  => '{{WRAPPER}} .kfw-ele-widget-image-stack-item.kfw-ele-widget-image-stack-last-item.kfw-ele-widget-image-stack-text-type .kfw-elementor-icon-wrap',
                'condition' => [
                    'last_element' => 'text',
                    'le_text!'     => ''
                ],
			]);
            $this->add_control('le_rotate',[
                'label'     => esc_html__( 'Rotate', 'onnat-extra' ),
                'type'      => KFCM::SLIDER,
                'default'   => [
                    'size' => 0,
                    'unit' => 'deg',
                ],
                'range'     => [
                    'deg' => [
                        'max' => 360,
                        'min' => -360,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .kfw-ele-widget-image-stack-item.kfw-ele-widget-image-stack-last-item .kfw-elementor-icon-wrap' => 'transform: rotate({{SIZE}}{{UNIT}});',
                ],
            ]);
		$this->end_controls_section();
    }

	/**
	 * Render icon from icon manger
	 */
    public function _render_icons( $icon , $echo = true ) {
		echo '<span class="kfw-elementor-icon-wrap">';
			KFIM::render_icon( $icon, [ 'aria-hidden' => 'true' ] );
		echo '</span>';
	}

	public function _render_img( $settings = [] ) {
		if( !empty( $settings['image']['url'] ) ) {
			$image_src = $settings['image']['url'];

			$image_title = $this->_get_image_title( $settings['image'] );
			$image_title = empty( $image_title ) ? $this->get_title() : $image_title;

			$image_alt = $this->_get_image_alt( $settings['image'] );
			$image_alt = empty( $image_alt ) ? $image_title : $image_title;

            $key = 'image_attr'. $settings['_id'];

			$this->add_render_attribute( $key, [
                'alt'   => esc_attr( $image_alt ),
				'title' => esc_attr( $image_title ),
				'src'   => esc_url( $image_src ),
			] );

			echo '<span class="kfw-elementor-icon-wrap">';
				echo '<img '.$this->get_render_attribute_string( $key ).'>';
			echo '</span>';
		}
    }

	public function _get_image_title( $attachment ) {
		if ( empty( $attachment['id'] ) ) {
			return '';
		}

		return get_the_title( $attachment['id'] );
	}

	public function _get_image_alt( $instance ) {
		if ( empty( $instance['id'] ) ) {
			// For `Insert From URL` images.
			return isset( $instance['alt'] ) ? trim( strip_tags( $instance['alt'] ) ) : '';
		}

		$attachment_id = $instance['id'];
		if ( ! $attachment_id ) {
			return '';
		}

		$attachment = get_post( $attachment_id );
		if ( ! $attachment ) {
			return '';
		}

		$alt = get_post_meta( $attachment_id, '_wp_attachment_image_alt', true );
		if ( ! $alt ) {
			$alt = $attachment->post_excerpt;
			if ( ! $alt ) {
				$alt = $attachment->post_title;
			}
		}
		return trim( strip_tags( $alt ) );
	}

    public function _last_ele( $settings ) {
        $last_ele_type = $settings['last_element'];
        $last_ele_link = $settings['le_link'];

        $last_ele_classes = [
            'elementor-repeater-item-last',
            'kfw-ele-widget-image-stack-last-item',
            'kfw-ele-widget-image-stack-item'
        ];

        $link_attr = '';
        if( !empty( $last_ele_link['url'] ) ) {
            $link   = $last_ele_link['url'];
            $target = ( !empty( $last_ele_link['is_external'] ) ) ? '_blank' : '_self';

            $last_ele_classes[] = 'kfw-ele-img-stack-linked-list-item';

            $mode      = $this->_is_editor_mode();
            $link_attr = '';

            if( !$mode ) {
                $link_attr = sprintf('onclick="window.open(\'%s\', \'%s\');"', esc_url($link), $target);
            }
        }

        if( $last_ele_type === 'icon' ) {
            $last_item = array_filter( $settings['le_icon'] );

            if( count( $last_item ) ) {

                $last_ele_classes[] = 'kfw-ele-widget-image-stack-icon-type';
                $e_classes          = implode( " ", $last_ele_classes);

                echo '<li class="'. $e_classes.'" '. $link_attr.'>';
                    $this->_render_icons( $last_item );
                echo '</li>';
            }
        } else if( $last_ele_type === 'image' ) {
            $last_item = $settings['le_image'];

            if( !empty( $last_item['url'] ) ) {
                $last_ele_classes[] = 'kfw-ele-widget-image-stack-image-type';
                $e_classes          = implode( " ", $last_ele_classes);

                echo '<li class="'. $e_classes.'" '. $link_attr.'>';
                    $last_img['image'] = $last_item;
                    $last_img['_id']   = '_last_img'.$this->get_id();
                    $this->_render_img( $last_img );
                echo '</li>';
            }

        } else if( $last_ele_type === 'text' ) {
            $last_item = $settings['le_text'];
            if( !empty( $last_item ) ) {

                $last_ele_classes[] = 'kfw-ele-widget-image-stack-text-type';
                $e_classes          = implode( " ", $last_ele_classes);

                echo '<li class="'. $e_classes.'" '. $link_attr.'>';
                    echo '<span class="kfw-elementor-icon-wrap">';
                        echo $last_item;
                    echo '</span>';
                echo '</li>';
            }
        }
    }

    public function _is_editor_mode() {
        $preview = KFEl::instance()->preview->is_preview_mode();
        $editor  = KFEl::instance()->editor->is_edit_mode();

        return ( $preview || $editor ) ? true : false;
    }

	/**
	 * Render widget output on the frontend.
	 */
    protected function render() {
		$render = kf_extra_plugin_elementor()->get_template("kfw-image-stack");
		include $render;
    }

}